﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/atk_CommandManager.h>
#include <nn/atk/atk_SoundThread.h>
#include <nn/os/os_MessageQueue.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{
inline int GetSendQueueCount(int queueCount)
{
    return queueCount;
}

inline int GetRecvQueueCount(int queueCount)
{
    return GetSendQueueCount(queueCount) + 1;
}

inline size_t GetSendQueueBufferSize(int queueCount)
{
    return sizeof(uintptr_t) * GetSendQueueCount(queueCount);
}

inline size_t GetRecvQueueBufferSize(int queueCount)
{
    return sizeof(uintptr_t) * GetRecvQueueCount(queueCount);
}

}

CommandBuffer::CommandBuffer() NN_NOEXCEPT
    : m_CommandMemoryArea(NULL)
    , m_CommandMemoryAreaSize(0)
    , m_CommandMemoryAreaBegin(0)
    , m_CommandMemoryAreaEnd(0)
    , m_CommandMemoryAreaZeroFlag(true)
{
}

CommandBuffer::~CommandBuffer() NN_NOEXCEPT
{
    Finalize();
}

void CommandBuffer::Initialize( void* commandBuffer, size_t commandBufferSize ) NN_NOEXCEPT
{
    m_CommandMemoryAreaBegin = 0;
    m_CommandMemoryAreaEnd = 0;
    m_CommandMemoryAreaZeroFlag = false;

    m_CommandMemoryArea = static_cast<uint32_t*>(commandBuffer);
    m_CommandMemoryAreaSize = commandBufferSize / sizeof(m_CommandMemoryArea[0]);
}

void CommandBuffer::Finalize() NN_NOEXCEPT
{
}

void* CommandBuffer::AllocMemory( size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( size >= sizeof(Command) );

    void* ptr = NULL;

    if ( m_CommandMemoryAreaZeroFlag ) return NULL;

    const size_t count = ( size + sizeof(m_CommandMemoryArea[0]) - 1 ) / sizeof(m_CommandMemoryArea[0]);
    NN_SDK_ASSERT( count <= m_CommandMemoryAreaSize );

    // 処理途中で FreeMemory() が呼ばれて End の位置が変わる可能性があるため退避する
    volatile uintptr_t curEnd = m_CommandMemoryAreaEnd;

    if ( curEnd > m_CommandMemoryAreaBegin ) {
        // 連続領域の場合
        if ( m_CommandMemoryAreaBegin + count <= curEnd ) {
            ptr = &m_CommandMemoryArea[m_CommandMemoryAreaBegin];
            m_CommandMemoryAreaBegin += count;
        }
    }
    else {
        // 非連続領域の場合
        if ( m_CommandMemoryAreaBegin + count <= m_CommandMemoryAreaSize ) {
            ptr = &m_CommandMemoryArea[m_CommandMemoryAreaBegin];
            m_CommandMemoryAreaBegin += count;
        }
        else if ( count <= curEnd ) {
            ptr = &m_CommandMemoryArea[0];
            m_CommandMemoryAreaBegin = count;
        }
    }

    if ( ptr == NULL )
    {
        return NULL;
    }

    if ( m_CommandMemoryAreaBegin == curEnd ) {
        m_CommandMemoryAreaZeroFlag = true;
    }

    Command* command = reinterpret_cast<Command*>(ptr);
    NN_SDK_ASSERT_NOT_NULL(command);
    command->memory_next = m_CommandMemoryAreaBegin;

    return command;
}

//---------------------------------------------------------------------------
//! @brief    コマンドバッファのメモリを解放します。
//!
//! @param[in]    lastCommand  開放するメモリの最後のコマンド
//!
//---------------------------------------------------------------------------
void CommandBuffer::FreeMemory( Command* lastCommand ) NN_NOEXCEPT
{
    m_CommandMemoryAreaEnd = lastCommand->memory_next;
    m_CommandMemoryAreaZeroFlag = false;
}

size_t CommandBuffer::GetCommandBufferSize() const NN_NOEXCEPT
{
    return m_CommandMemoryAreaSize * sizeof( m_CommandMemoryArea[0] );
}

size_t CommandBuffer::GetAllocatableCommandSize() const NN_NOEXCEPT
{
    if ( m_CommandMemoryAreaZeroFlag )
    {
        return 0;
    }

    size_t count;

    // 処理途中で AllocMemory(), FreeMemory() が呼ばれて Begin, End の位置が変わる可能性があるため退避する
    volatile uintptr_t curBegin = m_CommandMemoryAreaBegin;
    volatile uintptr_t curEnd = m_CommandMemoryAreaEnd;

    if ( curEnd > curBegin )
    {
        // 連続領域の場合
        count =  curEnd - curBegin;
    }
    else
    {
        // 非連続領域の場合
        count = std::max(
            m_CommandMemoryAreaSize - curBegin,
            static_cast<size_t>(curEnd)
        );
    }
    return count * sizeof( m_CommandMemoryArea[0] );
}

size_t CommandBuffer::GetAllocatedCommandBufferSize() const NN_NOEXCEPT
{
    if ( m_CommandMemoryAreaZeroFlag )
    {
        return GetCommandBufferSize();
    }

    size_t count;

    // 処理途中で AllocMemory(), FreeMemory() が呼ばれて Begin, End の位置が変わる可能性があるため退避する
    volatile uintptr_t curBegin = m_CommandMemoryAreaBegin;
    volatile uintptr_t curEnd = m_CommandMemoryAreaEnd;

    if ( curEnd > curBegin )
    {
        // 連続領域の場合
        count = curBegin + ( m_CommandMemoryAreaSize - curEnd );
    }
    else
    {
        // 非連続領域の場合
        count = curBegin - curEnd;
    }

    return count * sizeof( m_CommandMemoryArea[0] );
}

NN_DEFINE_STATIC_CONSTANT( const uint32_t CommandManager::InvalidCommand );

CommandManager::CommandManager() NN_NOEXCEPT
: m_Available( false )
, m_pProcessCommandListFunc( NULL )
, m_pRequestProcessCommandFunc( NULL )
, m_IsInitializedSendMessageQueue(false)
, m_IsInitializedRecvMessageQueue(false)
{
}

CommandManager::~CommandManager() NN_NOEXCEPT
{
    Finalize();
}

void CommandManager::Initialize( void* buffer, size_t bufferSize, size_t commandBufferSize, int queueCount, ProcessCommandListFunc func ) NN_NOEXCEPT
{
    NN_UNUSED(bufferSize);
    NN_SDK_ASSERT_NOT_NULL( func );
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, GetRequiredMemSize(commandBufferSize, queueCount));

    m_pProcessCommandListFunc = func;
    m_pRequestProcessCommandFunc = NULL;

    nn::util::BytePtr ptr(buffer);
    m_CommandBuffer.Initialize( buffer, commandBufferSize );
    ptr.Advance(commandBufferSize);

    m_CommandListBegin = NULL;
    m_CommandListEnd = NULL;

    m_CommandTag = 0;

    m_pSendCommandQueueBuffer = reinterpret_cast<uintptr_t*>(ptr.Get());
    ptr.Advance(GetSendQueueBufferSize(queueCount));
    m_pRecvCommandQueueBuffer = reinterpret_cast<uintptr_t*>(ptr.Get());

    if (!m_IsInitializedSendMessageQueue)
    {
        nn::os::InitializeMessageQueue(&m_SendCommandQueue, m_pSendCommandQueueBuffer, GetSendQueueCount(queueCount));
        m_IsInitializedSendMessageQueue = true;
    }

    if (!m_IsInitializedRecvMessageQueue)
    {
        nn::os::InitializeMessageQueue(&m_RecvCommandQueue, m_pRecvCommandQueueBuffer, GetRecvQueueCount(queueCount));
        m_IsInitializedRecvMessageQueue = true;
    }

    m_CommandListCount = 0;

    m_Available = true;
}

size_t CommandManager::GetRequiredMemSize(size_t commandBufferSize, int queueCount)
{
    return commandBufferSize + GetSendQueueBufferSize(queueCount) + GetRecvQueueBufferSize(queueCount);
}

void CommandManager::Finalize() NN_NOEXCEPT
{
    if (m_IsInitializedSendMessageQueue)
    {
        nn::os::FinalizeMessageQueue(&m_SendCommandQueue);
        m_IsInitializedSendMessageQueue = false;
    }
    if (m_IsInitializedRecvMessageQueue)
    {
        nn::os::FinalizeMessageQueue(&m_RecvCommandQueue);
        m_IsInitializedRecvMessageQueue = false;
    }

    m_CommandBuffer.Finalize();

    m_Available = false;
}

void* CommandManager::AllocMemory( size_t size, bool forceProcessCommandFlag  ) NN_NOEXCEPT
{
#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    NN_UNUSED( forceProcessCommandFlag );
#endif
    void* ptr = TryAllocMemory( size );
    if ( ptr != nullptr )
    {
        // メモリ取得に成功したのでコマンド数を増やす
        ++m_AllocatedCommandCount;
        return ptr;
    }

    RecvCommandReply();
    ptr = TryAllocMemory( size );
    if ( ptr != nullptr )
    {
        // メモリ取得に成功したのでコマンド数を増やす
        ++m_AllocatedCommandCount;
        return ptr;
    }

    // 強制フラッシュ
#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    while( ptr == nullptr )
    {
        if ( m_CommandListBegin != nullptr )
        {
            // VCMD 版ではないため、forceProcessCommandFlag を立てる必要はない
            uint32_t tag = FlushCommand( true, false );
            WaitCommandReply( tag );
        }
        else
        {
            RecvCommandReplySync();
        }
        ptr = TryAllocMemory( size );
    }
#else
    if ( forceProcessCommandFlag )
    {
        NN_ATK_WARNING( "Failed to allocate command buffer(size:%zu). (current:%zu/%zu) (waitSoundThread:%dms)"
            , size
            , GetAllocatedCommandBufferSize()
            , GetCommandBufferSize()
            , driver::SoundThread::GetInstance().GetRendererEventWaitTimeMilliSeconds() );
        // ドライバーコマンドのバッファが枯渇すると意図しない挙動が起こる可能性があるため
        // ProcessCommand を呼んでコマンドバッファを確保する。
        while ( ptr == nullptr )
        {
            ProcessCommand();
            ptr = TryAllocMemory( size );
        }
    }
#endif

    // メモリ取得に成功していればコマンド数を増やす
    if( ptr != nullptr )
    {
        ++m_AllocatedCommandCount;
    }
    else
    {
        // 非 VoiceCommand 版と VoiceCommand 版の forceProcessCommandFlag 有効時は確保に失敗することはないため、ここを通ることはない
        NN_SDK_ASSERT( false, "Failed to allocate command buffer(size:%zu). Please increase command buffer size. (current:%zu/%zu) (waitSoundThread:%dms)"
            , size
            , GetAllocatedCommandBufferSize()
            , GetCommandBufferSize()
            , driver::SoundThread::GetInstance().GetRendererEventWaitTimeMilliSeconds() );
    }

    return ptr;
}

void* CommandManager::TryAllocMemory( size_t size ) NN_NOEXCEPT
{
    return m_CommandBuffer.AllocMemory( size );
}

uint32_t CommandManager::PushCommand( Command* command ) NN_NOEXCEPT
{
    if ( m_CommandListEnd == NULL )
    {
        m_CommandListBegin = command;
        m_CommandListEnd = command;
    }
    else
    {
        m_CommandListEnd->next = command;
        m_CommandListEnd = command;
    }

    command->next = NULL;

    return m_CommandTag;
}

uint32_t CommandManager::FlushCommand( bool forceFlag ) NN_NOEXCEPT
{
    return FlushCommand(forceFlag, true);
}

//---------------------------------------------------------------------------
//! @brief    コマンドをフラッシュします。
//!
//!           forceFlag が true の時は、必ず処理を行います。
//!           コマンドが１つも登録されていない場合は、ダミーのコマンドを発行します。
//!           受信側の処理が滞っている場合は、空くまで待ちます。
//!
//!           forceFlag が false の時は、必要以上の処理は行いません。
//!           コマンドが１つも登録されていない場合は、何もせずに返ります。
//!           受信側の処理が滞っている場合も、何もせずに返ります。
//!           何もせずに返った場合、返り値は0（無効値）になることに注意してください。
//!
//! @param[in]    forceFlag  フラッシュ処理を強制するかどうかのフラグです。
//! @param[in]    forceProcessCommandFlag forceFlag 有効時、強制フラッシュ直前に ProcessCommand を呼び出すかどうかのフラグです。
//!               非 VCMD 版や forceFlag が false の場合は、動作に影響しません。
//!
//!               m_SendCommandQueue の send と recv が同一スレッドで行われる場合、SendMessageQueue が同期待ちし続ける可能性があるための措置です。
//!               現状 VCMD 版の DriverCommand だけ、send と recv が同一スレッドで行われる可能性があります。
//!
//! @return       コマンドタグを返します。
//!               ただし、forceFlag が false の時に、コマンドフラッシュが
//!               行われない場合は、0（無効値）を返します。
//!
//---------------------------------------------------------------------------
uint32_t CommandManager::FlushCommand( bool forceFlag, bool forceProcessCommandFlag ) NN_NOEXCEPT
{
#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    NN_UNUSED( forceProcessCommandFlag );
#endif
    if ( m_CommandListBegin == NULL ) {
        if ( ! forceFlag ) return 0;

        // ダミーコマンド発行
        Command* command = AllocCommand<Command>();
        command->id = InvalidCommand;
        PushCommand(command);
    }

    uintptr_t msg = reinterpret_cast<uintptr_t>(m_CommandListBegin);
    uint32_t tag = m_CommandTag;


    m_CommandListCount.fetch_add(1);

    bool result = nn::os::TrySendMessageQueue( &m_SendCommandQueue, msg );
    if ( result == false )
    {
        if ( forceFlag )
        {
            if ( m_pRequestProcessCommandFunc )
            {
                m_pRequestProcessCommandFunc();
            }
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            if ( forceProcessCommandFlag )
            {
                // VCMD 版の DriverCommand で m_SendCommandQueue を Send するスレッド(FlushCommandを呼ぶスレッド)と
                // Recv するスレッド(VoiceCommandProcessを呼ぶスレッド)が同一の場合に
                // SendMessageQueue から返ってこなくなるため、一度だけ ProcessCommand を呼んでキュー処理を行う
                ProcessCommand();
            }
#endif
            nn::os::SendMessageQueue( &m_SendCommandQueue, msg );
        }
        else
        {
            return 0;
        }
    }

    m_CommandListBegin->tag = tag;
    m_CommandTag++;

    m_CommandListBegin = NULL;
    m_CommandListEnd = NULL;

    return tag;
}

void CommandManager::FinalizeCommandList( Command* commandList ) NN_NOEXCEPT
{
    m_FinishCommandTag = commandList->tag;

    Command* command = commandList;

    while( command != NULL )
    {
        --m_AllocatedCommandCount;
        if ( command->next == NULL ) {
            m_CommandBuffer.FreeMemory( command );
            break;
        }
        command = command->next;
    }
}

bool CommandManager::IsFinishCommand( uint32_t tag ) const NN_NOEXCEPT
{
    static const uint32_t limit = (1 << ((sizeof(tag) * 8) - 2) );

    if ( tag > m_FinishCommandTag )
    {
        return ( tag - m_FinishCommandTag >= limit );
    }
    else
    {
        return ( m_FinishCommandTag - tag < limit );
    }
}

void CommandManager::RecvCommandReply() NN_NOEXCEPT
{
    uintptr_t msg;
    while( nn::os::TryReceiveMessageQueue( &msg, &m_RecvCommandQueue ) )
    {
        Command* commandList = reinterpret_cast<Command*>(msg);
        FinalizeCommandList( commandList );
    }
}

Command* CommandManager::RecvCommandReplySync() NN_NOEXCEPT
{
    uintptr_t msg;
    bool result = nn::os::TryReceiveMessageQueue( &msg, &m_RecvCommandQueue );
    if ( ! result )
    {
        if ( m_pRequestProcessCommandFunc )
        {
            m_pRequestProcessCommandFunc();
        }
        nn::os::ReceiveMessageQueue( &msg, &m_RecvCommandQueue );
    }

    Command* commandList = reinterpret_cast<Command*>(msg);
    FinalizeCommandList( commandList );

    return reinterpret_cast<Command*>(msg);
}

void CommandManager::WaitCommandReply( uint32_t tag ) NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    NN_UNUSED(tag);
    while( ProcessCommand() ) {}
#else
    for ( ;; )
    {
        Command* commandList = RecvCommandReplySync();

        if ( tag == commandList->tag ) break;
    }
#endif
}

size_t CommandManager::GetCommandBufferSize() const NN_NOEXCEPT
{
    return m_CommandBuffer.GetCommandBufferSize();
}

size_t CommandManager::GetAllocatableCommandSize() const NN_NOEXCEPT
{
    return m_CommandBuffer.GetAllocatableCommandSize();
}

size_t CommandManager::GetAllocatedCommandBufferSize() const NN_NOEXCEPT
{
    return m_CommandBuffer.GetAllocatedCommandBufferSize();
}

int CommandManager::GetAllocatedCommandCount() const NN_NOEXCEPT
{
    return m_AllocatedCommandCount;
}

bool CommandManager::ProcessCommand() NN_NOEXCEPT
{
    uintptr_t msg;
    bool result = nn::os::TryReceiveMessageQueue( &msg, &m_SendCommandQueue );
    if ( result == false )
    {
        return false;
    }

    m_CommandListCount.fetch_sub(1);

    Command* commandList = reinterpret_cast<Command*>(msg);

    if ( commandList->id != InvalidCommand )
    {
        m_pProcessCommandListFunc( commandList );

    }

    nn::os::SendMessageQueue( &m_RecvCommandQueue, msg );

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    RecvCommandReply();
#endif

    return true;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
