﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_StreamSoundLoader.h>

#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_WaveFileReader.h>
#include <nn/atk/detail/atk_MemoryFileStream.h>

#include <nn/atk/fnd/io/atkfnd_FileStreamImpl.h>
#include <nn/atk/detail/atk_Macro.h>

//#define NN_ATK_DEBUG_DUMP_LOOP_PARAM

//#define NN_ATK_ENABLE_DEBUG_TRACE_FUNC_STREAM_SOUND_LOADER
#if defined(NN_ATK_ENABLE_DEBUG_TRACE_FUNC_STREAM_SOUND_LOADER)
    #define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) \
            NN_DETAIL_ATK_INFO("[%p](%s) %s%s\n", this, NN_ATK_FILENAME, __FUNCTION__, AdditionalMessage)
    #define NN_ATK_DEBUG_TRACE_FUNC() \
            NN_ATK_DEBUG_TRACE_FUNC_MSG("")
#else
    #define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) static_cast<void>(0)
    #define NN_ATK_DEBUG_TRACE_FUNC() static_cast<void>(0)
#endif // NN_ATK_ENABLE_DEBUG_TRACE_FUNC_STREAM_SOUND_LOADER

//#define NN_ATK_ENABLE_DEBUG_DUMP_STREAM_SOUND_LOADER
#if defined(NN_ATK_ENABLE_DEBUG_DUMP_STREAM_SOUND_LOADER)
    #define NN_ATK_DEBUG_DUMP_VARIABLE(variable) (variable).Dump()
#else
    #define NN_ATK_DEBUG_DUMP_VARIABLE(variable) static_cast<void>(0)
#endif // NN_ATK_ENABLE_DEBUG_DUMP_STREAM_SOUND_LOADER

namespace {

const uint8_t DefaultLpfFreq       = 64;   // フィルタがかかっていない
const uint8_t DefaultBiquadType    = 0;    // フィルタ未使用
const uint8_t DefaultBiquadValue   = 0;    // フィルタがかかっていない

const int LoopDecodeStartOffset = 1;

typedef nn::util::IntrusiveList<nn::atk::detail::IStreamDataDecoderManager, nn::util::IntrusiveListMemberNodeTraits<nn::atk::detail::IStreamDataDecoderManager, &nn::atk::detail::IStreamDataDecoderManager::m_Link>> StreamDataDecoderManagerList;
static StreamDataDecoderManagerList g_StreamDataDecoderManagerList;

} // anonymous namespace

namespace nn {
namespace atk {
namespace detail {

void StreamDataInfoDetail::SetStreamSoundInfo(const StreamSoundFile::StreamSoundInfo& info, bool isCrc32CheckEnabled) NN_NOEXCEPT
{
    sampleFormat = WaveFileReader::GetSampleFormat( info.encodeMethod );
    sampleRate = info.sampleRate;
    loopFlag = (info.isLoop != 0);
    loopStart = info.loopStart;
    sampleCount = info.frameCount;
    originalLoopStart = info.originalLoopStart;
    originalLoopEnd = info.originalLoopEnd;
    blockSampleCount = info.oneBlockSamples;
    blockSize = info.oneBlockBytes;
    lastBlockSampleCount = info.lastBlockSamples;
    lastBlockSize = info.lastBlockPaddedBytes;
    revisionValue = info.crc32Value;
    isRevisionCheckEnabled = isCrc32CheckEnabled;
    regionCount = info.regionCount;
}

namespace driver {

NN_DEFINE_STATIC_CONSTANT( const size_t StreamSoundLoader::DataBlockSizeBase );
NN_DEFINE_STATIC_CONSTANT( const size_t StreamSoundLoader::DataBlockSizeMargin );
NN_DEFINE_STATIC_CONSTANT( const size_t StreamSoundLoader::DataBlockSizeMax );
NN_DEFINE_STATIC_CONSTANT( const int StreamSoundLoader::FileStreamBufferSize );
NN_DEFINE_STATIC_CONSTANT( const int StreamSoundLoader::LoadBufferChannelCount );
NN_DEFINE_STATIC_CONSTANT( const size_t StreamSoundLoader::LoadBufferSize );
NN_ALIGNAS(256) uint8_t StreamSoundLoader::g_LoadBuffer[ LoadBufferSize ];

StreamSoundLoader::StreamSoundLoader() NN_NOEXCEPT
: m_pFileStream(nullptr)
, m_pStreamDataDecoder(nullptr)
, m_pStreamDataDecoderManager(nullptr)
{
    uint32_t taskCount = m_StreamDataLoadTaskPool.Create(
        m_StreamDataLoadTaskArea,
        sizeof(m_StreamDataLoadTaskArea)
    );
#if defined(NN_SDK_BUILD_RELEASE)
    NN_UNUSED( taskCount );
#endif
    NN_SDK_ASSERT( taskCount == StreamDataLoadTaskMax );
    std::memset(m_FilePath, 0, sizeof(char) * FilePathMax);
}

StreamSoundLoader::~StreamSoundLoader() NN_NOEXCEPT
{
    WaitFinalize();

    if (m_pStreamDataDecoderManager != nullptr)
    {
        if (m_pStreamDataDecoder != nullptr)
        {
            m_pStreamDataDecoderManager->FreeDecoder(m_pStreamDataDecoder);
            m_pStreamDataDecoder = nullptr;
        }
        m_pStreamDataDecoderManager = nullptr;
    }

    NN_SDK_ASSERT( m_StreamDataLoadTaskPool.Count() == StreamDataLoadTaskMax );
    m_StreamDataLoadTaskPool.Destroy();
}

void StreamSoundLoader::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(SoundSystem::detail_IsTaskThreadEnabled(), "Task thread is not enabled.");
    NN_ATK_DEBUG_TRACE_FUNC();

    WaitFinalize();

    m_LoadingDataBlockIndex = 0;
    m_LastBlockIndex = 0xffffffff;
    m_LoopStartBlockIndex = 0;
    m_LoopStartFilePos = 0;
    m_LoopStartBlockSampleOffset = 0;
    m_LoopJumpFlag = false;
    m_LoadFinishFlag = false;

    m_RegionManager.Initialize();

    m_SampleFormat = SampleFormat_DspAdpcm;
    m_DecodeMode = detail::DecodeMode_Invalid;

    m_pStreamDataDecoderManager = nullptr;
    m_pStreamDataDecoder = nullptr;
}

void StreamSoundLoader::Finalize() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    CancelRequest();
    RequestClose();
}

void StreamSoundLoader::RegisterStreamDataDecoderManager(IStreamDataDecoderManager * pManager) NN_NOEXCEPT
{
    g_StreamDataDecoderManagerList.push_back(*pManager);
}

void StreamSoundLoader::UnregisterStreamDataDecoderManager(IStreamDataDecoderManager * pManager) NN_NOEXCEPT
{
    g_StreamDataDecoderManagerList.erase(g_StreamDataDecoderManagerList.iterator_to(*pManager));
}

void StreamSoundLoader::WaitFinalize() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    m_StreamHeaderLoadTask.Wait();
    m_StreamCloseTask.Wait();
    for ( StreamDataLoadTaskList::iterator itr = m_StreamDataLoadTaskList.begin();
          itr != m_StreamDataLoadTaskList.end();
    )
    {
        StreamDataLoadTaskList::iterator curItr = itr++;
        StreamDataLoadTask* task = &*curItr;
        task->Wait();
        m_StreamDataLoadTaskList.erase( m_StreamDataLoadTaskList.iterator_to( *task ) );
        m_StreamDataLoadTaskPool.Free( task );
        // NN_DETAIL_ATK_INFO( "Task Cancel %08x\n", task );
    }
}

void StreamSoundLoader::CancelRequest() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();
    // TODO: キャストは clang ビルドエラー回避のための暫定対応
    TaskManager::GetInstance().CancelTaskById( static_cast<uint32_t>( reinterpret_cast<uint64_t>(this) ) );
}

void* StreamSoundLoader::detail_SetFsAccessLog( detail::fnd::FsAccessLog* pFsAccessLog ) NN_NOEXCEPT
{
    if( m_pFileStream != nullptr )
    {
        if ( m_pFileStream->CanSetFsAccessLog() )
        {
            return m_pFileStream->SetFsAccessLog(pFsAccessLog);
        }
    }
    return nullptr;
}

position_t StreamSoundLoader::detail_GetCurrentPosition() NN_NOEXCEPT
{
    if( m_pFileStream != nullptr )
    {
        if ( m_pFileStream->IsCacheEnabled() )
        {
            return m_pFileStream->GetCurrentPosition();
        }
    }
    return 0;
}

position_t StreamSoundLoader::detail_GetCachePosition() NN_NOEXCEPT
{
    if( m_pFileStream != nullptr )
    {
        if ( m_pFileStream->IsCacheEnabled() )
        {
            return m_pFileStream->GetCachePosition();
        }
    }
    return 0;
}

size_t StreamSoundLoader::detail_GetCachedLength() NN_NOEXCEPT
{
    if( m_pFileStream != nullptr )
    {
        if ( m_pFileStream->IsCacheEnabled() )
        {
            return m_pFileStream->GetCachedLength();
        }
    }
    return 0;
}

void StreamSoundLoader::RequestLoadHeader() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    m_StreamHeaderLoadTask.m_pLoader = this;

    // TODO: キャストは clang ビルドエラー回避のための暫定対応
    m_StreamHeaderLoadTask.SetId(static_cast<uint32_t>(reinterpret_cast<uint64_t>(this)));
    // NN_DETAIL_ATK_INFO("[%s](%p) AppendTask ", __FUNCTION__, this);
    detail::TaskManager::GetInstance().AppendTask(
            &m_StreamHeaderLoadTask,
            detail::TaskManager::TaskPriority_Middle );
    // NN_DETAIL_ATK_INFO(" -- [%s](%p) end\n", __FUNCTION__, this);
}

void StreamSoundLoader::RequestClose() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    m_StreamCloseTask.Wait();
    m_StreamCloseTask.m_pLoader = this;
    detail::TaskManager::GetInstance().AppendTask(
            &m_StreamCloseTask,
            detail::TaskManager::TaskPriority_Middle );
}


void StreamSoundLoader::RequestLoadData( void* bufferAddress[], uint32_t bufferBlockIndex, position_t startOffsetSamples, position_t prefetchOffsetSamples, int priority ) NN_NOEXCEPT
{
    // ロードタスクの発行
    StreamDataLoadTask* task = m_StreamDataLoadTaskPool.Alloc();
    NN_SDK_ASSERT_NOT_NULL( task );

    task->m_pLoader = this;
    task->m_BufferBlockIndex = bufferBlockIndex;
    task->m_StartOffsetSamples = startOffsetSamples;
    task->m_PrefetchOffsetSamples = prefetchOffsetSamples;
    // TODO: キャストは clang ビルドエラー回避のための暫定対応
    task->SetId(static_cast<uint32_t>(reinterpret_cast<uint64_t>(this)));

    for ( auto ch = 0; ch < m_ChannelCount; ch++ )
    {
        task->m_BufferAddress[ch] = bufferAddress[ch];
    }

    m_StreamDataLoadTaskList.push_back( *task );
    // NN_DETAIL_ATK_INFO("[%s](%p) AppendTask ", __FUNCTION__, this);
    detail::TaskManager::GetInstance().AppendTask( task, detail::TaskManager::TaskPriority(priority) );
    // NN_DETAIL_ATK_INFO(" -- [%s](%p) end\n", __FUNCTION__, this);

    // NN_DETAIL_ATK_INFO("loadtask append buf(%d)\n", m_LoadingBufferBlockIndex);
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  タスクの状態をチェックし、完了したものの解放処理を行う

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSoundLoader::Update() NN_NOEXCEPT
{
    for ( StreamDataLoadTaskList::iterator itr = m_StreamDataLoadTaskList.begin();
          itr != m_StreamDataLoadTaskList.end();
        )
    {
        StreamDataLoadTaskList::iterator curItr = itr++;
        StreamDataLoadTask* task = &*curItr;
        if ( task->GetStatus() != Task::Status_Done && task->GetStatus() != Task::Status_Cancel )
        {
            break;
        }

        task->Wait();
        m_StreamDataLoadTaskList.erase( m_StreamDataLoadTaskList.iterator_to( *task ) );
        m_StreamDataLoadTaskPool.Free( task );

        // NN_DETAIL_ATK_INFO("Free %08x %d\n",task,m_StreamDataLoadTaskList.size());
    }
}

void StreamSoundLoader::ForceFinish() NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstanceForTaskThread();
    DriverCommandStreamSoundForceFinish* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundForceFinish>( false );
    command->id = DriverCommandId_StrmForceFinish;
    command->player = m_PlayerHandle;
    cmdmgr.PushCommand(command);
    cmdmgr.FlushCommand(true, false);
}

bool StreamSoundLoader::IsBusy() const NN_NOEXCEPT
{
    if ( m_LoadFinishFlag ) return false;
    if ( ! m_StreamDataLoadTaskList.empty() ) return true;
    return false;
}

bool StreamSoundLoader::IsInUse() NN_NOEXCEPT
{
    Update();

    if ( !m_StreamDataLoadTaskList.empty() ) return true;

    return false;
}

fnd::FndResult StreamSoundLoader::Open() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    if (m_pExternalData != nullptr)
    {
        m_pFileStream = new(m_FileStreamBuffer) detail::MemoryFileStream( m_pExternalData, m_ExternalDataSize );
    }
    else
    {
        // フックが有効な時はそちらからのオープンを試みる
        if (m_FileStreamHookParam.IsHookEnabled())
        {
            m_pFileStream = m_FileStreamHookParam.pSoundArchiveFilesHook->OpenFile(
                    m_FileStreamBuffer,
                    FileStreamBufferSize,
                    m_pCacheBuffer,
                    m_CacheSize,
                    m_FileStreamHookParam.itemLabel,
                    detail::SoundArchiveFilesHook::FileTypeStreamBinary);
        }

        // フックされてないときか、オープンできなかったときは通常のオープン
        if (m_pFileStream == nullptr)
        {
            // NN_DETAIL_ATK_INFO("(%p) [%s] m_pFilePath(%p:%s)\n", this, __FUNCTION__, m_pFilePath, m_pFilePath);
            // ファイルのオープン
            detail::fnd::FileStream* stream = new(m_FileStreamBuffer) detail::fnd::FileStreamImpl();
            NN_SDK_ASSERT_NOT_NULL(stream);
            fnd::FndResult result = stream->Open(m_FilePath, fnd::FileStream::AccessMode_Read);
            if (result.IsFailed())
            {
                return result;
            }
            // キャッシュ用バッファを設定
            if (stream->IsOpened())
            {
                if (IsStreamCacheEnabled())
                {
                    stream->EnableCache(m_pCacheBuffer, m_CacheSize);
                }
            }

            m_pFileStream = stream;
            // NN_DETAIL_ATK_INFO("(%p) [%s] stream(%p)\n", this, __FUNCTION__, stream);
        }
    }

    m_FileLoader.Initialize( m_pFileStream );

    return fnd::FndResult(fnd::FndResultType::FndResultType_True);
}

void StreamSoundLoader::Close() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    // デコーダを返却
    if (m_pStreamDataDecoderManager != nullptr)
    {
        if (m_pStreamDataDecoder != nullptr)
        {
            m_pStreamDataDecoderManager->FreeDecoder(m_pStreamDataDecoder);
            m_pStreamDataDecoder = nullptr;
        }
        m_pStreamDataDecoderManager = nullptr;
    }

    if (m_pFileStream)
    {
        m_pFileStream->Close();
        m_pFileStream = nullptr;
        m_FileLoader.Finalize();
    }
}

void StreamSoundLoader::LoadHeader() NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    // ヘッダーロード
    DriverCommand& cmdmgr = DriverCommand::GetInstanceForTaskThread();
    DriverCommandStreamSoundLoadHeader* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundLoadHeader>( false );
    command->id = DriverCommandId_StrmLoadHeader;
    command->player = m_PlayerHandle;
    command->assignNumber = m_AssignNumber;

    bool result;

    switch( static_cast<int32_t>(m_FileType) )
    {
    case StreamFileType_Bfstm:
        result = LoadHeader1( command );
        break;
    case StreamFileType_Opus:
        result = LoadHeaderForOpus( command, m_FileType, m_DecodeMode );
        break;
    default:
        NN_ATK_WARNING("Unexpected file type.");
        result = false;
        break;
    }
    command->result = result;

    cmdmgr.PushCommand(command);
    cmdmgr.FlushCommand(true, false);
}

bool StreamSoundLoader::LoadHeader1( DriverCommandStreamSoundLoadHeader* command ) NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    // ファイルヘッダのロード
    StreamSoundFileReader reader;
    if ( ! m_FileLoader.LoadFileHeader( &reader, g_LoadBuffer, LoadBufferSize ) )
    {
        return false;
    }

    // ストリーム情報の読み取り
    detail::StreamSoundFile::StreamSoundInfo info;
    if ( ! reader.ReadStreamSoundInfo( &info ) )
    {
        return false;
    }
    // StartInfo によりループ情報が書き換えられる場合の処理
    {
        if( m_IsLoopFlagEnabled )
        {
            info.isLoop = m_LoopFlag;
        }
    }

    uint32_t channelCount = reader.GetChannelCount();
    m_ChannelCount = channelCount;
    m_DataInfo->channelCount = channelCount;
    m_DataInfo->SetStreamSoundInfo(info, reader.IsCrc32CheckAvailable());

    // bXstm 内にトラック情報が埋められている場合 (バイナリバージョン 0.2.0.0 まで)
    if ( reader.IsTrackInfoAvailable() )
    {
        if ( !ReadTrackInfoFromStreamSoundFile(reader) )
        {
            return false;
        }
    }

    // ADPCM
    m_SampleFormat = m_DataInfo->sampleFormat;
    if ( m_SampleFormat == SampleFormat_DspAdpcm )
    {
        // ADPCM の情報を読み取る
        if ( !SetAdpcmInfo(reader, channelCount, command->adpcmParam) )
        {
            return false;
        }
    }
    else
    {
        for ( unsigned int ch = 0; ch < channelCount; ch++ )
        {
            command->adpcmParam[ch] = nullptr;
        }
    }

    m_DataStartFilePos = reader.GetSampleDataOffset();
    m_LastBlockIndex = m_DataInfo->GetLastBlockIndex();
    m_LoopStartBlockIndex = m_DataInfo->GetLoopStartBlockIndex(0);
    m_LoopStartFilePos = m_DataStartFilePos + m_DataInfo->blockSize * m_ChannelCount * m_LoopStartBlockIndex;
    m_LoopStartBlockSampleOffset = 0;

    NN_SDK_ASSERT( m_DataInfo->GetLoopStartInBlock() == 0 );
    NN_SDK_ASSERT( m_LastBlockIndex == info.blockCount - 1 );

    m_DataInfo->isRegionIndexCheckEnabled = reader.IsRegionIndexCheckAvailable();
    if ( !m_RegionManager.InitializeRegion(&m_FileLoader, m_DataInfo) )
    {
        return false;
    }

    UpdateLoadingDataBlockIndex();

    return true;
}

bool StreamSoundLoader::LoadHeaderForOpus( DriverCommandStreamSoundLoadHeader* command, StreamFileType type, DecodeMode decodeMode ) NN_NOEXCEPT
{
    NN_ATK_DEBUG_TRACE_FUNC();

    DecodeMode actualDecodeMode = decodeMode;
    if(actualDecodeMode == detail::DecodeMode_Default)
    {
        actualDecodeMode = detail::DecodeMode_Cpu;
    }

    m_pStreamDataDecoderManager = SelectStreamDataDecoderManager(type, actualDecodeMode);
    if (m_pStreamDataDecoderManager == nullptr)
    {
        return false;
    }

    // デコーダを取得
    if (m_pStreamDataDecoder == nullptr)
    {
        m_pStreamDataDecoder = m_pStreamDataDecoderManager->AllocDecoder();
        if ( m_pStreamDataDecoder == nullptr )
        {
            NN_ATK_WARNING("Decoder allocation failed. Please check decoder count setting.");
            return false;
        }
    }

    // ストリーム情報の読み取り
    IStreamDataDecoder::DataInfo info;
    bool result = m_pStreamDataDecoder->ReadDataInfo( &info, m_pFileStream );
    if ( ! result )
    {
        return false;
    }

    uint32_t channelCount = info.channelCount;
    m_ChannelCount = channelCount;
    m_DataInfo->channelCount = channelCount;
    SetStreamSoundInfoForOpus(info);

    // 最後のブロックのサンプル数は計算で求める
    m_LastBlockIndex = m_DataInfo->GetLastBlockIndex();
    if ( m_DataInfo->loopFlag )
    {
        m_DataInfo->lastBlockSampleCount =
            m_DataInfo->sampleCount - m_DataInfo->blockSampleCount * m_LastBlockIndex;
    }
    else
    {
        m_DataInfo->lastBlockSampleCount = m_DataInfo->blockSampleCount;
    }

#if defined(NN_ATK_DEBUG_DUMP_LOOP_PARAM)
    // 補正前の情報表示
    m_DataInfo->Dump(false);
#endif

    // トラックの情報は読み込まない。（bXstmではないのでトラック情報は含まれていない）
    // （StreamSoundPlayer::Setup のタイミングで bXsar から読み込んだ値を設定しています。）

    // ADPCMパラメータを無効化
    for ( unsigned int ch = 0; ch < channelCount; ch++ )
    {
        command->adpcmParam[ch] = nullptr;
    }

    // ループ情報を計算
    uint32_t loopStartBlockIndex = m_DataInfo->GetLoopStartBlockIndex(0);
    position_t loopStartBlockSampleOffset = m_DataInfo->loopStart - loopStartBlockIndex * m_DataInfo->blockSampleCount;
#if defined(NN_ATK_DEBUG_DUMP_LOOP_PARAM)
    // 補正前の情報表示
    NN_DETAIL_ATK_INFO("lastBlockIndex(%d) loopStartIndex(%d) loopStartBlockSampleOffset(%lld)\n", m_LastBlockIndex, loopStartBlockIndex, loopStartBlockSampleOffset);
#endif

    if ( m_DataInfo->loopFlag )
    {
        // ループ先補正
        size_t loopStartToBlockEnd = m_DataInfo->blockSampleCount - loopStartBlockSampleOffset;
#if defined(NN_ATK_DEBUG_DUMP_LOOP_PARAM)
        NN_DETAIL_ATK_INFO("loopStartToBlockEnd=(%zu)\n", loopStartToBlockEnd);
#endif
        // loopStartBlockSampleOffset が 0 でない場合は次のブロックをループ先にする
        if ( loopStartBlockSampleOffset > 0 )
        {
            m_DataInfo->loopStart += loopStartToBlockEnd;
            m_DataInfo->lastBlockSampleCount += loopStartToBlockEnd;
        }

        // 最後のブロックの長さが短すぎる場合は長くする
        if ( m_DataInfo->lastBlockSampleCount < m_DataInfo->blockSampleCount )
        {
            m_DataInfo->loopStart += m_DataInfo->blockSampleCount;
            m_DataInfo->lastBlockSampleCount += m_DataInfo->blockSampleCount;
        }

#if defined(NN_ATK_DEBUG_DUMP_LOOP_PARAM)
        // 補正後の情報表示
        m_DataInfo->Dump(false);
#endif

        // m_DataInfo->loopStart が補正されるため再計算
        loopStartBlockIndex = m_DataInfo->GetLoopStartBlockIndex(0);
        loopStartBlockSampleOffset = m_DataInfo->loopStart - loopStartBlockIndex * m_DataInfo->blockSampleCount;
    }

    m_LoopStartBlockIndex = loopStartBlockIndex;
    m_LoopStartBlockSampleOffset = loopStartBlockSampleOffset;
#if defined(NN_ATK_DEBUG_DUMP_LOOP_PARAM)
    // 補正前の情報表示
    NN_DETAIL_ATK_INFO("lastBlockIndex(%d) loopStartIndex(%d) loopStartBlockSampleOffset(%lld)\n", m_LastBlockIndex, loopStartBlockIndex, loopStartBlockSampleOffset);
#endif

    m_DataInfo->isRegionIndexCheckEnabled = false;
    if ( !m_RegionManager.InitializeRegion(&m_FileLoader, m_DataInfo) )
    {
        return false;
    }

    return true;
}

IStreamDataDecoderManager* StreamSoundLoader::SelectStreamDataDecoderManager( StreamFileType type, DecodeMode decodeMode ) NN_NOEXCEPT
{
    if(!g_StreamDataDecoderManagerList.empty())
    {
        for(auto itr = g_StreamDataDecoderManagerList.begin(); itr != g_StreamDataDecoderManagerList.end(); ++itr)
        {
            if(itr->GetStreamFileType() == type && itr->GetDecodeMode() == decodeMode)
            {
                return &(*itr);
            }
        }
    }

    char decoderName[32];
    switch(decodeMode)
    {
    case detail::DecodeMode_Cpu:
        nn::util::SNPrintf(decoderName, sizeof(decoderName), "OpusDecoder");
        break;
    case detail::DecodeMode_Accelerator:
        nn::util::SNPrintf(decoderName, sizeof(decoderName), "HardwareOpusDecoder");
        break;
    default:
        nn::util::SNPrintf(decoderName, sizeof(decoderName), "Decoder");
        break;
    }

    NN_ATK_WARNING("Decoder selection failed. %s is not prepared for.", decoderName);
    return nullptr;
}

void StreamSoundLoader::LoadData( void* bufferAddress[], uint32_t bufferBlockIndex, size_t startOffsetSamples, size_t prefetchOffsetSamples, TaskProfileLogger& logger ) NN_NOEXCEPT
{
    if ( m_LoadFinishFlag )
    {
        return;
    }

    // NN_DETAIL_ATK_INFO("exec(0x%p): bufIdx(%d)\n", this, m_BufferBlockIndex );

    DriverCommand& cmdmgr = DriverCommand::GetInstanceForTaskThread();
    DriverCommandStreamSoundLoadData* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundLoadData>( false );
    command->id = DriverCommandId_StrmLoadData;
    command->assignNumber = m_AssignNumber;

    // ★ 見直しが必要
    bool result;
    if (m_pFileStream != nullptr)
    {
        switch( static_cast<int32_t>(m_FileType) )
        {
        case StreamFileType_Bfstm:
            result = LoadData1( command, bufferAddress, bufferBlockIndex, startOffsetSamples, prefetchOffsetSamples, logger );
            break;
        case StreamFileType_Opus:
            result = LoadDataForOpus( command, bufferAddress, bufferBlockIndex, startOffsetSamples, prefetchOffsetSamples, logger );
            break;
        default:
            NN_ATK_WARNING("Unexpected file type.");
            result = false;
            break;
        }
    }
    else
    {
        result = false;
    }
    command->result = result;
    command->player = m_PlayerHandle;

    NN_ATK_DEBUG_DUMP_VARIABLE(command->loadDataParam);

    cmdmgr.PushCommand(command);
    cmdmgr.FlushCommand(true, false);
}

bool StreamSoundLoader::LoadData1(
    DriverCommandStreamSoundLoadData* command,
    void* bufferAddress[],
    uint32_t bufferBlockIndex,
    size_t startOffsetSamples,
    size_t prefetchOffsetSamples,
    TaskProfileLogger& logger
) NN_NOEXCEPT
{
    const auto beginTick = nn::os::GetSystemTick();
    LoadDataParam& loadDataParam = command->loadDataParam;

    // 開始オフセット
    position_t startOffsetSamplesInFrame = 0; // 波形バッファ先頭からのサンプル位置
    bool updateAdpcmContext = false; // ADPCMコンテキストの更新が必要かどうか
    int loopCount = 0;

    // 開始オフセットが指定されている場合
    if ( startOffsetSamples > 0 || prefetchOffsetSamples > 0 )
    {
        if ( !ApplyStartOffset(startOffsetSamples + prefetchOffsetSamples, &loopCount) )
        {
            // 範囲外のオフセットが指定された
            loadDataParam.samples = 0;
            return true;
        }

        UpdateLoadingDataBlockIndex();

        if ( m_SampleFormat == SampleFormat_DspAdpcm )
        {
            updateAdpcmContext = true;
        }
    }

    size_t totalBlockSamples = 0;
    position_t destAddressOffset = 0;
    position_t sampleBegin = m_RegionManager.GetCurrentRegion().current;

    // 現在のブロックサンプルが小さいとき、または、
    // 次のブロックサンプルが小さいとき、一緒にロードする
    bool firstBlock = true;
    // プリフェッチデータからストリームデータへの移行時は最初のブロックとして扱わない
    if ( prefetchOffsetSamples > 0 )
    {
        firstBlock = false;
    }

    bool isFirstDataLoad = true;
    loadDataParam.isStartOffsetOfLastBlockApplied = false;

    while( totalBlockSamples == 0 ||
        totalBlockSamples < DataBlockSizeMarginSamples ||
        m_RegionManager.GetCurrentRegion().Rest() < DataBlockSizeMarginSamples )
    {
        BlockInfo blockInfo;
        CalculateBlockInfo(blockInfo);

        if ( firstBlock )
        {
            startOffsetSamplesInFrame = blockInfo.GetStartOffsetInFrame();

            if ( updateAdpcmContext )
            {
                if ( !LoadAdpcmContextForStartOffset() )
                {
                    return false;
                }
            }
        }

        if (IsStreamCacheEnabled())
        {
            if (!LoadOneBlockDataViaCache(bufferAddress, blockInfo, destAddressOffset, firstBlock, updateAdpcmContext))
            {
                return false;
            }
        }
        else
        {
            if (!LoadOneBlockData(bufferAddress, blockInfo, destAddressOffset, firstBlock, updateAdpcmContext))
            {
                return false;
            }
        }

        totalBlockSamples += blockInfo.samples;
        destAddressOffset += blockInfo.copyByte;
        m_RegionManager.AddPosition(blockInfo.samples);

        // ロードデータブロックインデックスの更新
        NN_SDK_ASSERT( m_LoadingDataBlockIndex <= m_LastBlockIndex );
        m_LoadingDataBlockIndex++;

        // リージョン終端処理
        if ( m_RegionManager.GetCurrentRegion().IsEnd() )
        {
            if ( MoveNextRegion(&loopCount) )
            {
                UpdateLoadingDataBlockIndex();
            }

            // リージョン終端が直接 StartOffset で指定された場合は、
            // サンプル数チェックを行わないことを StreamSoundPlayer に伝える
            if ( startOffsetSamples > 0 || prefetchOffsetSamples > 0 )
            {
                if(isFirstDataLoad && totalBlockSamples < DataBlockSizeMarginSamples)
                {
                    loadDataParam.isStartOffsetOfLastBlockApplied = true;
                }
            }

            // ADPCMコンテキストの更新ができないため
            // ブロックを跨いだ一括ロードは禁止
            // PCM8/PCM16は可能であるが非対応
            // 対応する場合、１バッファで複数回loopCountを増やす対応が必要
            break;
        }

        firstBlock = false;
        isFirstDataLoad = false;
    } // while(ブロック)

    loadDataParam.adpcmContextEnable = false;
    if ( m_SampleFormat == SampleFormat_DspAdpcm )
    {
        if ( sampleBegin == 0 )
        {
            for ( auto ch = 0 ; ch < m_ChannelCount ; ch++ )
            {
                loadDataParam.adpcmContext[ch].audioAdpcmContext = m_AdpcmInfo[ch].beginContext.audioAdpcmContext;
            }
            loadDataParam.adpcmContextEnable = true;
        }
        else if ( m_DataInfo->loopFlag && sampleBegin == m_DataInfo->loopStart )
        {
            for ( auto ch = 0 ; ch < m_ChannelCount ; ch++ )
            {
                loadDataParam.adpcmContext[ch].audioAdpcmContext = m_AdpcmInfo[ch].loopContext.audioAdpcmContext;
            }
            loadDataParam.adpcmContextEnable = true;
        }
        else if ( sampleBegin == m_RegionManager.GetStartOffsetFrame() )
        {
            for ( auto ch = 0; ch < m_ChannelCount; ch++ )
            {
                loadDataParam.adpcmContext[ch].audioAdpcmContext = m_RegionManager.GetAdpcmContextForStartOffset(ch).audioAdpcmContext;
            }
            loadDataParam.adpcmContextEnable = true;
        }
    }

    loadDataParam.blockIndex = bufferBlockIndex; // バッファ識別番号
    loadDataParam.samples = totalBlockSamples;
    loadDataParam.sampleBegin = sampleBegin; // バッファ先頭のサンプル位置
    loadDataParam.sampleOffset = startOffsetSamplesInFrame;
    loadDataParam.sampleBytes = destAddressOffset;
    loadDataParam.loopCount = loopCount;
    loadDataParam.lastBlockFlag = m_LoadFinishFlag;

    // プロファイル
    if( logger.IsProfilingEnabled() )
    {
        const auto endTick = nn::os::GetSystemTick();

        TaskProfile profile;
        profile.type = TaskProfile::TaskProfileType_LoadStreamBlock;
        IStreamDataDecoder::CacheProfile cacheProfile = IStreamDataDecoder::CacheProfile();

        if ( IsStreamCacheEnabled() )
        {
            cacheProfile.cacheStartPosition   = detail_GetCachePosition();
            cacheProfile.cachedLength         = detail_GetCachedLength();
            cacheProfile.cacheCurrentPosition = detail_GetCurrentPosition();
            cacheProfile.player               = m_PlayerHandle;
        }
        profile.loadStreamBlock.SetData( beginTick, endTick, cacheProfile );
        logger.Record( profile );
    }

    return true;
} // NOLINT(impl/function_size)

bool StreamSoundLoader::LoadDataForOpus(
    DriverCommandStreamSoundLoadData* command,
    void* bufferAddress[],
    uint32_t bufferBlockIndex,
    size_t startOffsetSamples,
    size_t prefetchOffsetSamples,
    TaskProfileLogger& logger
) NN_NOEXCEPT
{
    NN_UNUSED(prefetchOffsetSamples);
    const auto beginTick = nn::os::GetSystemTick();
    LoadDataParam& loadDataParam = command->loadDataParam;

    position_t startOffsetSamplesInFrame = 0; // 波形バッファ先頭からのサンプル位置
    int loopCount = 0;
    nn::os::Tick fsAccessTick = nn::os::Tick( 0 );

    const bool isProfileEnabled = logger.IsProfilingEnabled() && m_pStreamDataDecoder != nullptr;
    if( isProfileEnabled )
    {
        m_pStreamDataDecoder->ResetDecodeProfile();
    }

    // 最後のブロックを超えたとき
    if ( m_LoadingDataBlockIndex > m_LastBlockIndex )
    {
        if ( m_DataInfo->loopFlag )
        {
            m_LoadingDataBlockIndex = m_LoopStartBlockIndex;
            {
                const auto begin = nn::os::GetSystemTick();
                m_pFileStream->Seek( m_LoopStartFilePos, fnd::Stream::SeekOrigin::SeekOrigin_Begin );
                const auto end = nn::os::GetSystemTick();
                fsAccessTick += end - begin;
            }

            // ループスタートブロックより１ブロック前の場所を空デコード
            if ( m_LoopStartBlockIndex > LoopDecodeStartOffset - 1 )
            {
                for ( auto i = 0; i < LoopDecodeStartOffset; ++i )
                {
                    if ( !DecodeStreamData( bufferAddress, IStreamDataDecoder::DecodeType_Idling ) )
                    {
                        return false;
                    }
                }
            }
            m_RegionManager.SetPosition( m_DataInfo->loopStart );
            m_LoopJumpFlag = true;
        }
    }

    // 開始オフセットが指定されている場合
    if ( startOffsetSamples > 0 )
    {
        if ( !ApplyStartOffset(startOffsetSamples, &loopCount) )
        {
            // 範囲外のオフセットが指定された
            loadDataParam.samples = 0;
            return true;
        }

        UpdateLoadingDataBlockIndexForOpus( bufferAddress );
        startOffsetSamplesInFrame = m_RegionManager.GetCurrentRegion().current - m_LoadingDataBlockIndex * m_DataInfo->blockSampleCount;
    }
    else if ( m_LoopJumpFlag )
    {
        startOffsetSamplesInFrame = m_LoopStartBlockSampleOffset;
    }

    // ループスタートブロックより１ブロック前の場所を記憶
    if ( IsLoopStartFilePos(m_LoadingDataBlockIndex) )
    {
        m_LoopStartFilePos = m_pFileStream->GetCurrentPosition();
    }

    position_t sampleBegin = m_RegionManager.GetCurrentRegion().current;

    size_t blockSamples = 0;
    IStreamDataDecoder::DecodeType decodeType;
    if ( m_LoadingDataBlockIndex != m_LastBlockIndex )
    {
        // 通常ブロック
        blockSamples = m_DataInfo->blockSampleCount;
        decodeType = IStreamDataDecoder::DecodeType_Normal;
    }
    else
    {
        // 最終ブロック
        blockSamples = m_DataInfo->lastBlockSampleCount;
        decodeType = IStreamDataDecoder::DecodeType_Loop;
    }

    if ( m_pFileStream->GetCurrentPosition() >= static_cast<position_t>(m_pFileStream->GetSize()) )
    {
        blockSamples = 0;
    }

    // データをデコードしつつロード
    if ( blockSamples > 0 )
    {
        if ( !DecodeStreamData( bufferAddress, decodeType ) )
        {
            return false;
        }

        for (auto i = 0; i < m_ChannelCount; ++i)
        {
            nn::atk::detail::driver::HardwareManager::FlushDataCache(bufferAddress[i], blockSamples * sizeof(int16_t));
        }

        // ループ波形でないときは、m_LoopEnd に -1 が入るため、
        // LoadHeader で m_LastBlockIndex の計算ができないのでこの処理が必要
        // ループ波形の時はループエンド以降にデータがあるのでやるとまずい
        if ( !m_DataInfo->loopFlag )
        {
            if ( m_pFileStream->GetCurrentPosition() >= static_cast<position_t>(m_pFileStream->GetSize()) )
            {
                m_LastBlockIndex = m_LoadingDataBlockIndex;
            }
        }
    }

    m_RegionManager.AddPosition( blockSamples );
    m_LoopJumpFlag = false;

    // ロードデータブロックインデックスの更新
    m_LoadingDataBlockIndex++;
    if ( m_LoadingDataBlockIndex > m_LastBlockIndex )
    {
        if ( m_DataInfo->loopFlag )
        {
            ++loopCount;
        }
        else
        {
            m_LoadFinishFlag = true;
        }
    }

    loadDataParam.adpcmContextEnable = false;
    loadDataParam.blockIndex = bufferBlockIndex;
    loadDataParam.samples = blockSamples;
    loadDataParam.sampleBegin = sampleBegin;
    loadDataParam.sampleOffset = startOffsetSamplesInFrame;
    loadDataParam.sampleBytes = blockSamples * sizeof(int16_t);
    loadDataParam.loopCount = loopCount;
    loadDataParam.lastBlockFlag = m_LoadFinishFlag;

    // プロファイル
    if( isProfileEnabled )
    {
        const auto endTick = nn::os::GetSystemTick();
        auto decodeProfile = m_pStreamDataDecoder->GetDecodeProfile();
        decodeProfile.fsAccessTick += fsAccessTick;  //  この関数内で行われた fs の時間を加える

        TaskProfile profile;
        IStreamDataDecoder::CacheProfile cacheProfile = IStreamDataDecoder::CacheProfile();
        profile.type = TaskProfile::TaskProfileType_LoadOpusStreamBlock;
        if ( IsStreamCacheEnabled() )
        {
            cacheProfile.cacheStartPosition   = detail_GetCachePosition();
            cacheProfile.cachedLength         = detail_GetCachedLength();
            cacheProfile.cacheCurrentPosition = detail_GetCurrentPosition();
            cacheProfile.player               = m_PlayerHandle;
        }
        profile.loadOpusStreamBlock.SetData( beginTick, endTick, decodeProfile, cacheProfile );
        logger.Record( profile );
    }

    return true;
} // NOLINT(impl/function_size)

void StreamSoundLoader::SetStreamSoundInfoForOpus(const IStreamDataDecoder::DataInfo& info) NN_NOEXCEPT
{
    m_DataInfo->sampleFormat = SampleFormat_PcmS16;
    m_DataInfo->sampleRate = info.sampleRate;
    if(m_IsLoopFlagEnabled)
    {
        // 現状、常にこちら側を通る実装となっている
        m_DataInfo->loopFlag = m_LoopFlag;
    }
    else
    {
        m_DataInfo->loopFlag = false;
    }
    m_DataInfo->loopStart = m_LoopStart;
    m_DataInfo->sampleCount = m_LoopEnd;
    m_DataInfo->originalLoopStart = m_LoopStart;
    m_DataInfo->originalLoopEnd = m_LoopEnd;
    m_DataInfo->blockSampleCount = info.blockSampleCount;
    m_DataInfo->blockSize = info.blockSize;
    m_DataInfo->lastBlockSize = m_DataInfo->blockSize;
}

bool StreamSoundLoader::ApplyStartOffset(position_t startOffsetSamples, int* loopCount) NN_NOEXCEPT
{
    position_t startOffsetSamplesInRegion = startOffsetSamples; // リージョン先頭からのサンプル位置

    while ( !m_RegionManager.GetCurrentRegion().IsIn(startOffsetSamplesInRegion) )
    {
        startOffsetSamplesInRegion -= m_RegionManager.GetCurrentRegion().Rest();

        if ( !MoveNextRegion(loopCount) )
        {
            return false;
        }
    }

    m_RegionManager.AddPosition( startOffsetSamplesInRegion );

    return true;
}

bool StreamSoundLoader::MoveNextRegion(int* loopCount) NN_NOEXCEPT
{
    if ( m_RegionManager.TryMoveNextRegion(&m_FileLoader, m_DataInfo) )
    {
        // 次のリージョン
        ++*loopCount;
    }
    else
    {
        // 終端
        m_LoadFinishFlag = true;

        return false;
    }

    return true;
}

bool StreamSoundLoader::ReadTrackInfoFromStreamSoundFile(StreamSoundFileReader& reader) NN_NOEXCEPT
{
    uint32_t trackCount = reader.GetTrackCount();
    if ( trackCount > StreamTrackCount )
    {
        trackCount = StreamTrackCount;
    }

    // m_DataInfo->trackCount = trackCount;

    // トラックの情報を読み取る
    for ( unsigned int i = 0; i < trackCount; i++ )
    {
        StreamSoundFileReader::TrackInfo trackInfo;
        if ( ! reader.ReadStreamTrackInfo( &trackInfo, i ) )
        {
            return false;
        }

        m_DataInfo->trackInfo[i].volume = trackInfo.volume;
        m_DataInfo->trackInfo[i].pan = trackInfo.pan;
        m_DataInfo->trackInfo[i].channelCount = trackInfo.channelCount;
        for( int ch = 0; ch < trackInfo.channelCount ; ch++ )
        {
            m_DataInfo->trackInfo[i].channelIndex[ch] = trackInfo.globalChannelIndex[ch];
        }

        // bXstm 0.2.0.0 には、span, flags が含まれないので読み込まない
        m_DataInfo->trackInfo[i].span = 0;
        m_DataInfo->trackInfo[i].flags = 0;

        m_DataInfo->trackInfo[i].mainSend = 127;
        for( int j = 0; j < AuxBus_Count; j++ )
        {
            m_DataInfo->trackInfo[i].fxSend[j] = 0;
        }
        m_DataInfo->trackInfo[i].lpfFreq     = DefaultLpfFreq;
        m_DataInfo->trackInfo[i].biquadType  = DefaultBiquadType;
        m_DataInfo->trackInfo[i].biquadValue = DefaultBiquadValue;
    }

    return true;
}

bool StreamSoundLoader::IsLoopStartFilePos(uint32_t loadingDataBlockIndex) NN_NOEXCEPT
{
    return m_LoopStartBlockIndex > LoopDecodeStartOffset - 1 && loadingDataBlockIndex == m_LoopStartBlockIndex - LoopDecodeStartOffset;
}

void StreamSoundLoader::UpdateLoadingDataBlockIndex() NN_NOEXCEPT
{
    m_LoadingDataBlockIndex = static_cast<uint32_t>(m_RegionManager.GetCurrentRegion().current / m_DataInfo->blockSampleCount);
    position_t startFilePos = m_DataStartFilePos + m_DataInfo->blockSize * m_ChannelCount * m_LoadingDataBlockIndex;
    m_pFileStream->Seek( startFilePos, fnd::Stream::SeekOrigin::SeekOrigin_Begin );
}

void StreamSoundLoader::UpdateLoadingDataBlockIndexForOpus(void* bufferAddress[]) NN_NOEXCEPT
{
    m_LoadingDataBlockIndex = static_cast<uint32_t>(m_RegionManager.GetCurrentRegion().current / m_DataInfo->blockSampleCount);

    // Opus の全体ヘッダの読み込みまでが終わった状態でここに来るという想定です。
    // 開始オフセット以外の理由でこの関数が呼び出される場合には、
    // ここで Opus の全体ヘッダの直後までシークする処理を追加する必要があります。

    for( unsigned int i = 0; i < m_LoadingDataBlockIndex; i++ )
    {
        if ( IsLoopStartFilePos(i) )
        {
            // ループスタートブロックより１ブロック前の場所を記憶
            m_LoopStartFilePos = m_pFileStream->GetCurrentPosition();
        }

        if ( i == m_LoadingDataBlockIndex - 1)
        {
            // 最後のブロックを空デコード
            if ( i > 0 )
            {
                DecodeStreamData(bufferAddress, IStreamDataDecoder::DecodeType_Idling);
            }
        }
        else
        {
            // 1ブロック分スキップ
            NN_SDK_ASSERT_NOT_NULL(m_pStreamDataDecoder);
            m_pStreamDataDecoder->Skip( m_pFileStream );
        }
    }
}

int StreamSoundLoader::GetLoadChannelCount(int loadStartChannel) NN_NOEXCEPT
{
    int loadChannelCount = LoadBufferChannelCount;
    if ( loadStartChannel + loadChannelCount > m_ChannelCount )
    {
        loadChannelCount = m_ChannelCount - loadStartChannel;
    }

    return loadChannelCount;
}

bool StreamSoundLoader::LoadStreamBuffer(uint8_t* buffer, const BlockInfo& blockInfo, uint32_t loadChannelCount) NN_NOEXCEPT
{
    size_t loadSize = blockInfo.size * loadChannelCount;

    // ストリームキャッシュが無効な場合は、g_LoadBuffer に収まるかどうかを確認する。
    // ストリームキャッシュが有効な場合は、g_LoadBuffer を経由しないので確認不要。
    NN_SDK_REQUIRES( IsStreamCacheEnabled() || loadSize <= LoadBufferSize );

    return LoadStreamBuffer(buffer, loadSize);
}

bool StreamSoundLoader::LoadStreamBuffer(uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    // データのロード
    size_t resultSize = m_pFileStream->Read( buffer, size);
    if ( resultSize != size )
    {
        return false;
    }

#ifdef DEBUG_STRM
    std::fwrite( buffer, size, 1, s_pFile );
#endif

    return true;
}

bool StreamSoundLoader::SkipStreamBuffer(size_t skipSize) NN_NOEXCEPT
{
    return m_pFileStream->Seek(skipSize, nn::atk::detail::fnd::Stream::SeekOrigin_Current).IsSucceeded();
}

void StreamSoundLoader::CalculateBlockInfo(BlockInfo& blockInfo) NN_NOEXCEPT
{
    if ( m_LoadingDataBlockIndex != m_LastBlockIndex )
    {
        // 通常ブロック
        blockInfo.size = m_DataInfo->blockSize;
        blockInfo.samples = m_DataInfo->blockSampleCount;
    }
    else
    {
        // 最終ブロック
        blockInfo.size = m_DataInfo->lastBlockSize;
        blockInfo.samples = m_DataInfo->lastBlockSampleCount;
    }

    blockInfo.startOffsetSamples = (m_RegionManager.GetCurrentRegion().current % m_DataInfo->blockSampleCount);
    blockInfo.startOffsetSamplesAlign = blockInfo.startOffsetSamples;
    if ( m_SampleFormat == SampleFormat_DspAdpcm )
    {
        blockInfo.startOffsetSamplesAlign = static_cast<int>(blockInfo.startOffsetSamples / 14) * 14;
    }

    blockInfo.startOffsetByte = Util::GetByteBySample( blockInfo.startOffsetSamplesAlign, m_SampleFormat );
    blockInfo.copyByte = blockInfo.size - blockInfo.startOffsetByte;
    blockInfo.samples -= blockInfo.startOffsetSamples;

    if ( !m_RegionManager.GetCurrentRegion().IsInWithBorder(blockInfo.samples) )
    {
        blockInfo.samples = m_RegionManager.GetCurrentRegion().Rest();
        if (blockInfo.samples < detail::DataBlockSizeMarginSamples)
        {
            blockInfo.copyByte = StreamSoundLoader::DataBlockSizeMargin;
        }
    }
}

bool StreamSoundLoader::LoadOneBlockDataViaCache(void* bufferAddress[], const BlockInfo& blockInfo, position_t destAddressOffset, bool firstBlock, bool updateAdpcmContext) NN_NOEXCEPT
{
    // ストリームキャッシュを利用している前提で、再生用バッファに直接ロードする。

    NN_SDK_REQUIRES(IsStreamCacheEnabled());

    for (int ch = 0; ch < m_ChannelCount; ch++)
    {
        // 終了中であれば読み込みを中断
        NN_SDK_ASSERT_NOT_NULL(m_PlayerHandle);
        if (m_PlayerHandle->IsFinalizing())
        {
            return false;
        }

        uint8_t* dest = util::BytePtr(bufferAddress[ch], destAddressOffset).Get<uint8_t>();

        // 開始オフセットがある場合は、その分だけ読み捨てる
        if (blockInfo.startOffsetByte > 0)
        {
            SkipStreamBuffer(blockInfo.startOffsetByte);
        }

        // 再生用バッファにロードしてフラッシュ
        if (!LoadStreamBuffer(dest, blockInfo.copyByte))
        {
            return false;
        }

        nn::atk::detail::driver::HardwareManager::FlushDataCache(dest, blockInfo.copyByte);

        // 必要なときに、ADPCMコンテキストを更新
        if (firstBlock && updateAdpcmContext)
        {
            UpdateAdpcmInfoForStartOffset(dest, ch, blockInfo);
        }
    }

    return true;
}

bool StreamSoundLoader::LoadOneBlockData(void* bufferAddress[], const BlockInfo& blockInfo, position_t destAddressOffset, bool firstBlock, bool updateAdpcmContext) NN_NOEXCEPT
{
    // ファイルアクセス回数を削減するために、g_LoadBuffer を経由して、再生用バッファにコピーする。
    // ストリームキャッシュを利用しないケースを想定。

    int ch = 0;
    while ( ch < m_ChannelCount )
    {
        // 終了中であれば読み込みを中断
        NN_SDK_ASSERT_NOT_NULL( m_PlayerHandle );
        if ( m_PlayerHandle->IsFinalizing() )
        {
            return false;
        }

        // 同時に読み込むチャンネル数を確定し、ロードサイズを計算
        int loadChannelCount = GetLoadChannelCount(ch);
        if ( !LoadStreamBuffer(g_LoadBuffer, blockInfo, loadChannelCount) )
        {
            return false;
        }

        // 同時に読みこんだチャンネル数分のバッファを、再生用バッファにコピーしてフラッシュ
        for ( auto i = 0; i < loadChannelCount; i++ )
        {
            const void* blockBegin = util::BytePtr( g_LoadBuffer, blockInfo.size * i ).Get();
            const void* source = util::ConstBytePtr( blockBegin, blockInfo.startOffsetByte ).Get();
            void* dest = util::BytePtr( bufferAddress[ ch ], destAddressOffset ).Get();
            std::memcpy( dest, source, blockInfo.copyByte );

            nn::atk::detail::driver::HardwareManager::FlushDataCache( dest, blockInfo.copyByte );

            // 必要なときに、ADPCMコンテキストを更新
            if ( firstBlock && updateAdpcmContext )
            {
                UpdateAdpcmInfoForStartOffset(blockBegin, ch, blockInfo);
            }

            ++ ch;
        }

    } // while(チャンネル)

    return true;
}

bool StreamSoundLoader::LoadAdpcmContextForStartOffset() NN_NOEXCEPT
{
    position_t fpos = m_pFileStream->GetCurrentPosition();

    uint16_t yn1[ StreamChannelCount ];
    uint16_t yn2[ StreamChannelCount ];
    if ( ! m_FileLoader.ReadSeekBlockData(
             yn1,
             yn2,
             m_LoadingDataBlockIndex,
             m_ChannelCount
         ) )
    {
        return false;
    }
    m_pFileStream->Seek(fpos, fnd::Stream::SeekOrigin::SeekOrigin_Begin);

    for ( auto ch = 0; ch < m_ChannelCount; ch++ )
    {
        m_RegionManager.GetAdpcmContextForStartOffset(ch).audioAdpcmContext.history[0] = yn1[ch];
        m_RegionManager.GetAdpcmContextForStartOffset(ch).audioAdpcmContext.history[1] = yn2[ch];
    }

    return true;
}

void StreamSoundLoader::UpdateAdpcmInfoForStartOffset(const void* blockBegin, int channelIndex, const BlockInfo& blockInfo) NN_NOEXCEPT
{
    AdpcmContext& adpcmContext = m_RegionManager.GetAdpcmContextForStartOffset( channelIndex );
    adpcmContext.audioAdpcmContext.predScale = *reinterpret_cast<const uint8_t*>(blockBegin);

    // ADPCM フレーム (8バイト=14サンプル) の途中からはデコードできないので、
    // オフセットをフレーム境界に合わせる
    uint32_t offset = static_cast<uint32_t>(blockInfo.startOffsetSamples / 14) * 14;
    MultiVoice::CalcOffsetAdpcmParam(
        &adpcmContext,
        m_AdpcmInfo[channelIndex].param,
        offset,
        blockBegin
    );

    m_RegionManager.SetStartOffsetFrame( m_RegionManager.GetCurrentRegion().current );
}

bool StreamSoundLoader::SetAdpcmInfo(
    StreamSoundFileReader& reader,
    int channelCount,
    AdpcmParam* adpcmParam[]) NN_NOEXCEPT
{

    NN_SDK_ASSERT( channelCount <= StreamChannelCount );

    for ( auto ch = 0; ch < channelCount; ch++ )
    {
        DspAdpcmParam dspAdpcmParam;
        DspAdpcmLoopParam dspAdpcmLoopParam;
        if ( ! reader.ReadDspAdpcmChannelInfo(
                 &dspAdpcmParam,
                 &dspAdpcmLoopParam,
                 ch ) )
        {
            return false;
        }

        AdpcmInfo& adpcmInfo = m_AdpcmInfo[ch];
        for ( auto i = 0; i < 8; ++i )
        {
            for ( auto j = 0; j < 2; ++j )
            {
                adpcmInfo.param.coefficients[i * 2 + j] = dspAdpcmParam.coef[i][j];
            }
        }
        adpcmInfo.beginContext.audioAdpcmContext.predScale = dspAdpcmParam.predScale;
        adpcmInfo.beginContext.audioAdpcmContext.history[0] = dspAdpcmParam.yn1;
        adpcmInfo.beginContext.audioAdpcmContext.history[1] = dspAdpcmParam.yn2;

        adpcmInfo.loopContext.audioAdpcmContext.predScale = dspAdpcmLoopParam.loopPredScale;
        adpcmInfo.loopContext.audioAdpcmContext.history[0] = dspAdpcmLoopParam.loopYn1;
        adpcmInfo.loopContext.audioAdpcmContext.history[1] = dspAdpcmLoopParam.loopYn2;

        adpcmParam[ch] = &adpcmInfo.param;

    #ifdef ENABLE_PRINT_STREAM_INFO
        NN_DETAIL_ATK_INFO("*** STRM ADPCM INFO ch(%d) ***\n", ch);
        NN_DETAIL_ATK_INFO("    [coef] %04x %04x %04x %04x %04x %04x %04x %04x\n",
                adpcmInfo.param.coef[0][0], adpcmInfo.param.coef[0][1],
                adpcmInfo.param.coef[1][0], adpcmInfo.param.coef[1][1],
                adpcmInfo.param.coef[2][0], adpcmInfo.param.coef[2][1],
                adpcmInfo.param.coef[3][0], adpcmInfo.param.coef[3][1] );
        NN_DETAIL_ATK_INFO("           %04x %04x %04x %04x %04x %04x %04x %04x\n",
                adpcmInfo.param.coef[4][0], adpcmInfo.param.coef[4][1],
                adpcmInfo.param.coef[5][0], adpcmInfo.param.coef[5][1],
                adpcmInfo.param.coef[6][0], adpcmInfo.param.coef[6][1],
                adpcmInfo.param.coef[7][0], adpcmInfo.param.coef[7][1] );
        NN_DETAIL_ATK_INFO("    [begin] predScale(%04x) yn1(%6d) yn2(%6d)\n",
                adpcmInfo.beginContext.pred_scale,
                adpcmInfo.beginContext.yn1, adpcmInfo.beginContext.yn2 );
        NN_DETAIL_ATK_INFO("    [loop ] predScale(%04x) yn1(%6d) yn2(%6d)\n",
                adpcmInfo.loopContext.pred_scale,
                adpcmInfo.loopContext.yn1, adpcmInfo.loopContext.yn2 );
    #endif
    }

    return true;
}

bool StreamSoundLoader::DecodeStreamData(void* pOutBufferAddresses[], IStreamDataDecoder::DecodeType decodeType) NN_NOEXCEPT
{
    if (m_pStreamDataDecoderManager == nullptr)
    {
        NN_ATK_WARNING("DecodeStreamData failed. Decoder is not prepared for.");
        return false;
    }

    int16_t* pDecodedBufferAddresses[StreamChannelCount];
    for( auto i = 0; i < m_ChannelCount; i++ )
    {
        pDecodedBufferAddresses[i] = reinterpret_cast<int16_t*>(pOutBufferAddresses[i]);
    }

    NN_SDK_ASSERT_NOT_NULL(m_pStreamDataDecoder);
    return m_pStreamDataDecoder->Decode( pDecodedBufferAddresses, m_pFileStream, m_ChannelCount, decodeType );;
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

