﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_StreamSound.h>
#include <nn/atk/atk_StreamSoundHandle.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {
namespace detail {

/* ========================================================================
        member function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         StreamSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
StreamSound::StreamSound(
    StreamSoundInstanceManager& manager
) NN_NOEXCEPT
: m_Manager( manager )
, m_InitializeFlag(false)
, m_pCacheBuffer(NULL)
, m_CacheSize(0)
{
}


bool StreamSound::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    bool result = BasicSound::Initialize( pOutputReceiver );
    if( !result )
    {
        return false;
    }

    m_pTempSpecialHandle = NULL;

    for ( int i = 0; i < StreamTrackCount; i++ )
    {
        m_TrackVolume[ i ].InitValue( 0.0f );
        m_TrackVolume[ i ].SetTarget( 1.0f, 1 );
    }

    for ( int i = 0; i < WaveChannelMax; i++ )
    {
        m_AvailableTrackBitFlag[ i ] = 0x0;
    }

    m_InitializeFlag = true;
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Finalize

  Description:  サウンドの終了

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSound::Finalize() NN_NOEXCEPT
{
    if ( ! m_InitializeFlag ) return;
    m_InitializeFlag = false;

    BasicSound::Finalize();
    m_Manager.Free( this );
}

void StreamSound::Setup(const driver::StreamSoundPlayer::SetupArg& arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arg.pBufferPool );

    m_AllocTrackFlag = arg.allocTrackFlag;
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandStreamSoundSetup* command =
            cmdmgr.AllocCommand<DriverCommandStreamSoundSetup>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_StrmSetup;
        command->player = &m_PlayerInstance;
        command->arg = arg;  // 構造体コピー
        cmdmgr.PushCommand(command);
    }

    for (uint32_t trackNo = 0; trackNo < detail::StreamTrackCount; trackNo++)
    {
        const uint8_t trackChannelCount = arg.trackInfos.track[trackNo].channelCount;
        for (uint8_t channelNo = 0; channelNo < trackChannelCount; channelNo++ )
        {
            m_AvailableTrackBitFlag[channelNo] += 0x1 << trackNo;
        }
    }
}

void StreamSound::Prepare(const driver::StreamSoundPlayer::PrepareBaseArg& arg) NN_NOEXCEPT
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandStreamSoundPrepare* command =
            cmdmgr.AllocCommand<DriverCommandStreamSoundPrepare>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_StrmPrepare;
        command->player = &m_PlayerInstance;
        command->arg.baseArg = arg;     // 構造体コピー
        command->arg.cacheBuffer = GetCacheBuffer();
        command->arg.cacheSize = GetCacheSize();
        cmdmgr.PushCommand(command);
    }
}

void StreamSound::PreparePrefetch(const void* strmPrefetchFile, const driver::StreamSoundPlayer::PrepareBaseArg& arg) NN_NOEXCEPT
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandStreamSoundPreparePrefetch* command =
            cmdmgr.AllocCommand<DriverCommandStreamSoundPreparePrefetch>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_StrmPreparePrefetch;
        command->player = &m_PlayerInstance;
        command->arg.strmPrefetchFile = strmPrefetchFile;
        command->arg.baseArg = arg;
        cmdmgr.PushCommand(command);
    }
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateMoveValue

  Description:  サウンドのアップデート

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSound::UpdateMoveValue() NN_NOEXCEPT
{
    BasicSound::UpdateMoveValue();

    uint16_t bitFlag = m_AllocTrackFlag;
    for ( int trackNo = 0; trackNo < StreamTrackCount; trackNo++, bitFlag>>=1 )
    {
        if ( bitFlag & 0x01 )
        {
            m_TrackVolume[ trackNo ].Update();
        }
    }
}

void StreamSound::OnUpdateParam() NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();

    uint16_t bitFlag = m_AllocTrackFlag;
    for ( int trackNo = 0; trackNo < StreamTrackCount; trackNo++, bitFlag>>=1 )
    {
        if ( bitFlag & 0x01 )
        {
            DriverCommandStreamSoundTrackParam* command =
                cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            if (command == nullptr)
            {
                return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
            }
#else
            NN_SDK_ASSERT_NOT_NULL(command);
#endif
            command->id = DriverCommandId_StrmTrackVolume;
            command->player = &m_PlayerInstance;
            command->trackBitFlag = static_cast<uint32_t>( 1 << trackNo );
            command->value = m_TrackVolume[ trackNo ].GetValue();
            cmdmgr.PushCommand(command);
        }
    }
}

void StreamSound::SetTrackVolume( uint32_t trackBitFlag, float volume, int frames ) NN_NOEXCEPT
{
    if ( volume < 0.0f ) volume = 0.0f;

    uint16_t bitFlag = ( m_AllocTrackFlag & trackBitFlag );
    for( int trackNo = 0;
         trackNo < StreamTrackCount && trackBitFlag != 0;
         trackNo++, bitFlag >>=1
    )
    {
        if ( bitFlag & 0x01 )
        {
            m_TrackVolume[ trackNo ].SetTarget( volume, frames );
        }
    }
}

void StreamSound::SetTrackInitialVolume( uint32_t trackBitFlag, uint32_t volume ) NN_NOEXCEPT
{
    uint16_t bitFlag = ( m_AllocTrackFlag & trackBitFlag );
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackInitialVolume* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackInitialVolume>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackInitialVolume;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = bitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackOutputLine( uint32_t trackBitFlag, uint32_t lineFlag ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackOutputline;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->uint32Value = lineFlag;
    cmdmgr.PushCommand(command);
}

void StreamSound::ResetTrackOutputLine( uint32_t trackBitFlag ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackOutputlineReset;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    cmdmgr.PushCommand(command);
}



void StreamSound::SetTrackMainOutVolume( uint32_t trackBitFlag, float volume ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackTvVolume;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackChannelMixParameter( uint32_t trackBitFlag, uint32_t srcChNo, const MixParameter& mixParam ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( srcChNo < WaveChannelMax );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackMixParameter* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackMixParameter>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackTvMixParameter;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->srcChNo = srcChNo;
    for ( int channel = 0; channel < ChannelIndex_Count; channel++ )
    {
        command->param.ch[channel] = mixParam.ch[channel];
    }
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackPan( uint32_t trackBitFlag, float pan ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackTvPan;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pan;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackSurroundPan( uint32_t trackBitFlag, float span ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackTvSpan;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = span;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackMainSend( uint32_t trackBitFlag, float send ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackTvMainSend;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    cmdmgr.PushCommand(command);
}

void StreamSound::SetTrackFxSend( uint32_t trackBitFlag, AuxBus bus, float send ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, AuxBus_Count );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandStreamSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandStreamSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_StrmTrackTvFxSend;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->uint32Value = bus;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         OnUpdatePlayerPriority

  Description:  プレイヤープライオリティを変更

  Arguments:

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void StreamSound::OnUpdatePlayerPriority() NN_NOEXCEPT
{
    m_Manager.UpdatePriority( this, CalcCurrentPlayerPriority() );
}

bool StreamSound::IsAttachedTempSpecialHandle() NN_NOEXCEPT
{
    return m_pTempSpecialHandle != NULL;
}

void StreamSound::DetachTempSpecialHandle() NN_NOEXCEPT
{
    m_pTempSpecialHandle->DetachSound();
}

bool StreamSound::ReadStreamSoundDataInfo( StreamSoundDataInfo* info ) const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() )
    {
        NN_ATK_WARNING("ReadStreamSoundDataInfo failed. Please use this API after playback preparing finished.");
        return false;
    }
    return m_PlayerInstance.ReadStreamSoundDataInfo( info );
}

long StreamSound::GetPlayLoopCount() const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return 0;
    return m_PlayerInstance.GetPlayLoopCount();
}

position_t StreamSound::GetPlaySamplePosition(bool isOriginalSamplePosition) const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return 0;
    return m_PlayerInstance.GetPlaySamplePosition(isOriginalSamplePosition);
}

float StreamSound::GetFilledBufferPercentage() const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return 0.0f;
    return m_PlayerInstance.GetFilledBufferPercentage();
}

int StreamSound::GetBufferBlockCount(nn::atk::WaveBuffer::Status status) const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return 0;
    return m_PlayerInstance.GetBufferBlockCount(status);
}

int StreamSound::GetTotalBufferBlockCount() const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return 0;
    return m_PlayerInstance.GetTotalBufferBlockCount();
}

bool StreamSound::IsPrepared() const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return false;
    return m_PlayerInstance.IsPrepared();
}

bool StreamSound::IsSuspendByLoadingDelay() const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return false;
    return m_PlayerInstance.IsSuspendByLoadingDelay();
}

bool StreamSound::IsLoadingDelayState() const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return false;
    return m_PlayerInstance.IsLoadingDelayState();
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

