﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SequenceTrack.h>

#include <nn/atk/atk_Bank.h>
#include <nn/atk/atk_SequenceSound.h>
#include <nn/atk/atk_SequenceSoundPlayer.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>

namespace nn {
namespace atk {
namespace detail {
namespace driver {

NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::CallStackDepth );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::DefaultPriority );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::DefaultBendRange );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::DefaultPortaKey );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::InvalidEnvelope );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::MaxEnvelopeValue );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::ParserParamSize );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::TrackVariableCount );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::PauseReleaseValue );
NN_DEFINE_STATIC_CONSTANT( const int SequenceTrack::MuteReleaseValue );

/* ========================================================================
        member function
   ======================================================================== */

void SequenceTrack::SetPlayerTrackNo( int playerTrackNo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( playerTrackNo >= 0 && playerTrackNo <= SequenceSoundPlayer::TrackCountPerPlayer );
    m_PlayerTrackNo  = static_cast<uint8_t>( playerTrackNo );
}

SequenceTrack::SequenceTrack() NN_NOEXCEPT
: m_OpenFlag( false )
, m_pSequenceSoundPlayer( NULL )
, m_pChannelList( NULL )

{
    InitParam();
}

SequenceTrack::~SequenceTrack() NN_NOEXCEPT
{
    Close();
}

/*--------------------------------------------------------------------------------*
  Name:         InitParam

  Description:  トラック構造体を初期化します

  Arguments:    track - トラックポインタ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::InitParam() NN_NOEXCEPT
{
    m_ExtVolume                  = 1.0f;
    m_ExtPitch                   = 1.0f;
    m_PanRange                   = 1.0f;

    m_TvParam.Initialize();

    // Parser Param
    m_ParserTrackParam.baseAddr       = NULL;
    m_ParserTrackParam.currentAddr    = NULL;

    m_ParserTrackParam.cmpFlag        = true;
    m_ParserTrackParam.noteWaitFlag   = true;
    m_ParserTrackParam.tieFlag        = false;
    m_ParserTrackParam.monophonicFlag = false;

    m_ParserTrackParam.callStackDepth = 0;

    m_ParserTrackParam.wait           = 0;

    m_ParserTrackParam.muteFlag       = false;
    m_ParserTrackParam.silenceFlag    = false;
    m_ParserTrackParam.noteFinishWait = false;
    m_ParserTrackParam.portaFlag      = false;
    m_ParserTrackParam.damperFlag     = false;

    m_ParserTrackParam.bankIndex      = 0;
    m_ParserTrackParam.prgNo          = 0;

    for (int i = 0; i < Channel::ModCount; i++)
    {
        m_ParserTrackParam.lfoParam[i].Initialize();
        m_ParserTrackParam.lfoTarget[i] = Channel::LfoTarget_Pitch;
    }
    m_ParserTrackParam.sweepPitch     = 0.0f;

    m_ParserTrackParam.volume.InitValue( 127 );
    m_ParserTrackParam.volume2.InitValue( 127 );
    m_ParserTrackParam.pan.InitValue( 0 );
    m_ParserTrackParam.surroundPan.InitValue( 0 );
    m_ParserTrackParam.velocityRange  = 127;
    m_ParserTrackParam.pitchBend.InitValue( 0 );
    m_ParserTrackParam.bendRange      = DefaultBendRange;
    m_ParserTrackParam.initPan        = 0;
    m_ParserTrackParam.transpose      = 0;
    m_ParserTrackParam.priority       = DefaultPriority;
    m_ParserTrackParam.portaKey       = DefaultPortaKey;
    m_ParserTrackParam.portaTime      = 0;

    m_ParserTrackParam.attack         = InvalidEnvelope;
    m_ParserTrackParam.decay          = InvalidEnvelope;
    m_ParserTrackParam.sustain        = InvalidEnvelope;
    m_ParserTrackParam.release        = InvalidEnvelope;
    m_ParserTrackParam.envHold        = InvalidEnvelope;

    m_ParserTrackParam.mainSend       = 127;
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        m_ParserTrackParam.fxSend[i] = 0;
    }

    m_ParserTrackParam.lpfFreq        = 0.0f;
    m_ParserTrackParam.biquadType     = 0;
    m_ParserTrackParam.biquadValue    = 0;
    m_ParserTrackParam.outputLine     = -1;

    for( int varNo = 0; varNo < TrackVariableCount ; varNo++ )
    {
        m_TrackVariable[ varNo ] = SequenceSoundPlayer::VariableDefaultValue;
    }

    m_ForceMute = false;
}

/*--------------------------------------------------------------------------------*
  Name:         SetSeqData

  Description:  シーケンスデータをセットします

  Arguments:    seqBase - シーケンスデータベースアドレス
                seqOffset - シーケンスデータオフセット

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::SetSeqData( const void* seqBase, int seqOffset ) NN_NOEXCEPT
{
    m_ParserTrackParam.baseAddr    = static_cast<const uint8_t*>( seqBase );
    m_ParserTrackParam.currentAddr = m_ParserTrackParam.baseAddr + seqOffset;
}

void SequenceTrack::Open() NN_NOEXCEPT
{
    m_ParserTrackParam.noteFinishWait = false;
    m_ParserTrackParam.callStackDepth = 0;
    m_ParserTrackParam.wait           = 0;

    m_OpenFlag = true;
}

/*--------------------------------------------------------------------------------*
  Name:         Close

  Description:  トラックを閉じます

  Arguments:    player - プレイヤーポインタ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::Close() NN_NOEXCEPT
{
    ReleaseAllChannel( -1 );
    FreeAllChannel();

    m_OpenFlag = false;
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateChannelLength

  Description:  チャンネル長を更新します

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::UpdateChannelLength() NN_NOEXCEPT
{
    if ( ! m_OpenFlag ) return;

    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        if ( channel->GetLength() > 0 )
        {
            channel->SetLength( channel->GetLength() - 1 );
        }

        UpdateChannelRelease( channel );

        if ( ! channel->IsAutoUpdateSweep() )
        {
            channel->UpdateSweep( 1 );
        }

        channel = channel->GetNextTrackChannel();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateChannelRelease

  Description:  チャンネルをリリース状態を更新する

  Arguments:    channel - チャンネル

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void SequenceTrack::UpdateChannelRelease( Channel* channel ) NN_NOEXCEPT
{
    if ( ( channel->GetLength() == 0 ) && ( ! channel->IsRelease() ) )
    {
        if ( ! m_ParserTrackParam.damperFlag )
        {
            channel->NoteOff();
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         ParseNextTick

  Description:  トラックのシーケンス処理を進めます

  Arguments:    doNoteOn - ノートオンするかどうか

  Returns:      シーケンス継続時には正を、終了時には負を、未Open時には0を返します
 *--------------------------------------------------------------------------------*/
int SequenceTrack::ParseNextTick( bool doNoteOn ) NN_NOEXCEPT
{
    if ( ! m_OpenFlag ) {
        return 0;
    }

    // MoveValueの更新
    m_ParserTrackParam.volume.Update();
    m_ParserTrackParam.volume2.Update();
    m_ParserTrackParam.pan.Update();
    m_ParserTrackParam.surroundPan.Update();
    m_ParserTrackParam.pitchBend.Update();

    // ウェイトチェック
    if ( m_ParserTrackParam.noteFinishWait ) {
        if ( m_pChannelList != NULL ) {
            return 1;
        }
        m_ParserTrackParam.noteFinishWait = false;
    }
    if ( m_ParserTrackParam.wait > 0 ) {
        m_ParserTrackParam.wait--;
        if ( m_ParserTrackParam.wait > 0 ) {
            return 1;
        }
    }

    // シーケンス処理
    if ( m_ParserTrackParam.currentAddr != NULL )
    {
        int counter = 0;
        const int CounterMax = 10000;
        while( m_ParserTrackParam.wait == 0 && ! m_ParserTrackParam.noteFinishWait )
        {
            counter++;
            if ( counter > CounterMax )
            {
                NN_ATK_WARNING("cannot process %d SEQ command!", CounterMax);
                return 1;
            }

            ParseResult result = Parse( doNoteOn );
            if ( result == ParseResult_Finish )
            {
                return -1;
            }
        }
    }

    return 1;
}

/*--------------------------------------------------------------------------------*
  Name:         StopAllChannel

  Description:  トラックのチャンネルを全て停止します

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::StopAllChannel() NN_NOEXCEPT
{
    Channel* channel = m_pChannelList;

    while( channel != NULL )
    {
        Channel* nextChannel = channel->GetNextTrackChannel();

        channel->Stop();
        channel->CallChannelCallback( Channel::ChannelCallbackStatus_Stopped );
        Channel::FreeChannel(channel);

        channel = nextChannel;
    }

    m_pChannelList = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         ReleaseAllChannel

  Description:  トラックのチャンネルを全てリリースします

  Arguments:    release - リリース値（負値の場合は、既に設定されているリリース値を使用する）

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::ReleaseAllChannel( int release ) NN_NOEXCEPT
{
    UpdateChannelParam();

    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        if ( channel->IsActive() )
        {
            if ( release >= 0 )
            {
                NN_SDK_ASSERT( release >= 0 && release <= 127 );
                channel->SetRelease( static_cast<uint8_t>( release ) );
            }
            channel->Release();
        }
        channel = channel->GetNextTrackChannel();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         PauseAllChannel

  Description:  トラックのチャンネルを全てポーズします

  Arguments:    track - トラックポインタ
                player - プレイヤーポインタ
                release - リリース値（負値の場合は、既に設定されているリリース値を使用する）

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::PauseAllChannel( bool flag ) NN_NOEXCEPT
{
    Channel* channel = m_pChannelList;

    while( channel != NULL )
    {
        if ( channel->IsActive() && ( channel->IsPause() != flag ) )
        {
            channel->Pause( flag );
        }
        channel = channel->GetNextTrackChannel();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         AddChannel

  Description:  トラックへチャンネルを追加します

  Arguments:    channel - チャンネルポインタ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::AddChannel( Channel* channel ) NN_NOEXCEPT
{
    channel->SetNextTrackChannel( m_pChannelList );
    m_pChannelList = channel;
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateChannelParam

  Description:  トラックが保持しているチャンネルのパラメータを更新します

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::UpdateChannelParam() NN_NOEXCEPT
{
    if ( ! m_OpenFlag ) return;

    if ( m_pChannelList == NULL ) {
        // 更新不要
        return;
    }

    // volume
    float volume
        = m_ParserTrackParam.volume.GetValue()
        * m_ParserTrackParam.volume2.GetValue()
        * m_pSequenceSoundPlayer->GetParserPlayerParam().volume.GetValue()
        / (127.0f * 127.0f * 127.0f);
    volume = volume * volume * m_ExtVolume * m_pSequenceSoundPlayer->GetVolume();

    float pitch = ( m_ParserTrackParam.pitchBend.GetValue() / 128.0f ) * m_ParserTrackParam.bendRange;

    float pitchRatio = m_pSequenceSoundPlayer->GetPitch() * m_ExtPitch;

    // lpf freq
    float lpfFreq
        = m_ParserTrackParam.lpfFreq
        + m_pSequenceSoundPlayer->GetLpfFreq();

    // biquad filter
    int biquadType = m_ParserTrackParam.biquadType;
    float biquadValue = m_ParserTrackParam.biquadValue;
    if ( m_pSequenceSoundPlayer->GetBiquadFilterType() != BiquadFilterType_Inherit )
    {
        biquadType = m_pSequenceSoundPlayer->GetBiquadFilterType();
        biquadValue = m_pSequenceSoundPlayer->GetBiquadFilterValue();
    }

    // outputLine
    uint32_t outputLine = m_pSequenceSoundPlayer->GetOutputLine();
    if ( m_ParserTrackParam.outputLine != -1 )
    {
        outputLine = m_ParserTrackParam.outputLine;
    }



    // ベースデータ
    float panBase =
        nn::atk::detail::fnd::Clamp( static_cast<float>( m_ParserTrackParam.pan.GetValue() ) / 63.0f,
                   -1.0f, 1.0f )
        * m_PanRange
        * m_pSequenceSoundPlayer->GetPanRange();
    float spanBase =
        nn::atk::detail::fnd::Clamp( static_cast<float>( m_ParserTrackParam.surroundPan.GetValue() ) / 63.0f,
                   0.0f, 2.0f );
    float mainSendBase =
        static_cast<float>( m_ParserTrackParam.mainSend ) / 127.0f - 1.0f; // -1.0 ～ 0.0 に
    float fxSendBase[AuxBus_Count];
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        fxSendBase[i] = static_cast<float>( m_ParserTrackParam.fxSend[i] ) / 127.0f;
    }

    // TV 出力向けパラメータ
    OutputParam tvParam = m_pSequenceSoundPlayer->GetTvParam();
    {
        tvParam.volume *= m_TvParam.volume;
        for ( int i = 0; i < WaveChannelMax; i++ )
        {
            for ( int j = 0; j < ChannelIndex_Count; j++ )
            {
                tvParam.mixParameter[i].ch[j] *= m_TvParam.mixParameter[i].ch[j];
            }
        }
        tvParam.pan += panBase;
        tvParam.pan += m_TvParam.pan;
        tvParam.span += spanBase;
        tvParam.span += m_TvParam.span;

        const int BusCount = DefaultBusCount;
        for ( int i = 0; i < BusCount; i++ )
        {
            tvParam.send[i] += m_TvParam.send[i];
        }

        tvParam.send[Util::GetSubMixBusFromMainBus()] += mainSendBase;
        for ( int i = 0; i < AuxBus_Count; i++ )
        {
            tvParam.send[Util::GetSubMixBus(i)] += fxSendBase[i];
        }
    }

    // set channel params of track
    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        channel->SetUserVolume( volume );
        channel->SetUserPitch( pitch );
        channel->SetUserPitchRatio( pitchRatio );
        channel->SetUserLpfFreq( lpfFreq );
        channel->SetBiquadFilter( biquadType, biquadValue );
        channel->SetOutputLine( outputLine );
        for (int i = 0; i < Channel::ModCount; i++)
        {
            channel->SetLfoParam( m_ParserTrackParam.lfoParam[i], i );
            channel->SetLfoTarget( static_cast<Channel::LfoTarget>( m_ParserTrackParam.lfoTarget[i] ), i );
        }

        channel->SetTvParam( tvParam );
        if( m_pSequenceSoundPlayer->GetTvAdditionalParamAddr() != nullptr )
        {
            channel->SetTvAdditionalParam( *m_pSequenceSoundPlayer->GetTvAdditionalParamAddr() );
        }

        channel = channel->GetNextTrackChannel();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         FreeAllChannel

  Description:  トラックのチャンネルを全て解放します

  Arguments:    track - トラックポインタ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::FreeAllChannel() NN_NOEXCEPT
{
    Channel* channel = m_pChannelList;

    while( channel != NULL ) {
        Channel::DetachChannel( channel );
        channel = channel->GetNextTrackChannel();
    }
    m_pChannelList = NULL;
}


/*--------------------------------------------------------------------------------*
  Name:         GetChannelCount

  Description:  トラックが保持しているチャンネル数を取得する

  Arguments:    なし

  Returns:      チャンネル数
 *--------------------------------------------------------------------------------*/
int SequenceTrack::GetChannelCount() const NN_NOEXCEPT
{
    Channel* channel = m_pChannelList;
    int count = 0;
    while( channel != NULL ) {
        ++count;
        channel = channel->GetNextTrackChannel();
    }

    return count;
}

/*--------------------------------------------------------------------------------*
  Name:         ChannelCallbackFunc

  Description:  チャンネルから呼びだされるコールバック関数

  Arguments:    dropChannel - チャンネルポインタ
                status - チャンネルコールバックステータス
                userData - トラックポインタを格納したコールバックユーザーデータ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceTrack::ChannelCallbackFunc(
    Channel* dropChannel,
    Channel::ChannelCallbackStatus /* status */,
    void* userData
) NN_NOEXCEPT
{
    SequenceTrack* track = reinterpret_cast<SequenceTrack*>( userData );

    NN_SDK_ASSERT_NOT_NULL( dropChannel );
    NN_SDK_ASSERT_NOT_NULL( track );

    if ( track->m_pSequenceSoundPlayer != NULL ) {
        track->m_pSequenceSoundPlayer->ChannelCallback( dropChannel );
    }

    // チャンネル参照の切断
    if ( track->m_pChannelList == dropChannel ) {
        track->m_pChannelList = dropChannel->GetNextTrackChannel();
        return;
    }

    Channel* channel = track->m_pChannelList;
    NN_SDK_ASSERT_NOT_NULL( channel );
    while( channel->GetNextTrackChannel() != NULL ) {
        if ( channel->GetNextTrackChannel() == dropChannel ) {
            channel->SetNextTrackChannel( dropChannel->GetNextTrackChannel() );
            return;
        }
        channel = channel->GetNextTrackChannel();
    }

    NN_SDK_ASSERT( false );
}

void SequenceTrack::SetMute( SequenceMute mute ) NN_NOEXCEPT
{
    if ( m_ForceMute )
    {
        return;
    }

    switch ( mute )
    {
    case SequenceMute_Off:
        m_ParserTrackParam.muteFlag = false;
        break;
    case SequenceMute_Stop:
        StopAllChannel();
        m_ParserTrackParam.muteFlag = true;
        break;
    case SequenceMute_Release:
        ReleaseAllChannel( -1 );
        FreeAllChannel();
        m_ParserTrackParam.muteFlag = true;
        break;
    case SequenceMute_NoStop:
        m_ParserTrackParam.muteFlag = true;
        break;
    default:
        break;
    }
}

void SequenceTrack::ForceMute() NN_NOEXCEPT
{
    m_ForceMute = true;
    m_ParserTrackParam.muteFlag = true;
}

void SequenceTrack::SetSilence( bool silenceFlag, int fadeTimes ) NN_NOEXCEPT
{
    m_ParserTrackParam.silenceFlag = silenceFlag;

    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        channel->SetSilence(
            silenceFlag,
            ( fadeTimes + HardwareManager::SoundFrameIntervalMsec - 1 ) /
              HardwareManager::SoundFrameIntervalMsec );
        channel = channel->GetNextTrackChannel();
    }
}

void SequenceTrack::SetBiquadFilter( int type, float value ) NN_NOEXCEPT
{
    m_ParserTrackParam.biquadType = static_cast<int8_t>( type );
    m_ParserTrackParam.biquadValue = value;
}

void SequenceTrack::SetBankIndex( int bankIndex ) NN_NOEXCEPT
{
    // SequenceSoundHandle::SetTrackBankIndex でも範囲チェックしているが、念のため。
    NN_SDK_ASSERT_RANGE( bankIndex, SequenceSound::BankIndexMin, SequenceSound::BankIndexMax );
    m_ParserTrackParam.bankIndex = static_cast<uint8_t>( bankIndex );
}

void SequenceTrack::SetTranspose( int8_t transpose ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( transpose, SequenceSound::TransposeMin, SequenceSound::TransposeMax );
    m_ParserTrackParam.transpose = transpose;
}

void SequenceTrack::SetVelocityRange( uint8_t range ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( range <= SequenceSound::VelocityRangeMax );
    m_ParserTrackParam.velocityRange = range;
}

void SequenceTrack::SetOutputLine( int32_t outputLine ) NN_NOEXCEPT
{
    m_ParserTrackParam.outputLine = outputLine;
}

void SequenceTrack::SetTvMixParameter(uint32_t srcChNo, int32_t mixChNo, float param) NN_NOEXCEPT
{
    m_TvParam.mixParameter[srcChNo].ch[mixChNo] = param;
}

/*--------------------------------------------------------------------------------*
  Name:         GetTrackVariable

  Description:  シーケンストラック変数を取得します。

  Arguments:    varNo - 変数番号

  Returns:      変数の値
 *--------------------------------------------------------------------------------*/
int16_t SequenceTrack::GetTrackVariable( int varNo ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, TrackVariableCount );

    return m_TrackVariable[ varNo ];
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackVariable

  Description:  シーケンスローカル変数を設定します。

  Arguments:    player - プレイヤー
                varNo - 変数番号
                var - 変数値

  Returns:      変数の値
 *--------------------------------------------------------------------------------*/
void SequenceTrack::SetTrackVariable( int varNo, int16_t var ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, TrackVariableCount );

    m_TrackVariable[ varNo ] = var;
}

/*--------------------------------------------------------------------------------*
  Name:         GetVariablePtr

  Description:  変数のポインタを取得します

  Arguments:    player - プレイヤーポインタ
                varNo    - 変数番号

  Returns:      変数のポインタ
 *--------------------------------------------------------------------------------*/
volatile int16_t* SequenceTrack::GetVariablePtr( int varNo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( varNo >= 0 && varNo <= TrackVariableCount );

    if ( varNo < TrackVariableCount )
    {
        return & m_TrackVariable[ varNo ];
    }
    else
    {
        return NULL;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         NoteOn

  Description:  ノートオン

  Arguments:    key      - キー番号
                velocity - ベロシティ
                length   - ノート長 ( -1 なら無限長 )
                tieFlag  - タイフラグ

  Returns:      発音したチャンネルのポインタ
 *--------------------------------------------------------------------------------*/
Channel* SequenceTrack::NoteOn(
    int key,
    int velocity,
    int32_t length,
    bool tieFlag
) NN_NOEXCEPT
{
    NN_SDK_ASSERT( key >= 0 && key <= 127 );
    NN_SDK_ASSERT( velocity >= 0 && velocity <= 127 );

    const SequenceSoundPlayer::ParserPlayerParam& playerParam = m_pSequenceSoundPlayer->GetParserPlayerParam();

    Channel* channel = NULL;

    // ベロシティレンジ適用
    velocity = velocity * m_ParserTrackParam.velocityRange / 127;

    if ( tieFlag )
    {
        // タイ
        channel = GetLastChannel();
        if ( channel != NULL )
        {
            channel->SetKey( static_cast<uint8_t>( key ) );
            channel->SetVelocity( Bank::CalcChannelVelocityVolume(static_cast<uint8_t>(velocity)) );
            // TODO: パンも更新するか？
        }
    }

    if ( GetParserTrackParam().monophonicFlag )
    {
        // モノフォニック
        channel = GetLastChannel();
        if ( channel != NULL )
        {
            if ( channel->IsRelease() )
            {
                channel->Stop();
                channel->CallChannelCallback( Channel::ChannelCallbackStatus_Stopped );
                Channel::FreeChannel(channel);
                channel = NULL;
            }
            else
            {
                channel->SetKey( static_cast<uint8_t>( key ) );
                channel->SetVelocity( Bank::CalcChannelVelocityVolume(static_cast<uint8_t>(velocity)) );
                channel->SetLength( length );
            }
        }
    }

    if ( channel == NULL )
    {
        NoteOnInfo noteOnInfo = {
            m_ParserTrackParam.prgNo,
            key,
            velocity,
            tieFlag ? -1 : length,
            m_ParserTrackParam.initPan,
            playerParam.priority + m_ParserTrackParam.priority,
            SequenceTrack::ChannelCallbackFunc,
            this,
            m_pSequenceSoundPlayer->GetOutputReceiver(),
            m_pSequenceSoundPlayer->GetUpdateType(),
        };
        channel = m_pSequenceSoundPlayer->NoteOn(
            m_ParserTrackParam.bankIndex,
            noteOnInfo
        );
        if ( channel == NULL ) return NULL;

        // オルタネートアサイン処理
        if ( channel->GetKeyGroupId() > 0 )
        {
            Channel* itr = m_pChannelList;
            while( itr != NULL )
            {
                if ( itr->GetKeyGroupId() == channel->GetKeyGroupId() )
                {
                    itr->SetRelease(126);
                    itr->Release();
                }
                itr = itr->GetNextTrackChannel();
            }
        }

        // チャンネルリストへの結合
        AddChannel( channel );
    }

    // 初期パラメータ設定

    // エンベロープ更新
    if ( m_ParserTrackParam.attack <= SequenceTrack::MaxEnvelopeValue )
    {
        channel->SetAttack( m_ParserTrackParam.attack );
    }

    if ( m_ParserTrackParam.decay <= SequenceTrack::MaxEnvelopeValue )
    {
        channel->SetDecay( m_ParserTrackParam.decay );
    }

    if ( m_ParserTrackParam.sustain <= SequenceTrack::MaxEnvelopeValue )
    {
        channel->SetSustain( m_ParserTrackParam.sustain );
    }

    if ( m_ParserTrackParam.release <= SequenceTrack::MaxEnvelopeValue )
    {
        channel->SetRelease( m_ParserTrackParam.release );
    }
    if ( m_ParserTrackParam.envHold <= SequenceTrack::MaxEnvelopeValue )
    {
        channel->SetHold( m_ParserTrackParam.envHold );
    }

    // スイープ＆ポルタメント更新
    float sweepPitch = m_ParserTrackParam.sweepPitch;

    if ( m_ParserTrackParam.portaFlag )
    {
        sweepPitch += m_ParserTrackParam.portaKey - key;
    }

    if ( m_ParserTrackParam.portaTime == 0 )
    {
        if (length == 0)
        {
            NN_ATK_WARNING("portatime zero is invalid.");
        }
        channel->SetSweepParam( sweepPitch, length, false );
    }
    else
    {
        int sweepTime = m_ParserTrackParam.portaTime;
        sweepTime *= sweepTime;
        sweepTime = static_cast<int>( sweepTime * ( sweepPitch >= 0 ? sweepPitch : -sweepPitch ) );
        sweepTime >>= 5;
        sweepTime *= 5; // 単位はmsec

        channel->SetSweepParam( sweepPitch, sweepTime, true );
    }

    // ポルタメント開始キーを更新
    m_ParserTrackParam.portaKey = static_cast<uint8_t>( key );

    // その他
    channel->SetSilence( m_ParserTrackParam.silenceFlag != 0, 0 );
    channel->SetReleasePriorityFix( m_pSequenceSoundPlayer->IsReleasePriorityFix() );
    channel->SetPanMode( m_pSequenceSoundPlayer->GetPanMode() );
    channel->SetPanCurve( m_pSequenceSoundPlayer->GetPanCurve() );

    return channel;
} // NOLINT(impl/function_size)

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

