﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SequenceSoundFile.h>
#include <nn/atk/atk_ElementType.h>
#include <cstring>      // strlen, strncmp

namespace nn {
namespace atk {
namespace detail {

//
// SequenceSoundFile::FileHeader
//
const SequenceSoundFile::DataBlock* SequenceSoundFile::FileHeader::GetDataBlock() const NN_NOEXCEPT
{
    return reinterpret_cast<const DataBlock*>( GetBlock( ElementType_SequenceSoundFile_DataBlock ) );
}
const SequenceSoundFile::LabelBlock* SequenceSoundFile::FileHeader::GetLabelBlock() const NN_NOEXCEPT
{
    return reinterpret_cast<const LabelBlock*>( GetBlock( ElementType_SequenceSoundFile_LabelBlock ) );
}


//
// SequenceSoundFile::LabelBlockBody
//

const SequenceSoundFile::LabelInfo* SequenceSoundFile::LabelBlockBody::GetLabelInfo( int index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( index < GetLabelCount() );

    return reinterpret_cast<const LabelInfo*>(
            labelInfoReferenceTable.GetReferedItem( index ) );
}

const char* SequenceSoundFile::LabelBlockBody::GetLabel( int index ) const NN_NOEXCEPT
{
    const LabelInfo* labelInfo = GetLabelInfo( index );
    NN_SDK_ASSERT_NOT_NULL(labelInfo);
    return labelInfo->label;
}

const char* SequenceSoundFile::LabelBlockBody::GetLabelByOffset( uint32_t offset ) const NN_NOEXCEPT
{
    // 線形探索
    for ( int i = 0; i < GetLabelCount(); i++ )
    {
        const LabelInfo* labelInfo = GetLabelInfo( i );
        NN_SDK_ASSERT_NOT_NULL(labelInfo);
        if ( labelInfo->referToSequenceData.offset == static_cast<int32_t>(offset) )
        {
            return labelInfo->label;
        }
    }
    return NULL;
}

bool SequenceSoundFile::LabelBlockBody::GetOffset( int index, uint32_t* offsetPtr ) const NN_NOEXCEPT
{
    const LabelInfo* labelInfo = GetLabelInfo( index );
    NN_SDK_ASSERT_NOT_NULL(labelInfo);
    *offsetPtr = labelInfo->referToSequenceData.offset;
    return true;
}

bool SequenceSoundFile::LabelBlockBody::GetOffsetByLabel( const char* label, uint32_t* offsetPtr ) const NN_NOEXCEPT
{
    const std::size_t labelLength = std::strlen( label );

    // 線形探索
    for ( int i = 0; i < GetLabelCount(); i++ )
    {
        const LabelInfo* labelInfo = GetLabelInfo( i );
        NN_SDK_ASSERT_NOT_NULL(labelInfo);
        if ( std::strncmp( label, labelInfo->label, labelLength ) == 0 )
        {
            *offsetPtr = labelInfo->referToSequenceData.offset;
            return true;
        }
    }
    return false;
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

