﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>
#include <nn/atk/atk_DspadpcmReader.h>

struct InternalDSPADPCMInfo
{
    // for header generation during decode
    uint32_t sampleCount;       // total number of RAW samples
    uint32_t adpcmNibbleCount;  // number of ADPCM nibbles (including frame headers)
    uint32_t sampleRate;        // Sample rate, in Hz

    // DSP addressing and decode context
    uint16_t loopFlag;    // 1=LOOPED, 0=NOT LOOPED
    uint16_t format;       // Always 0x0000, for ADPCM
    uint32_t sa;           // Start offset address for looped samples (2 for non-looped)
    uint32_t ea;           // End offset address for looped samples
    uint32_t ca;           // always 2
    uint16_t coef[16];     // decode coefficients (eight pairs of 16-bit words)

    // DSP decoder initial state
    uint16_t gain;         // always zero for ADPCM
    uint16_t ps;           // predictor/scale
    uint16_t yn1;          // sample history
    uint16_t yn2;          // sample history

    // DSP decoder loop context
    uint16_t lps;          // predictor/scale for loop context
    uint16_t lyn1;         // sample history (n-1) for loop context
    uint16_t lyn2;         // sample history (n-2) for loop context

    uint16_t pad[11];      // reserved
};

namespace nn {
namespace atk {
namespace detail {

DspadpcmReader::DspadpcmReader() NN_NOEXCEPT
: m_pDspadpcmData( NULL )
{
}

bool DspadpcmReader::ReadWaveInfo(WaveInfo* info) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(info);
    const InternalDSPADPCMInfo& data = *reinterpret_cast<const InternalDSPADPCMInfo*>(m_pDspadpcmData);

    info->sampleFormat = SampleFormat_DspAdpcm;
    info->loopFlag = false; // ループ未対応。対応時は、data.loopFlag == 1 ? true : false; とする。
    info->channelCount = 1;
    info->sampleRate = data.sampleRate;
    info->loopStartFrame = 0;
    info->loopEndFrame = data.sampleCount;

    // dspadpcm はモノラル波形しか出力しない
    info->channelParam[0].dataAddress = util::ConstBytePtr(m_pDspadpcmData, sizeof(InternalDSPADPCMInfo)).Get();
    std::memcpy(info->channelParam[0].adpcmParam.coef, &data.coef, sizeof(uint16_t) * 16);
    std::memcpy(&info->channelParam[0].adpcmParam.predScale, &data.ps, sizeof(uint16_t) * 3);

    return true;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

