﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Debug.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <cstring>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

namespace nn {
namespace atk {

namespace {

const uint32_t NotEnoughSeqSound    = 0x00000001;
const uint32_t NotEnoughStrmSound   = 0x00000002;
const uint32_t NotEnoughWaveSound   = 0x00000004;
const uint32_t NotEnoughSeqTrack    = 0x00000008;
const uint32_t NotEnoughStrmChannel = 0x00000010;
const uint32_t NotEnoughInstance    = NotEnoughSeqSound |
                                   NotEnoughStrmSound |
                                   NotEnoughWaveSound |
                                   NotEnoughSeqTrack |
                                   NotEnoughStrmChannel;

uint32_t gWarningFlag = NotEnoughInstance;

uint32_t GetWarningBitFlag( DebugWarningFlag warning ) NN_NOEXCEPT
{
    uint32_t bitFlag = 0;
    switch ( warning )
    {
    case DebugWarningFlag_NotEnoughInstance:
        bitFlag = NotEnoughInstance;
        break;
    case DebugWarningFlag_NotEnoughSeqsound:
        bitFlag = NotEnoughSeqSound;
        break;
    case DebugWarningFlag_NotEnoughStrmsound:
        bitFlag = NotEnoughStrmSound;
        break;
    case DebugWarningFlag_NotEnoughWavesound:
        bitFlag = NotEnoughWaveSound;
        break;
    case DebugWarningFlag_NotEnoughSeqtrack:
        bitFlag = NotEnoughSeqTrack;
        break;
    case DebugWarningFlag_NotEnoughStrmchannel:
        bitFlag = NotEnoughStrmChannel;
        break;
    default:
        break;
    }
    return bitFlag;
}

} // anonymous namespace

/*--------------------------------------------------------------------------------*
  Name:         Debug_SetWarningFlag

  Description:  警告メッセージの出力を設定する

  Arguments:    warning -   警告メッセージの種類
                enable -    出力するなら true

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Debug_SetWarningFlag( DebugWarningFlag warning, bool enable ) NN_NOEXCEPT
{
    uint32_t bitFlag = GetWarningBitFlag( warning );

    if ( enable )
    {
        gWarningFlag |= bitFlag;
    }
    else
    {
        gWarningFlag &= ~bitFlag;
    }
}


/* ========================================================================
        非公開
   ======================================================================== */

namespace detail {

DebugLogFunc g_DebugLogHookFunc = NULL;

bool Debug_GetWarningFlag( DebugWarningFlag warning ) NN_NOEXCEPT
{
    uint32_t bitFlag = GetWarningBitFlag( warning );

    return ( gWarningFlag & bitFlag ) == bitFlag;
}

DebugWarningFlag Debug_GetDebugWarningFlagFromSoundType( DebugSoundType type ) NN_NOEXCEPT
{
    switch ( type )
    {
    case DebugSoundType_Seqsound:
        return DebugWarningFlag_NotEnoughSeqsound;
    case DebugSoundType_Strmsound:
        return DebugWarningFlag_NotEnoughStrmsound;
    case DebugSoundType_Wavesound:
        return DebugWarningFlag_NotEnoughWavesound;
    default:
        NN_SDK_ASSERT( false );
        return DebugWarningFlag_NotEnoughSeqsound;
    }
}

const char* Debug_GetSoundTypeString( DebugSoundType type ) NN_NOEXCEPT
{
    switch ( type )
    {
    case DebugSoundType_Seqsound:
        return "seq";
    case DebugSoundType_Strmsound:
        return "strm";
    case DebugSoundType_Wavesound:
        return "wave";
    default:
        NN_SDK_ASSERT( false );
        return "";
    }
}

#if !defined( NN_SDK_BUILD_RELEASE )
void Debug_OutputLog( const char* category, const char* message, ... ) NN_NOEXCEPT
{
    static const int TextMaxLength = 255;
    char text[TextMaxLength + 1];

    int categoryLength = category == NULL ? 0 : static_cast<int>(std::strlen(category));

    if ( categoryLength < 0 ) {
        return;
    }

    util::Strlcpy(text, category, categoryLength + 1);

    int length = categoryLength;
    int restLength = TextMaxLength + 1 - categoryLength;

    if(restLength > 0)
    {
        va_list arglist;
        va_start(arglist, message);
        length += nn::util::VSNPrintf(text + length, restLength, message, arglist);
        va_end(arglist);
    }

    if ( length < categoryLength ) {
        // format error
        return;
    }

    if ( length > TextMaxLength ) length = TextMaxLength;
    text[length] = '\0';

    if ( g_DebugLogHookFunc != NULL )
    {
        g_DebugLogHookFunc( text );
    }
}

#endif

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

