﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveArchiveFileReader.h>

// #define NN_ATK_DEBUG_PRINT_ENABLE

namespace nn {
namespace atk {
namespace detail {

namespace
{

// const uint32_t SignatureInfoBlock = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );
// const uint32_t SignatureFileBlock = NN_UTIL_CREATE_SIGNATURE_4( 'F', 'I', 'L', 'E' );

const uint32_t SupportedFileVersionWar = 0x00010000;  // ライブラリがサポートする最低バージョン
const uint32_t CurrentFileVersionWar   = 0x00010000;  // ライブラリがサポートする最新バージョン

bool IsValidFileHeaderWar( const void* waveArchiveData ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( waveArchiveData );

    // シグニチャ確認
    NN_SDK_ASSERT( header.signature == WaveArchiveFileReader::SignatureFile,
            "invalid file signature." );
    if ( header.signature != WaveArchiveFileReader::SignatureFile )
    {
        return false;
    }

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header.byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header.byteOrder );
    if ( header.byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionWar <= header.version) &&
                                   (header.version <= CurrentFileVersionWar) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfwar file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionWar, CurrentFileVersionWar, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t WaveArchiveFileReader::SignatureFile );
NN_DEFINE_STATIC_CONSTANT( const uint32_t WaveArchiveFileReader::SignatureWarcTable );

WaveArchiveFileReader::WaveArchiveFileReader() NN_NOEXCEPT
: m_pHeader( NULL )
, m_pInfoBlockBody( NULL )
, m_pLoadTable( NULL )
, m_IsInitialized( false )
{
}

WaveArchiveFileReader::WaveArchiveFileReader(
        const void* pWaveArchiveFile,
        bool isIndividual ) NN_NOEXCEPT
: m_pHeader( NULL )
, m_pInfoBlockBody( NULL )
, m_pLoadTable( NULL )
, m_IsInitialized( false )
{
    Initialize( pWaveArchiveFile, isIndividual );
}

void WaveArchiveFileReader::Initialize(
        const void* pWaveArchiveFile,
        bool isIndividual ) NN_NOEXCEPT
{
    if ( pWaveArchiveFile == NULL )
    {
        return;
    }

    if ( ! IsValidFileHeaderWar( pWaveArchiveFile ) )
    {
        return;
    }

    m_pHeader =
        reinterpret_cast<const WaveArchiveFile::FileHeader*>( pWaveArchiveFile );
    m_pInfoBlockBody = &m_pHeader->GetInfoBlock()->body;

    m_IsInitialized = true;

    // MemorySoundArchive 内の波形アーカイブが個別ロード ON であっても、
    // Hasindividualloadtable が false を返すので、m_pLoadTable は NULL のままに
    // なり、IsLoaded が正しく動作する。
    m_pLoadTable = NULL;
    if ( isIndividual && HasIndividualLoadTable() )
    {
        m_pLoadTable = reinterpret_cast<IndividualLoadTable*>(
                util::BytePtr(
                    const_cast<void*>( pWaveArchiveFile ),
                    m_pHeader->GetFileBlockOffset() +
                    sizeof( SignatureWarcTable ) ).Get() );
            // INFO ブロックと FILE ブロックの間に、
            // アライメントがはさまっているので、その分ムダになるが、
            // 見通しのよさを優先する。
    }
}

void WaveArchiveFileReader::Finalize() NN_NOEXCEPT
{
    if ( m_IsInitialized )
    {
        m_pHeader = NULL;
        m_pInfoBlockBody = NULL;
        m_pLoadTable = NULL;
        m_IsInitialized = false;
    }
}

void WaveArchiveFileReader::InitializeFileTable() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );

    // ロード管理用ファイルテーブルの初期化
    for ( uint32_t i = 0; i < GetWaveFileCount(); i++ )
    {
        m_pLoadTable->waveFile[ i ] = nullptr;
    }
}

uint32_t WaveArchiveFileReader::GetWaveFileCount() const NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetWaveFileCount();
}

const void* WaveArchiveFileReader::GetWaveFile( uint32_t waveIndex ) const NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    if ( waveIndex >= GetWaveFileCount() ) return NULL;

    if ( m_pLoadTable != NULL )
    {
        return GetWaveFileForIndividual( waveIndex );
    }
    return GetWaveFileForWhole( waveIndex );
}

uint32_t WaveArchiveFileReader::GetWaveFileSize( uint32_t waveIndex ) const NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetSize( waveIndex );
}

uint32_t WaveArchiveFileReader::GetWaveFileOffsetFromFileHead( uint32_t waveIndex ) const NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    uint32_t result =
        + m_pHeader->GetFileBlockOffset()
        + offsetof( WaveArchiveFile::FileBlock, body )
        + m_pInfoBlockBody->GetOffsetFromFileBlockBody( waveIndex );
    return result;
}

const void*
WaveArchiveFileReader::SetWaveFile(
        uint32_t waveIndex, const void* pWaveFile ) NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    if ( m_pLoadTable == NULL ) return NULL;
    if ( waveIndex >= GetWaveFileCount() ) return NULL;

    const void* preAddress = GetWaveFileForIndividual( waveIndex );
    m_pLoadTable->waveFile[ waveIndex ] = pWaveFile;

#ifdef NN_ATK_DEBUG_PRINT_ENABLE
    // デバッグ
    for ( uint32_t i = 0; i < GetWaveFileCount(); i++ )
    {
        NN_DETAIL_ATK_INFO("  [%3d] %p\n", i, m_pLoadTable->waveFile[i] );
    }
#endif /* NN_ATK_DEBUG_PRINT_ENABLE */

    return preAddress;
}

bool WaveArchiveFileReader::HasIndividualLoadTable() const NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    const uint32_t* signature = reinterpret_cast<const uint32_t*>(
            util::ConstBytePtr( m_pHeader, m_pHeader->GetFileBlockOffset() ).Get() );
    if ( *signature == SignatureWarcTable )
    {
        return true;
    }
    return false;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
