﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundArchiveFileReader.h>
#include <nn/atk/atk_ItemType.h>
#include <nn/atk/fnd/binary/atkfnd_Binary.h>

namespace nn {
namespace atk {
namespace detail {

namespace {

// const uint32_t SignatureStringBlock        = NN_UTIL_CREATE_SIGNATURE_4( 'S', 'T', 'R', 'G' );
// const uint32_t SignatureInfoBlock          = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );
// const uint32_t SignatureDataBlock          = NN_UTIL_CREATE_SIGNATURE_4( 'D', 'A', 'T', 'A' );
const uint32_t SupportedFileVersionSar         = NN_ATK_MAKE_VERSION( 0, 1, 0, 0 );
const uint32_t CurrentFileVersionSar           = NN_ATK_MAKE_VERSION( 0, 2, 5, 0 );
const uint32_t StrmsendSupportedVersionSar     = NN_ATK_MAKE_VERSION( 0, 2, 1, 0 );
const uint32_t FilterSupportedVersionSar       = NN_ATK_MAKE_VERSION( 0, 2, 1, 0 );
const uint32_t StrmprefetchSupportedVersionSar = NN_ATK_MAKE_VERSION( 0, 2, 2, 0 );

bool IsValidFileHeaderSar( const void* soundArchiveData ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( soundArchiveData );

    // シグニチャ確認
    NN_SDK_ASSERT( header.signature == SoundArchiveFileReader::SignatureFile,
            "invalid file signature." );
    if ( header.signature != SoundArchiveFileReader::SignatureFile )
    {
        return false;
    }

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header.byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header.byteOrder );
    if ( header.byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionSar <= header.version) &&
                                   (header.version <= CurrentFileVersionSar) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfsar file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionSar, CurrentFileVersionSar, header.version
    );
    return isSupportedVersion;
}

const SoundArchiveFile::StreamTrackInfo*
GetStreamTrackInfo(uint32_t index, const SoundArchiveFile::StreamTrackInfoTable& table) NN_NOEXCEPT
{
    return static_cast<const SoundArchiveFile::StreamTrackInfo*>(
            table.table.GetReferedItem(index, ElementType_SoundArchiveFile_StreamSoundTrackInfo));
}

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchiveFileReader::SignatureFile );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchiveFileReader::InvalidOffset );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchiveFileReader::InvalidSize );

SoundArchiveFileReader::SoundArchiveFileReader() NN_NOEXCEPT
: m_pStringBlockBody( NULL )
, m_pInfoBlockBody( NULL )
{
}

void SoundArchiveFileReader::Initialize( const void* soundArchiveData ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( soundArchiveData );
    if ( ! IsValidFileHeaderSar( soundArchiveData ) ) return;

    const SoundArchiveFile::FileHeader& header =
        *reinterpret_cast<const SoundArchiveFile::FileHeader*>( soundArchiveData );
    m_Header = header;  // 構造体コピー
}

void SoundArchiveFileReader::Finalize() NN_NOEXCEPT
{
    m_pStringBlockBody = NULL;
    m_pInfoBlockBody = NULL;
}

bool SoundArchiveFileReader::IsStreamSendAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( &m_Header );
    if (header.version >= StrmsendSupportedVersionSar)
    {
        return true;
    }
    return false;
}

bool SoundArchiveFileReader::IsFilterSupportedVersion() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( &m_Header );
    if (header.version >= FilterSupportedVersionSar)
    {
        return true;
    }
    return false;
}

bool SoundArchiveFileReader::IsStreamPrefetchAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( &m_Header );
    if (header.version >= StrmprefetchSupportedVersionSar)
    {
        return true;
    }
    return false;
}

void SoundArchiveFileReader::SetStringBlock( const void* stringBlock ) NN_NOEXCEPT
{
    m_pStringBlockBody =
        &( reinterpret_cast<const SoundArchiveFile::StringBlock*>( stringBlock )->body );
}

void SoundArchiveFileReader::SetInfoBlock( const void* infoBlock ) NN_NOEXCEPT
{
    m_pInfoBlockBody =
        &( reinterpret_cast<const SoundArchiveFile::InfoBlock*>( infoBlock )->body );
}

uint32_t SoundArchiveFileReader::GetStringCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pStringBlockBody );
    return m_pStringBlockBody->GetStringCount();
}
const char* SoundArchiveFileReader::GetString( SoundArchive::StringId stringId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pStringBlockBody );
    return m_pStringBlockBody->GetString( stringId );
}

void SoundArchiveFileReader::DumpTree() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pStringBlockBody );
    m_pStringBlockBody->DumpTree();
}

SoundArchive::ItemId SoundArchiveFileReader::GetItemId( const char* pStr ) const NN_NOEXCEPT
{
    if ( m_pStringBlockBody == NULL )
    {
        return SoundArchive::InvalidId;
    }
    return m_pStringBlockBody->GetItemId( pStr );
}
const char* SoundArchiveFileReader::GetItemLabel( SoundArchive::ItemId id ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    if ( m_pStringBlockBody == NULL )
    {
        return NULL;
    }
    SoundArchive::StringId stringId = m_pInfoBlockBody->GetItemStringId( id );
    if ( stringId == SoundArchive::InvalidId )
    {
        return NULL;
    }
    return m_pStringBlockBody->GetString( stringId );
}
SoundArchive::FileId SoundArchiveFileReader::GetItemFileId( SoundArchive::ItemId id ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetItemFileId( id );
}

SoundArchive::FileId SoundArchiveFileReader::GetItemPrefetchFileId( SoundArchive::ItemId id ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetItemPrefetchFileId( id );
}

uint32_t SoundArchiveFileReader::GetSoundCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetSoundCount();
}

uint32_t SoundArchiveFileReader::GetBankCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetBankCount();
}

uint32_t SoundArchiveFileReader::GetPlayerCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetPlayerCount();
}

uint32_t SoundArchiveFileReader::GetSoundGroupCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetSoundGroupCount();
}

uint32_t SoundArchiveFileReader::GetGroupCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetGroupCount();
}

uint32_t SoundArchiveFileReader::GetWaveArchiveCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetWaveArchiveCount();
}

uint32_t SoundArchiveFileReader::GetFileCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetFileCount();
}

bool SoundArchiveFileReader::ReadSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::SoundInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundInfo* data = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( data == NULL ) return false;

    info->fileId = data->fileId;
    info->playerId = data->playerId;
    info->volume = data->volume;
    info->panMode = data->GetPanMode();
    info->panCurve = data->GetPanCurve();
    info->singlePlayType = data->GetSinglePlayType();
    info->singlePlayEffectiveDuration = data->GetSinglePlayEffectiveDuration();
    info->playerPriority = data->GetPlayerPriority();
    info->actorPlayerId = data->GetActorPlayerId();
    info->remoteFilter = data->remoteFilter;
    info->isFrontBypass = data->IsFrontBypass();

    return true;
}

bool SoundArchiveFileReader::ReadBankInfo(
        SoundArchive::ItemId bankId, SoundArchive::BankInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::BankInfo* data = m_pInfoBlockBody->GetBankInfo( bankId );
    if ( data == NULL ) return false;

    info->fileId = data->fileId;
    return true;
}

bool SoundArchiveFileReader::ReadPlayerInfo(
        SoundArchive::ItemId playerId, SoundArchive::PlayerInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::PlayerInfo* data = m_pInfoBlockBody->GetPlayerInfo( playerId );
    if ( data == NULL ) return false;

    info->playableSoundMax = data->playableSoundMax;
    info->playerHeapSize = data->GetPlayerHeapSize();

    return true;
}

bool SoundArchiveFileReader::ReadSoundGroupInfo(
        SoundArchive::ItemId soundGroupId, SoundArchive::SoundGroupInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundGroupInfo* data = m_pInfoBlockBody->GetSoundGroupInfo( soundGroupId );
    if ( data == NULL ) return false;

    info->startId = data->startId;
    info->endId = data->endId;
    info->fileIdTable = data->GetFileIdTable();
    return true;
}

bool SoundArchiveFileReader::ReadGroupInfo(
        SoundArchive::ItemId groupId, SoundArchive::GroupInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    // ファイル ID
    {
        const SoundArchiveFile::GroupInfo* data = m_pInfoBlockBody->GetGroupInfo( groupId );
        if ( data == NULL )
        {
            return false;
        }
        info->fileId = data->fileId;
    }

    // ファイルサイズ
    {
        SoundArchive::FileInfo fileInfo;
        bool result = ReadFileInfo( info->fileId, &fileInfo );
        if ( ! result )
        {
            return false;
        }
        info->groupFileSize = fileInfo.fileSize;
    }

    return true;
}

bool SoundArchiveFileReader::ReadWaveArchiveInfo(
        SoundArchive::ItemId warcId, SoundArchive::WaveArchiveInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::WaveArchiveInfo* data = m_pInfoBlockBody->GetWaveArchiveInfo( warcId );
    if ( data == NULL ) return false;

    info->fileId = data->fileId;
    info->waveCount = data->GetWaveCount();
    info->isLoadIndividual = data->isLoadIndividual;
    return true;
}

bool SoundArchiveFileReader::ReadFileInfo(
        SoundArchive::FileId id, SoundArchive::FileInfo* info, int index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );
    NN_UNUSED( index );

    const SoundArchiveFile::FileInfo* data = m_pInfoBlockBody->GetFileInfo( id );
    if ( data == NULL ) return false;

    switch ( data->GetFileLocationType() )
    {
        case SoundArchiveFile::FileLocationType_Internal:
            {
                const SoundArchiveFile::InternalFileInfo* location = data->GetInternalFileInfo();
                info->fileSize = location->GetFileSize();
                info->offsetFromFileBlockHead = location->GetOffsetFromFileBlockHead();
            }
            break;
        case SoundArchiveFile::FileLocationType_External:
            {
                const SoundArchiveFile::ExternalFileInfo* location = data->GetExternalFileInfo();
                info->externalFilePath = location->filePath;
            }
            break;
        case SoundArchiveFile::FileLocationType_None:
            return false;
        default:
            return false;
    }
    return true;
}

bool SoundArchiveFileReader::ReadSoundArchivePlayerInfo(
        SoundArchive::SoundArchivePlayerInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundArchivePlayerInfo* data = m_pInfoBlockBody->GetSoundArchivePlayerInfo();
    if ( data == NULL ) return false;

    info->sequenceSoundCount  = data->sequenceSoundCount;
    info->sequenceTrackCount  = data->sequenceTrackCount;
    info->streamSoundCount    = data->streamSoundCount;
    info->streamTrackCount    = data->streamTrackCount;
    info->streamChannelCount  = data->streamChannelCount;
    info->waveSoundCount      = data->waveSoundCount;
    info->waveTrackCount      = data->waveTrackCount;
    if ( IsStreamPrefetchAvailable() )
    {
        info->streamBufferTimes = data->streamBufferTimes;
    }
    else
    {
        info->streamBufferTimes = 1;
    }
    info->isAdvancedWaveSoundEnabled = (data->developFlags & 0x1) == 0x1;

    return true;
}

bool SoundArchiveFileReader::ReadSound3DInfo(
        SoundArchive::ItemId soundId, SoundArchive::Sound3DInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }

    const SoundArchiveFile::Sound3DInfo* sound3dInfo = soundInfo->GetSound3DInfo();
    if ( sound3dInfo == NULL )
    {
        return false;
    }
    info->flags = sound3dInfo->flags;
    info->decayRatio = sound3dInfo->decayRatio;
    info->decayCurve = sound3dInfo->decayCurve;
    info->dopplerFactor = sound3dInfo->dopplerFactor;
    return true;
}

bool SoundArchiveFileReader::ReadSequenceSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::SequenceSoundInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SoundType_Sequence )
    {
        return false;
    }

    const SoundArchiveFile::SequenceSoundInfo& seqSoundInfo = soundInfo->GetSequenceSoundInfo();
    info->startOffset = seqSoundInfo.GetStartOffset();
    info->allocateTrackFlags = seqSoundInfo.allocateTrackFlags;
    info->channelPriority = seqSoundInfo.GetChannelPriority();
    info->isReleasePriorityFix = seqSoundInfo.IsReleasePriorityFix();
    seqSoundInfo.GetBankIds( info->bankIds );
    return true;
}

bool SoundArchiveFileReader::ReadStreamSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::StreamSoundInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SoundType_Stream )
    {
        return false;
    }

    const SoundArchiveFile::StreamSoundInfo& strmSoundInfo = soundInfo->GetStreamSoundInfo();
    info->allocateTrackFlags   = strmSoundInfo.allocateTrackFlags;
    info->allocateChannelCount = strmSoundInfo.allocateChannelCount;
    const SoundArchiveFile::StreamSoundExtension* strmSoundExt = strmSoundInfo.GetStreamSoundExtension();
    info->streamFileType = (strmSoundExt == NULL) ? SoundArchive::StreamFileType_NwStreamBinary : strmSoundExt->GetStreamFileType();
    info->decodeMode = (strmSoundExt == NULL) ? SoundArchive::DecodeMode_Default : strmSoundExt->GetDecodeMode();
    if ( IsStreamSendAvailable() )
    {
        info->pitch                            = strmSoundInfo.pitch;
        const SoundArchiveFile::SendValue send = strmSoundInfo.GetSendValue();
        info->mainSend                         = send.mainSend;
        for (uint32_t i = 0; i < AuxBus_Count; i++)
        {
            info->fxSend[i]                    = send.fxSend[i];
        }
    }
    else
    {
        info->pitch = 1.0f;
        info->mainSend = 127;
        for (uint32_t i = 0; i < AuxBus_Count; i++)
        {
            info->fxSend[i] = 0;
        }
    }

    if ( IsStreamPrefetchAvailable() )
    {
        info->prefetchFileId = strmSoundInfo.prefetchFileId;
    }
    else
    {
        info->prefetchFileId = SoundArchive::InvalidId;
    }

    const SoundArchiveFile::StreamTrackInfoTable* table = strmSoundInfo.GetTrackInfoTable();
    if ( table == NULL )
    {
        return true;
    }

    uint32_t tableCount = table->GetTrackCount();
    for ( uint32_t i = 0; i < SoundArchive::StreamTrackCount && i < tableCount; i++ )
    {
        SoundArchive::StreamTrackInfo& tmp = info->trackInfo[i];
        const SoundArchiveFile::StreamTrackInfo* pTrackInfo = GetStreamTrackInfo(i, *table);
        NN_SDK_ASSERT_NOT_NULL(pTrackInfo);
        const SoundArchiveFile::StreamTrackInfo& trackInfo = *pTrackInfo;
        tmp.volume = trackInfo.volume;
        tmp.pan = trackInfo.pan;
        tmp.surroundPan = trackInfo.span;
        tmp.flags = trackInfo.flags;
        tmp.channelCount = static_cast<uint8_t>(trackInfo.GetTrackChannelCount());
        if ( IsStreamSendAvailable() )
        {
            const SoundArchiveFile::SendValue send = trackInfo.GetSendValue();
            tmp.mainSend                           = send.mainSend;
            for (uint32_t j = 0; j < AuxBus_Count; j++)
            {
                tmp.fxSend[j]                      = send.fxSend[j];
            }
        }
        else
        {
            tmp.mainSend = 127;
            for (uint32_t j = 0; j < AuxBus_Count; j++)
            {
                tmp.fxSend[j] = 0;
            }
        }
        if ( IsFilterSupportedVersion() )
        {
            tmp.lowPassFilterFrequency = trackInfo.lpfFreq;
            tmp.biquadType   = trackInfo.biquadType;
            tmp.biquadValue  = trackInfo.biquadValue;
        }
        else
        {
            tmp.lowPassFilterFrequency     = 64;
            tmp.biquadType  = 0;
            tmp.biquadValue = 0;
        }

        uint32_t count = std::min(static_cast<uint32_t>(tmp.channelCount), WaveChannelMax);
        for (uint32_t j = 0; j < count; j++)
        {
            tmp.globalChannelIndex[j] = trackInfo.GetGlobalChannelIndex(j);
        }
    }
    return true;
}

bool SoundArchiveFileReader::ReadStreamSoundInfo2(
    SoundArchive::ItemId soundId, SoundArchive::StreamSoundInfo2* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SoundType_Stream )
    {
        return false;
    }

    const SoundArchiveFile::StreamSoundInfo& strmSoundInfo = soundInfo->GetStreamSoundInfo();
    const SoundArchiveFile::StreamSoundExtension* strmSoundExt = strmSoundInfo.GetStreamSoundExtension();

    if( strmSoundExt == NULL )
    {
        return false;
    }

    info->isLoop         = strmSoundExt->IsLoop();
    info->loopStartFrame = strmSoundExt->loopStartFrame;
    info->loopEndFrame   = strmSoundExt->loopEndFrame;

    return true;
}

bool SoundArchiveFileReader::ReadWaveSoundInfo(
        SoundArchive::ItemId soundId, SoundArchive::WaveSoundInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SoundType_Wave )
    {
        return false;
    }

    const SoundArchiveFile::WaveSoundInfo& waveSoundInfo = soundInfo->GetWaveSoundInfo();
    info->index                 = waveSoundInfo.index;
    info->allocateTrackCount    = waveSoundInfo.allocateTrackCount;
    info->channelPriority       = waveSoundInfo.GetChannelPriority();
    info->isReleasePriorityFix  = waveSoundInfo.GetIsReleasePriorityFix() > 0 ? true : false;

    return true;
}

bool SoundArchiveFileReader::ReadAdvancedWaveSoundInfo(
    SoundArchive::ItemId soundId,
    SoundArchive::AdvancedWaveSoundInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    if ( soundInfo->GetSoundType() != SoundArchive::SoundType_Wave )
    {
        return false;
    }

    const SoundArchiveFile::AdvancedWaveSoundInfo& advancedWaveSoundInfo =
        soundInfo->GetAdvancedWaveSoundInfo();

    info->waveArchiveId = advancedWaveSoundInfo.waveArchiveId;

    return true;
}

const detail::Util::Table<uint32_t>*
SoundArchiveFileReader::GetWaveArchiveIdTable( SoundArchive::ItemId id ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );

    switch ( Util::GetItemType( id ) )
    {
        case ItemType_Sound:
            {
                const SoundArchiveFile::SoundInfo* pInfo =
                    m_pInfoBlockBody->GetSoundInfo( id );
                NN_SDK_ASSERT_NOT_NULL( pInfo );

                if ( pInfo->GetSoundType() == SoundArchive::SoundType_Wave )
                {
                    // ウェーブサウンド ID → ウェーブサウンドセット ID を引く
                    for ( uint32_t i = 0; i < m_pInfoBlockBody->GetSoundGroupCount(); i++ )
                    {
                        const SoundArchiveFile::SoundGroupInfo* pGroupInfo =
                            m_pInfoBlockBody->GetSoundGroupInfo(
                                    Util::GetMaskedItemId( i, ItemType_SoundGroup ) );
                        NN_SDK_ASSERT_NOT_NULL( pGroupInfo );

                        if ( id < pGroupInfo->startId || pGroupInfo->endId < id )
                        {
                            continue;
                        }

                        // 該当するウェーブサウンドセットの波形アーカイブテーブルを返す
                        const SoundArchiveFile::WaveSoundGroupInfo*
                            pWaveSoundGroupInfo = pGroupInfo->GetWaveSoundGroupInfo();
                        NN_SDK_ASSERT_NOT_NULL( pWaveSoundGroupInfo );

                        return pWaveSoundGroupInfo->GetWaveArchiveItemIdTable();
                    }
                }
                return NULL;
            }
        case ItemType_SoundGroup:
            {
                const SoundArchiveFile::SoundGroupInfo* pInfo =
                    m_pInfoBlockBody->GetSoundGroupInfo( id );
                NN_SDK_ASSERT_NOT_NULL( pInfo );

                const SoundArchiveFile::WaveSoundGroupInfo*
                    pWaveSoundGroupInfo = pInfo->GetWaveSoundGroupInfo();
                NN_SDK_ASSERT_NOT_NULL( pWaveSoundGroupInfo );

                return pWaveSoundGroupInfo->GetWaveArchiveItemIdTable();
            }
        case ItemType_Bank:
            {
                const SoundArchiveFile::BankInfo* pInfo =
                    m_pInfoBlockBody->GetBankInfo( id );
                NN_SDK_ASSERT_NOT_NULL( pInfo );

                return pInfo->GetWaveArchiveItemIdTable();
            }
        default:
            return NULL;
    }
}

SoundArchive::SoundType
SoundArchiveFileReader::GetSoundType( SoundArchive::ItemId soundId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return SoundArchive::SoundType_Invalid;
    }
    return soundInfo->GetSoundType();
}

uint32_t SoundArchiveFileReader::GetSoundUserParam( SoundArchive::ItemId soundId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return SoundArchive::ResultInvalidSoundId;
    }
    return soundInfo->GetUserParam();
}

bool SoundArchiveFileReader::ReadSoundUserParam(
        uint32_t* pOutValue, SoundArchive::ItemId soundId, int index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );

    const SoundArchiveFile::SoundInfo* soundInfo = m_pInfoBlockBody->GetSoundInfo( soundId );
    if ( soundInfo == NULL )
    {
        return false;
    }
    return soundInfo->ReadUserParam( pOutValue, index );

}

const Util::Table<uint32_t>* SoundArchiveFileReader::GetAttachedGroupTable(
    SoundArchive::FileId id ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );

    const SoundArchiveFile::FileInfo* fileInfo = m_pInfoBlockBody->GetFileInfo( id );
    if ( fileInfo == NULL )
    {
        return NULL;
    }

    const SoundArchiveFile::InternalFileInfo* internalFileInfo =
        fileInfo->GetInternalFileInfo();

    if ( internalFileInfo == NULL )
    {
        return NULL;
    }

    return internalFileInfo->GetAttachedGroupTable();
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
