﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/detail/atk_OpusDecoderImpl.h>
#include <nn/util/util_BytePtr.h>
#include <nn/atk/detail/atk_Macro.h>

//#define NN_ATK_ENABLE_DEBUG_OPUS_INFO_DUMP

//#define NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER
#if defined(NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER)
    #define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) \
            NN_DETAIL_ATK_INFO("[%p](%s) %s%s\n", this, NN_ATK_FILENAME, __FUNCTION__, AdditionalMessage)
    #define NN_ATK_DEBUG_TRACE_FUNC() \
            NN_ATK_DEBUG_TRACE_FUNC_MSG("")
#else
    #define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) static_cast<void>(0)
    #define NN_ATK_DEBUG_TRACE_FUNC() static_cast<void>(0)
#endif // NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER

namespace
{
    const int OpusFrameHeaderSize = 8;

    // TODO: 現状はframeSize = 20 の場合の値固定になっています。
    // ヘッダの情報などでこの値を変更する必要があるが、今のところ取得できないようなので固定にしています。
    // オーディオフレーム当たりのサンプル数 240 やオーディオフレームの時間も
    // 本来はHardwareManager などから取得して計算するべきだが、ひとまず依存したくないのでここも固定値としています。
    const int OpusFrameSize = 20;    // 20 ms
    const int AudioFrameSize = 5;    // 5 ms
    const int OpusSampleCount = 240 * ( OpusFrameSize / AudioFrameSize );

    // Opus デコーダが処理できる最大チャンネル数
    const int OpusChannelCountMax = 2;

    // １サンプルのバイト数
    const int BytePerSample = sizeof(int16_t);

    // 一度にデコードする数
    const int DecodeBlockCount = 1;

    struct OpusFrameHeaderInfo
    {
        size_t frameHeaderSize;
        size_t frameBodySize;
    };

#if defined(NN_ATK_ENABLE_DEBUG_OPUS_INFO_DUMP)
    void DebugDumpFrameHeader(const uint8_t* frameHeader) NN_NOEXCEPT
    {
        NN_DETAIL_ATK_INFO("[FrameHeader]");
        for (auto i = 0; i < OpusFrameHeaderSize; ++i)
        {
            NN_DETAIL_ATK_INFO("%x ", frameHeader[i]);
        }
        NN_DETAIL_ATK_INFO("\n");
    }
#endif

    size_t GetFrameBodySize(const uint8_t* frameHeader) NN_NOEXCEPT
    {
        size_t frameBodySize = (static_cast<size_t>(frameHeader[0]) << 24) | (static_cast<size_t>(frameHeader[1]) << 16)
                             | (static_cast<size_t>(frameHeader[2]) << 8) | (static_cast<size_t>(frameHeader[3]) << 0);
        return frameBodySize;
    }

    // TODO: 正式対応。今は codec_OpusDecoder.cpp を参考にした仮実装
    bool ReadFrameHeader(nn::atk::detail::fnd::FileStream* pFileStream, void* buffer, OpusFrameHeaderInfo* pOutInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);

        nn::atk::detail::fnd::FndResult result;
        pFileStream->Read(buffer, OpusFrameHeaderSize, &result);
        if (result.IsFailed())
        {
            result.PrintResult();
            return false;
        }

#if defined(NN_ATK_ENABLE_DEBUG_OPUS_INFO_DUMP)
        DebugDumpFrameHeader(static_cast<uint8_t*>(buffer));
#endif

        pOutInfo->frameHeaderSize = OpusFrameHeaderSize;
        pOutInfo->frameBodySize = GetFrameBodySize(static_cast<uint8_t*>(buffer));

        return true;
    }

    bool ReadFrameBody(nn::atk::detail::fnd::FileStream* pFileStream, void* buffer, const OpusFrameHeaderInfo& info) NN_NOEXCEPT
    {
        nn::atk::detail::fnd::FndResult result;
        pFileStream->Read(buffer, info.frameBodySize, &result);
        if (result.IsFailed())
        {
            result.PrintResult();
            return false;
        }

        return true;
    }
}

namespace nn { namespace atk { namespace detail { namespace driver {

    NN_DEFINE_STATIC_CONSTANT(const size_t OpusDecoderImpl::InputBufferSize);
    const size_t OpusDecoderImpl::DecodedBufferSizeMax = OpusSampleCount * OpusChannelCountMax * BytePerSample;

    OpusDecoderImpl::OpusDecoderImpl() NN_NOEXCEPT
        : m_InputBuffer(nullptr)
        , m_DecodedSampleBuffer(nullptr)
        , m_WorkBuffer(nullptr)
        , m_WorkBufferSize(0)
        , m_SampleCount(0)
        , m_pDecoder(nullptr)
    {
    }

    OpusDecoderImpl::~OpusDecoderImpl() NN_NOEXCEPT
    {
        Finalize();
    }

    void OpusDecoderImpl::Initialize( void* buffer, size_t bufferSize, size_t workBufferAlignment, nn::codec::IOpusDecoder* decoder ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(decoder);
        NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, InputBufferSize + DecodedBufferSizeMax);

        nn::util::BytePtr pMemory(buffer);

        m_InputBuffer = pMemory.Get();
        pMemory.Advance(InputBufferSize);

        m_DecodedSampleBuffer = static_cast<int16_t*>(pMemory.Get());
        pMemory.Advance(DecodedBufferSizeMax);

        m_WorkBuffer = reinterpret_cast<void*>(nn::util::align_up(reinterpret_cast<uintptr_t>(pMemory.Get()), workBufferAlignment));
        ptrdiff_t diff = pMemory.Distance(m_WorkBuffer);
        m_WorkBufferSize = bufferSize - InputBufferSize - DecodedBufferSizeMax - diff;
        m_pDecoder = decoder;
    }

    void OpusDecoderImpl::Finalize() NN_NOEXCEPT
    {
        if (m_pDecoder == nullptr)
        {
            return;
        }

        if (m_pDecoder->IsInitialized())
        {
            m_pDecoder->Finalize();
        }

        m_pDecoder = nullptr;
        m_SampleCount = 0;

        m_WorkBufferSize = 0;
        m_WorkBuffer = nullptr;

        m_DecodedSampleBuffer = nullptr;
        m_InputBuffer = nullptr;
    }

    bool OpusDecoderImpl::ReadDataInfo( DataInfo* pOutValue, detail::fnd::FileStream* pFileStream ) NN_NOEXCEPT
    {
        detail::fnd::FndResult result;

        result = pFileStream->Seek( 0, detail::fnd::Stream::SeekOrigin_Begin );
        if (result.IsFailed())
        {
            result.PrintResult();
            return false;
        }

        // OpusBasicInfo の読み込み
        nn::codec::OpusBasicInfo basicInfo;
        pFileStream->Read(m_InputBuffer, sizeof(nn::codec::OpusBasicInfo), &result);
        if (result.IsFailed())
        {
            result.PrintResult();
            NN_ATK_WARNING("read OpusBasicInfo failed.");
            return false;
        }
        std::memcpy(&basicInfo, m_InputBuffer, sizeof(nn::codec::OpusBasicInfo));

        // デコーダの初期化
        if (m_pDecoder->IsInitialized())
        {
            m_pDecoder->Finalize();
        }
        size_t workBufferSize = m_pDecoder->GetWorkBufferSize(basicInfo.sampleRate, basicInfo.channelCount);
        NN_SDK_ASSERT(workBufferSize <= m_WorkBufferSize);
        nn::codec::OpusResult opusResult =
            m_pDecoder->Initialize(basicInfo.sampleRate, basicInfo.channelCount, m_WorkBuffer, workBufferSize);
        if (opusResult != nn::codec::OpusResult_Success)
        {
            NN_ATK_WARNING("OpusDecoder::Initialize failed. [%d]", opusResult);
            return false;
        }

        // OpusInfo ヘッダの読み込み
        nn::codec::OpusInfoHeader infoHeader;
        pFileStream->Read(m_InputBuffer, sizeof(nn::codec::OpusInfoHeader), &result);
        if ( result.IsFailed() )
        {
            result.PrintResult();
            NN_ATK_WARNING("read OpusInfoHeader failed.");
            return false;
        }
        std::memcpy(&infoHeader, m_InputBuffer, sizeof(nn::codec::OpusInfoHeader));
        if (infoHeader.type != nn::codec::OpusInfoType_DataInfo)
        {
            NN_ATK_WARNING("invalid OpusInfoHeader type.");
            return false;
        }

        // サンプルレートによって１フレーム当たりのサンプル数を計算
        int sampleCount = OpusSampleCount;
        if ( basicInfo.sampleRate == 24000 )
        {
            sampleCount /= 2;
        }
        else if ( basicInfo.sampleRate == 16000 )
        {
            sampleCount /= 3;
        }
        else if ( basicInfo.sampleRate == 12000 )
        {
            sampleCount /= 4;
        }
        else if ( basicInfo.sampleRate == 8000 )
        {
            sampleCount /= 6;
        }

        m_SampleCount = sampleCount;

        pOutValue->channelCount = basicInfo.channelCount;
        pOutValue->sampleRate = basicInfo.sampleRate;
        pOutValue->blockSampleCount = sampleCount * DecodeBlockCount;
        pOutValue->blockSize = sampleCount * DecodeBlockCount * BytePerSample;

        return true;
    }

    bool OpusDecoderImpl::Skip( detail::fnd::FileStream* pFileStream ) NN_NOEXCEPT
    {
        for ( auto i = 0; i < DecodeBlockCount; ++i )
        {
            if ( !SingleBlockSkip( pFileStream ) )
            {
                return false;
            }
        }

        return true;
    }

    bool OpusDecoderImpl::Decode(int16_t* pOutDecodedBufferAddresses[], detail::fnd::FileStream* pFileStream, int channelCount, DecodeType decodeType ) NN_NOEXCEPT
    {
        return MultiBlockDecode(pOutDecodedBufferAddresses, pFileStream, channelCount, decodeType);
    }

    void OpusDecoderImpl::ResetDecodeProfile() NN_NOEXCEPT
    {
        m_DecodeProfile.decodeTick = nn::os::Tick( 0 );
        m_DecodeProfile.decodedSampleCount = 0;
        m_DecodeProfile.fsAccessTick = nn::os::Tick( 0 );
        m_DecodeProfile.fsReadSize = 0;
    }

    IStreamDataDecoder::DecodeProfile OpusDecoderImpl::GetDecodeProfile() const NN_NOEXCEPT
    {
        return m_DecodeProfile;
    }

    bool OpusDecoderImpl::MultiBlockDecode( int16_t* pOutDecodedBufferAddresses[], detail::fnd::FileStream* pFileStream, int channelCount, DecodeType decodeType ) NN_NOEXCEPT
    {
        bool isLoop = decodeType == DecodeType_Loop;
        bool isIdling = decodeType == DecodeType_Idling;

        int decodeBlockCount = isLoop ? DecodeBlockCount * 2 : DecodeBlockCount;
        if ( isIdling )
        {
            for ( auto i = 0; i < DecodeBlockCount - 1; ++i )
            {
                if ( !SingleBlockSkip( pFileStream ) )
                {
                    return false;
                }
            }

            SingleBlockDecode(pOutDecodedBufferAddresses, pFileStream, channelCount);
        }
        else
        {
            for ( auto i = 0; i < decodeBlockCount; ++i )
            {
                if ( !SingleBlockDecode( pOutDecodedBufferAddresses, pFileStream, channelCount ) )
                {
                    return false;
                }

                for ( auto channelIndex = 0; channelIndex < channelCount; ++channelIndex )
                {
                    nn::util::BytePtr bytePtr(pOutDecodedBufferAddresses[channelIndex]);
                    bytePtr.Advance(m_SampleCount * BytePerSample);
                    pOutDecodedBufferAddresses[channelIndex] = static_cast<int16_t*>(bytePtr.Get());
                }
            }
        }

        return true;
    }

    bool OpusDecoderImpl::SingleBlockDecode( int16_t* pOutDecodedBufferAddresses[], detail::fnd::FileStream* pFileStream, int channelCount ) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();

        detail::fnd::FndResult result;

        nn::util::BytePtr bufferPtr(m_InputBuffer);

        NN_ATK_DEBUG_TRACE_FUNC_MSG(" - ReadFrameHeader");
        // ヘッダを読んで、1フレーム分のボディサイズを取得
        OpusFrameHeaderInfo info;
        {
            const auto begin = nn::os::GetSystemTick();
            if ( !ReadFrameHeader( pFileStream, bufferPtr.Get(), &info ) )
            {
                NN_ATK_WARNING("read DataHeader failed.");
                return false;
            }
            const auto end = nn::os::GetSystemTick();
            m_DecodeProfile.fsAccessTick += end - begin;
            m_DecodeProfile.fsReadSize += info.frameHeaderSize;
        }
        bufferPtr.Advance(info.frameHeaderSize);

        NN_ATK_DEBUG_TRACE_FUNC_MSG(" - ReadFrameData");
        // ボディ部分の読み込み
        {
            const auto begin = nn::os::GetSystemTick();
            if ( !ReadFrameBody( pFileStream, bufferPtr.Get(), info) )
            {
                NN_ATK_WARNING("read DataBuffer failed.");
                return false;
            }
            const auto end = nn::os::GetSystemTick();
            m_DecodeProfile.fsAccessTick += end - begin;
            m_DecodeProfile.fsReadSize += info.frameBodySize;
        }

        NN_ATK_DEBUG_TRACE_FUNC_MSG(" - DecodeData");
        // 1フレーム分だけデコード結果を取得する
        {
            const auto begin = nn::os::GetSystemTick();

            std::size_t consumed;
            int sampleCount;
            nn::codec::OpusResult opusResult =
                m_pDecoder->DecodeInterleaved( &consumed, &sampleCount, m_DecodedSampleBuffer, DecodedBufferSizeMax, m_InputBuffer, InputBufferSize );
            if ( opusResult != nn::codec::OpusResult_Success )
            {
                NN_ATK_WARNING("decode failed.[%d]", opusResult);
                return false;
            }

            // HACK: インタリーブ関連の処理をしなくてよくなった場合にはこの処理を削除できる
            for ( auto i = 0; i < m_SampleCount; ++i )
            {
                for ( auto channelIndex = 0; channelIndex < channelCount; ++channelIndex )
                {
                    pOutDecodedBufferAddresses[channelIndex][i] = m_DecodedSampleBuffer[i * channelCount + channelIndex];
                }
            }

            // インタリーブされたデータの展開にかかる時間も併せてデコード時間とします
            const auto end = nn::os::GetSystemTick();
            m_DecodeProfile.decodeTick += end - begin;
            m_DecodeProfile.decodedSampleCount += sampleCount;
        }

        return true;
    }

    bool OpusDecoderImpl::SingleBlockSkip( detail::fnd::FileStream* pFileStream ) NN_NOEXCEPT
    {
        OpusFrameHeaderInfo info;
        {
            const auto begin = nn::os::GetSystemTick();
            if (!ReadFrameHeader(pFileStream, m_InputBuffer, &info))
            {
                NN_ATK_WARNING("read DataHeader failed.");
                return false;
            }
            const auto end = nn::os::GetSystemTick();
            m_DecodeProfile.fsAccessTick += end - begin;
            m_DecodeProfile.fsReadSize += info.frameHeaderSize;
        }

        detail::fnd::FndResult result;
        {
            const auto begin = nn::os::GetSystemTick();
            result = pFileStream->Seek(info.frameBodySize, detail::fnd::Stream::SeekOrigin_Current);
            const auto end = nn::os::GetSystemTick();
            m_DecodeProfile.fsAccessTick += end - begin;
        }
        if (result.IsFailed())
        {
            result.PrintResult();
            NN_ATK_WARNING("seek failed.");
            return false;
        }

        return true;
    }

}}}}

