﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/detail/atk_HardwareOpusDecoder.h>
#include <nn/util/util_BytePtr.h>
#include <nn/atk/detail/atk_Macro.h>

//#define NN_ATK_ENABLE_DEBUG_OPUS_INFO_DUMP

//#define NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER
#if defined(NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER)
#define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) \
            NN_DETAIL_ATK_INFO("[%p](%s) %s%s\n", this, NN_ATK_FILENAME, __FUNCTION__, AdditionalMessage)
#define NN_ATK_DEBUG_TRACE_FUNC() \
            NN_ATK_DEBUG_TRACE_FUNC_MSG("")
#else
#define NN_ATK_DEBUG_TRACE_FUNC_MSG(AdditionalMessage) static_cast<void>(0)
#define NN_ATK_DEBUG_TRACE_FUNC() static_cast<void>(0)
#endif // NN_ATK_ENABLE_DEBUG_TRACE_FUNC_OPUS_DECODER

namespace nn { namespace atk {

        detail::driver::HardwareOpusDecoderManager g_HardwareOpusDecoderManager;

        void InitializeHardwareOpusDecoder(void* buffer, size_t bufferSize) NN_NOEXCEPT
        {
            InitializeHardwareOpusDecoder(buffer, bufferSize, detail::driver::HardwareOpusDecoderManager::DefaultOpusDecoderCount);
        }

        void InitializeHardwareOpusDecoder(void* buffer, size_t bufferSize, int decoderCount) NN_NOEXCEPT
        {
            if (!g_HardwareOpusDecoderManager.IsInitialized())
            {
                g_HardwareOpusDecoderManager.Initialize(buffer, bufferSize, decoderCount);
                nn::atk::detail::driver::StreamSoundLoader::RegisterStreamDataDecoderManager(&g_HardwareOpusDecoderManager);
            }
            else
            {
                NN_ATK_WARNING("HardwareOpusDecoder is already initialized.");
            }
        }

        void FinalizeHardwareOpusDecoder() NN_NOEXCEPT
        {
            if (g_HardwareOpusDecoderManager.IsInitialized())
            {
                nn::atk::detail::driver::StreamSoundLoader::UnregisterStreamDataDecoderManager(&g_HardwareOpusDecoderManager);
                g_HardwareOpusDecoderManager.Finalize();
            }
        }

        size_t GetRequiredHardwareOpusDecoderBufferSize() NN_NOEXCEPT
        {
            return GetRequiredHardwareOpusDecoderBufferSize(detail::driver::HardwareOpusDecoderManager::DefaultOpusDecoderCount);
        }

        size_t GetRequiredHardwareOpusDecoderBufferSize(int decoderCount) NN_NOEXCEPT
        {
            return g_HardwareOpusDecoderManager.GetRequiredMemorySize(decoderCount);
        }

}}

namespace nn { namespace atk { namespace detail { namespace driver {

    HardwareOpusDecoder::HardwareOpusDecoder() NN_NOEXCEPT
        : m_Buffer(nullptr)
        , m_Decoder(0)
    {
    }

    HardwareOpusDecoder::~HardwareOpusDecoder() NN_NOEXCEPT
    {
        Finalize();
    }

    void HardwareOpusDecoder::Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        m_Buffer = buffer;
        m_OpusDecoderImpl.Initialize(buffer, bufferSize, nn::os::MemoryPageSize, &m_Decoder);
    }

    void* HardwareOpusDecoder::Finalize() NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        m_OpusDecoderImpl.Finalize();

        void* pWorkBufferHead = m_Buffer;
        m_Buffer = nullptr;

        return pWorkBufferHead;
    }

    bool HardwareOpusDecoder::ReadDataInfo(DataInfo* pOutValue, detail::fnd::FileStream* pFileStream) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        return m_OpusDecoderImpl.ReadDataInfo(pOutValue, pFileStream);
    }

    bool HardwareOpusDecoder::Skip(detail::fnd::FileStream* pFileStream) NN_NOEXCEPT
    {
        return m_OpusDecoderImpl.Skip(pFileStream);
    }

    bool HardwareOpusDecoder::Decode(int16_t* pOutDecodedBufferAddresses[], detail::fnd::FileStream* pFileStream, int channelCount, DecodeType decodeType) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();
        return m_OpusDecoderImpl.Decode(pOutDecodedBufferAddresses, pFileStream, channelCount, decodeType);
    }

    void HardwareOpusDecoder::ResetDecodeProfile() NN_NOEXCEPT
    {
        m_OpusDecoderImpl.ResetDecodeProfile();
    }

    IStreamDataDecoder::DecodeProfile HardwareOpusDecoder::GetDecodeProfile() const NN_NOEXCEPT
    {
        return m_OpusDecoderImpl.GetDecodeProfile();
    }

}}}}

namespace nn { namespace atk { namespace detail { namespace driver {

    NN_DEFINE_STATIC_CONSTANT(const int HardwareOpusDecoderManager::DefaultOpusDecoderCount);

    HardwareOpusDecoderManager::HardwareOpusDecoderManager() NN_NOEXCEPT
    {
    }

    HardwareOpusDecoderManager::~HardwareOpusDecoderManager() NN_NOEXCEPT
    {
    }

    void HardwareOpusDecoderManager::Initialize(void* buffer, size_t bufferSize, int decoderCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(bufferSize, GetRequiredMemorySize(decoderCount));
        m_DecoderManagerImpl.Initialize(buffer, bufferSize, GetWorkBufferSizePerDecoder(), decoderCount, detail::StreamFileType_Opus, detail::DecodeMode_Accelerator );
    }

    void HardwareOpusDecoderManager::Finalize() NN_NOEXCEPT
    {
        m_DecoderManagerImpl.Finalize();
    }

    size_t HardwareOpusDecoderManager::GetRequiredMemorySize(int decoderCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(decoderCount, 0);
        return (sizeof(HardwareOpusDecoder) + GetWorkBufferSizePerDecoder()) * decoderCount;
    }

    bool HardwareOpusDecoderManager::IsInitialized() const NN_NOEXCEPT
    {
        return m_DecoderManagerImpl.IsInitialized();
    }

    IStreamDataDecoder* HardwareOpusDecoderManager::AllocDecoder() NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();

        return m_DecoderManagerImpl.AllocDecoder();
    }

    void HardwareOpusDecoderManager::FreeDecoder(IStreamDataDecoder* pDecoder) NN_NOEXCEPT
    {
        NN_ATK_DEBUG_TRACE_FUNC();

        m_DecoderManagerImpl.FreeDecoder(pDecoder);
    }

    int HardwareOpusDecoderManager::GetStreamFileType() const NN_NOEXCEPT
    {
        return m_DecoderManagerImpl.GetStreamFileType();
    }

    int HardwareOpusDecoderManager::GetDecodeMode() const NN_NOEXCEPT
    {
        return m_DecoderManagerImpl.GetDecodeMode();
    }

    size_t HardwareOpusDecoderManager::GetWorkBufferSizePerDecoder() NN_NOEXCEPT
    {
        size_t size = 0;
        size += nn::atk::detail::driver::OpusDecoderImpl::InputBufferSize;
        size += nn::atk::detail::driver::OpusDecoderImpl::DecodedBufferSizeMax;
        // アラインメントのサイズ分
        size += nn::os::MemoryPageSize;
        // HACK： 2016/02/04 現在一番重い組み合わせ
        int sampleRate = 48000;
        int channelCount = 2;

        const int32_t noOption = 0; // 現在、指定可能なオプションはありません
        nn::codec::HardwareOpusDecoder decoder(noOption);
        size += decoder.GetWorkBufferSize(sampleRate, channelCount);

        return size;
    }

}}}}
