﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_ExternalSoundPlayer.h>

#include <nn/atk/atk_SoundThread.h>
#include <nn/atk/atk_SoundActor.h>

namespace nn {
namespace atk {
namespace detail {

/*--------------------------------------------------------------------------------*
  Name:         ExternalSoundPlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
ExternalSoundPlayer::ExternalSoundPlayer() NN_NOEXCEPT
: m_PlayableCount( 1 )
{
}

/*--------------------------------------------------------------------------------*
  Name:         ExternalSoundPlayer

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
ExternalSoundPlayer::~ExternalSoundPlayer() NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->DetachExternalSoundPlayer( this );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         StopAllSound

  Description:  全てのサウンドを停止する

  Arguments:    fadeFrames - フェードアウトフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void ExternalSoundPlayer::StopAllSound( int fadeFrames ) NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->Stop( fadeFrames );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void ExternalSoundPlayer::PauseAllSound( bool flag, int fadeFrames ) NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->Pause( flag, fadeFrames );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数
                pauseMode  - ポーズの設定

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void ExternalSoundPlayer::PauseAllSound( bool flag, int fadeFrames, PauseMode pauseMode ) NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->Pause( flag, fadeFrames, pauseMode );
    }
}

void ExternalSoundPlayer::Finalize( SoundActor* actor ) NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        auto pActor = curItr->GetSoundActor();
        if (pActor == actor)
        {
            curItr->DetachSoundActor( actor );
            RemoveSound( &*curItr );
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         AppendSound

  Description:  サウンドを登録する

  Arguments:    sound - サウンド

  Returns:      登録できたらtrue
 *--------------------------------------------------------------------------------*/
bool ExternalSoundPlayer::AppendSound( detail::BasicSound* sound ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( sound );

    int allocPriority = sound->CalcCurrentPlayerPriority();

    // 最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 ) return false;
    while ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        detail::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL ) return false;
        if ( allocPriority < dropSound->CalcCurrentPlayerPriority() ) return false;
        dropSound->Finalize();
    }

    m_SoundList.push_back( *sound );

    sound->AttachExternalSoundPlayer( this );

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  同時再生数を設定

  Arguments:    count - 同時再生数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void ExternalSoundPlayer::SetPlayableSoundCount( int count ) NN_NOEXCEPT
{
    m_PlayableCount = count;

    // 新しく設定された同時再生数を越えるサウンドを終了する
    while ( GetPlayingSoundCount() > GetPlayableSoundCount() )
    {
        detail::BasicSound* dropSound = GetLowestPrioritySound();
        NN_SDK_ASSERT_NOT_NULL( dropSound );
        dropSound->Finalize();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         RemoveSound

  Description:  サウンドをプレイヤーリストから削除する

  Arguments:    sound - シーケンスサウンド

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void ExternalSoundPlayer::RemoveSound( detail::BasicSound* sound ) NN_NOEXCEPT
{
    // 再生リストから削除する
    m_SoundList.erase( m_SoundList.iterator_to( *sound ) );
    sound->DetachExternalSoundPlayer( this );
}

/*--------------------------------------------------------------------------------*
  Name:         CanPlaySound

  Description:  指定したプライオリティのサウンドを再生できるかどうかを調べる

  Arguments:    startPriority - プライオリティ

  Returns:      再生可能ならtrue
 *--------------------------------------------------------------------------------*/
bool ExternalSoundPlayer::CanPlaySound( int startPriority ) NN_NOEXCEPT
{
    // 最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 )
    {
        return false;
    }
    if ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        detail::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL )
        {
            return false;
        }
        if ( startPriority < dropSound->CalcCurrentPlayerPriority() )
        {
            return false;
        }
    }

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         GetLowestPrioritySound

  Description:  サウンドリストからもっともプライオリティの低いサウンドを探す

  Arguments:    None.

  Returns:      プライオリティの低いサウンドへのポインタ
 *--------------------------------------------------------------------------------*/
detail::BasicSound* ExternalSoundPlayer::GetLowestPrioritySound() NN_NOEXCEPT
{
    // insert to priority list
    int priority = nn::atk::PlayerPriorityMax + 1;
    detail::BasicSound* sound = NULL;
    SoundList::iterator itr = m_SoundList.begin();
    while ( itr != m_SoundList.end() )
    {
        int itrPriority = itr->CalcCurrentPlayerPriority();
        if ( priority > itrPriority )
        {
            sound = &*itr;
            priority = itrPriority;
        }
        (void)++itr;
    }
    return sound;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

