﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_AnimEventPlayer.h>
#include <nn/atk/atk_SequenceSoundHandle.h>
#include <nn/atk/atk_SoundStartable.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{

int16_t ConvertSpeedF32ToS16( float speed ) NN_NOEXCEPT
{
    static const int32_t S16SpeedMin = -32768;
    static const int32_t S16SpeedMax = 32767;
    int32_t int32Speed = static_cast<int32_t>( speed * 100 );
    int16_t int16Speed = static_cast<int16_t>( nn::atk::detail::fnd::Clamp( int32Speed, S16SpeedMin, S16SpeedMax ) );
    return int16Speed;
}

} // anonymous namespace

AnimEventPlayer::AnimEventPlayer() NN_NOEXCEPT
: m_pEventInfo( NULL )
, m_IsStopWhenFinalize( false )
{
}

AnimEventPlayer::~AnimEventPlayer() NN_NOEXCEPT
{
    Finalize();
}

void AnimEventPlayer::InitParam(
        const AnimSoundFile::EventInfo& eventInfo, bool isStopWhenFinalize ) NN_NOEXCEPT
{
    // 初期パラメータ設定
    m_Handle.SetVolume( static_cast<float>( eventInfo.volume ) / 127.0f );

    if ( eventInfo.pitch != 1.0f )
    {
        m_Handle.SetPitch( eventInfo.pitch );
    }
    m_pEventInfo = &eventInfo;
    m_IsStopWhenFinalize =
        isStopWhenFinalize && (! eventInfo.IsNotStopSoundWhenAnimationFinish());
}

void AnimEventPlayer::Finalize() NN_NOEXCEPT
{
    // 再生中のサウンドを停止する
    if ( m_Handle.IsAttachedSound() && m_IsStopWhenFinalize )
    {
        m_Handle.Stop( 0 );
    }
}

bool AnimEventPlayer::StartEvent(
    const AnimSoundFile::EventInfo& eventInfo,
    SoundStartable& starter,
    bool isStopWhenFinalize
) NN_NOEXCEPT
{
    // 再生
    if ( eventInfo.placeForSoundId == SoundArchive::InvalidId )
    {
        const char* soundLabel = eventInfo.GetSoundLabel();
        if ( ! starter.StartSound( &m_Handle, soundLabel ).IsSuccess() )
        {
            return false;
        }
    }
    else
    {
        if ( ! starter.StartSound( &m_Handle, eventInfo.placeForSoundId ).IsSuccess() )
        {
            return false;
        }
    }

    InitParam( eventInfo, isStopWhenFinalize );
    return true;
}

bool AnimEventPlayer::HoldEvent(
    const AnimSoundFile::EventInfo& eventInfo,
    SoundStartable& starter,
    bool isStopWhenFinalize
) NN_NOEXCEPT
{
    // 再生
    if ( eventInfo.placeForSoundId == SoundArchive::InvalidId )
    {
        const char* soundLabel = eventInfo.GetSoundLabel();
        if ( ! starter.HoldSound( &m_Handle, soundLabel ).IsSuccess() )
        {
            return false;
        }
    }
    else
    {
        if ( ! starter.HoldSound( &m_Handle, eventInfo.placeForSoundId ).IsSuccess() )
        {
            return false;
        }
    }

    // HoldSoundの自動停止をやめる
    m_Handle.detail_GetAttachedSound()->SetAutoStopCounter( 0 );

    InitParam( eventInfo, isStopWhenFinalize );
    return true;
}

void AnimEventPlayer::ForceStop() NN_NOEXCEPT
{
    m_Handle.Stop( 0 );
    m_pEventInfo = NULL;
}

void AnimEventPlayer::UpdateFrame() NN_NOEXCEPT
{
    // 自動停止をチェック
    if ( ! m_Handle.IsAttachedSound() )
    {
        m_pEventInfo = NULL;
    }
}

void AnimEventPlayer::WritePlaySpeedToSequenceVariable( uint8_t sequenceVariableNo, float speed ) NN_NOEXCEPT
{
    // ローカル変数反映
    if ( sequenceVariableNo < 16 )
    {
        if ( ! m_Handle.IsAttachedSound() )
        {
            return;
        }
        SequenceSoundHandle handle( &m_Handle );
        handle.WriteVariable( sequenceVariableNo, ConvertSpeedF32ToS16( speed ) );
    }
    // グローバル変数反映
    else if ( sequenceVariableNo < 32 )
    {
        SequenceSoundHandle::WriteGlobalVariable(
                sequenceVariableNo - 16,
                ConvertSpeedF32ToS16( speed ) );
    }
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

