﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundDataManager.h>

#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_SoundArchiveLoader.h>
#include <nn/atk/atk_GroupFileReader.h>
#include <nn/atk/atk_WaveArchiveFile.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_WaveArchiveFileReader.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundDataManager::BufferAlignSize );

SoundDataManager::SoundDataManager() NN_NOEXCEPT
: m_pFileTable ( NULL )
, m_pFileManager( NULL )
{
}

SoundDataManager::~SoundDataManager() NN_NOEXCEPT
{
}

size_t SoundDataManager::GetRequiredMemSize( const SoundArchive* arc ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arc );
    size_t size = 0;

    size += nn::util::align_up(
            sizeof( uint32_t ) +
            sizeof( FileAddress ) * arc->detail_GetFileCount(), BufferAlignSize );
    return size;
}

bool SoundDataManager::Initialize( const SoundArchive* pArchive, void* buffer, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pArchive );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES_ALIGNED( buffer, BufferAlignSize ); // BufferAlignSize バイト境界でないと CreateFileAddressTable が失敗する
    NN_SDK_REQUIRES_GREATER_EQUAL( size, GetRequiredMemSize( pArchive ) );

    void* endp = static_cast<char*>(buffer) + size;
    void* buf = buffer;

    if ( ! CreateTables( &buf, pArchive, endp ) )
    {
        return false;
    }

    NN_SDK_ASSERT( static_cast<char*>(buf) - static_cast<char*>(buffer) == static_cast<int32_t>(GetRequiredMemSize( pArchive )) );
    SetSoundArchive( pArchive );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();

    detail::DriverCommandDisposeCallback* command =
        cmdmgr.AllocCommand<detail::DriverCommandDisposeCallback>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_RegistDisposeCallback;
    command->callback = this;
    cmdmgr.PushCommand(command);

    return true;
}

void SoundDataManager::Finalize() NN_NOEXCEPT
{
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();

    detail::DriverCommandDisposeCallback* command =
        cmdmgr.AllocCommand<detail::DriverCommandDisposeCallback>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_UnregistDisposeCallback;
    command->callback = this;
    cmdmgr.PushCommand(command);
    uint32_t tag = cmdmgr.FlushCommand( true );
    cmdmgr.WaitCommandReply( tag );

    m_pFileManager = NULL;
    m_pFileTable = NULL;
}

bool SoundDataManager::CreateTables(
    void** pOutBuffer,
    const SoundArchive* pArchive,
    void* endAddress
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutBuffer );
    NN_SDK_ASSERT_NOT_NULL( pArchive );
    NN_SDK_ASSERT_NOT_NULL( endAddress );

    size_t fileTableSize =
        sizeof( uint32_t ) +
        sizeof( FileAddress ) * pArchive->detail_GetFileCount();

    void* ep = util::BytePtr( util::BytePtr( *pOutBuffer, fileTableSize ).Get() ).AlignUp( BufferAlignSize ).Get();
    if ( util::ConstBytePtr(endAddress).Distance(ep) > 0 )
    {
        return false;
    }
    m_pFileTable = reinterpret_cast< FileTable* >( *pOutBuffer );
    *pOutBuffer = ep;


    m_pFileTable->count = pArchive->detail_GetFileCount();
    for( uint32_t i = 0 ; i < m_pFileTable->count ; i++ )
    {
        m_pFileTable->item[ i ].address = NULL;
    }

    return true;
}


/*--------------------------------------------------------------------------------*
  Name:         InvalidateData

  Description:  ロードされたデータが破棄されたときにテーブルから削除する

  Arguments:    start - 開始アドレス
                end   - 終了アドレス

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundDataManager::InvalidateData( const void* start, const void* end ) NN_NOEXCEPT
{
    // ファイルアドレステーブルから破棄
    if ( m_pFileTable != NULL )
    {
        for( uint32_t i = 0 ; i < m_pFileTable->count ; i++ )
        {
            const void* addr = m_pFileTable->item[ i ].address;
            if ( start <= addr && addr <= end )
            {
                m_pFileTable->item[ i ].address = NULL;
            }
        }
    }

    // まだロードされている「波形アーカイブテーブル」上のから、
    // 波形ファイルアドレスを削除
    const SoundArchive* arc = GetSoundArchive();
    if ( arc )
    {
        uint32_t waveArchiveCount = arc->GetWaveArchiveCount();
        for ( uint32_t i = 0; i < waveArchiveCount; i++ )
        {
            SoundArchive::WaveArchiveInfo info;
            if ( arc->ReadWaveArchiveInfo(
                        SoundArchive::GetWaveArchiveIdFromIndex( i ), &info ) )
            {
                if ( info.isLoadIndividual )
                {
                    const void* pWarcTable = GetFileAddressFromTable( info.fileId );
                    if ( pWarcTable != NULL )
                    {
                        detail::WaveArchiveFileReader reader( pWarcTable, true );
                        for ( uint32_t j = 0; j < info.waveCount; j++ )
                        {
                            const void* waveAddr = reader.GetWaveFile(j);
                            if ( waveAddr && ( start <= waveAddr && waveAddr <= end ) )
                            {
                                reader.SetWaveFile(j, NULL);
                            }
                        }
                    }
                }
            }
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         detail_GetFileAddress

  Description:  記憶させたファイルアドレスを取得

  Arguments:    fileId - ファイルID

  Returns:      ファイルアドレス
 *--------------------------------------------------------------------------------*/
const void* SoundDataManager::detail_GetFileAddress( SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    return GetFileAddressImpl( fileId );
}

const void* SoundDataManager::GetFileAddressImpl( SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    // 外部のファイルマネージャに問い合わせ
    if ( m_pFileManager != NULL )
    {
        const void* addr = m_pFileManager->GetFileAddress( fileId );
        if ( addr != NULL ) return addr;
    }

    // サウンドアーカイブに問い合わせ
    {
        const void* addr = GetFileAddressFromSoundArchive( fileId );
        if ( addr != NULL ) return addr;
    }

    // ファイルアドレステーブルを参照
    {
        const void* fileData = GetFileAddressFromTable( fileId );
        if ( fileData != NULL ) return fileData;
    }
    return NULL;
}

const void* SoundDataManager::SetFileAddressToTable(
        SoundArchive::FileId fileId,
        const void* address ) NN_NOEXCEPT
{
    if ( m_pFileTable == NULL )
    {
        NN_ATK_WARNING("Failed to SoundDataManager::SetFileAddress because file table is not allocated.");
        return NULL;
    }

    NN_SDK_ASSERT( fileId <= m_pFileTable->count - 1 );

    const void* preAddress = m_pFileTable->item[ fileId ].address;
    m_pFileTable->item[ fileId ].address = address;
    return preAddress;
}


/*--------------------------------------------------------------------------------*
  Name:         GetFileAddressFromTable

  Description:  記憶させたファイルアドレスを取得

  Arguments:    fileId - ファイルID

  Returns:      ファイルアドレス
 *--------------------------------------------------------------------------------*/
const void* SoundDataManager::GetFileAddressFromTable( SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    if ( m_pFileTable == NULL )
    {
        NN_ATK_WARNING("Failed to SoundDataManager::GetFileAddress because file table is not allocated.");
        return NULL;
    }

    if ( fileId >= m_pFileTable->count ) return NULL;

    return m_pFileTable->item[ fileId ].address;
}

/*--------------------------------------------------------------------------------*
  Name:         SetFileAddress

  Description:  ファイルアドレスを記憶

  Arguments:    fileId - ファイル番号
                address - ファイルアドレス

  Returns:      セットする前に書かれていたファイルアドレス
 *--------------------------------------------------------------------------------*/
const void*
SoundDataManager::SetFileAddress( SoundArchive::FileId fileId, const void* address ) NN_NOEXCEPT
{
    return SetFileAddressToTable( fileId, address );
}

// ロード済みファイルテーブルの中から、
// 引数と一致するアドレスを持つファイルのファイル ID を取得する
SoundArchive::FileId SoundDataManager::detail_GetFileIdFromTable( const void* address ) const NN_NOEXCEPT
{
    for ( uint32_t i = 0; i < m_pFileTable->count; i++ )
    {
        if ( address == m_pFileTable->item[i].address )
        {
            return i;
        }
    }
    return SoundArchive::InvalidId;
}

// Clear 関数との対称性のため、size 引数をつけてあるが、現状は使わない
bool SoundDataManager::SetFileAddressInGroupFile( const void* address, size_t /*size*/ ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(address);
    if ( address == NULL )
    {
        return false;
    }

    // 以下の実装は、SoundArchiveLoader::PostProcessForLoadedGroupFile と似ている
    detail::GroupFileReader reader(address);
    uint32_t groupItemCount = reader.GetGroupItemCount();
    for ( uint32_t i = 0; i < groupItemCount; i++ )
    {
        detail::GroupItemLocationInfo info;
        if ( ! reader.ReadGroupItemLocationInfo( &info, i ) )
        {
            return false;
        }

        if ( info.address == NULL )
        {
            return false;
        }
        SetFileAddressToTable( info.fileId, info.address );
    }

    return true;
}

void SoundDataManager::ClearFileAddressInGroupFile( const void* address, size_t size ) NN_NOEXCEPT
{
    InvalidateSoundData(address, size);
}

void SoundDataManager::InvalidateSoundData( const void* address, size_t size ) NN_NOEXCEPT
{
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();

    if ( cmdmgr.IsAvailable() ) {
        detail::DriverCommandInvalidateData* command =
            cmdmgr.AllocCommand<detail::DriverCommandInvalidateData>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = detail::DriverCommandId_InvalidateData;
        command->mem = address;
        command->size = size;
        cmdmgr.PushCommand(command);

        uint32_t tag = cmdmgr.FlushCommand( true );
        cmdmgr.WaitCommandReply( tag );
    }
}

} // namespace nn::atk
} // namespace nn

