﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundActor.h>
#include <nn/atk/atk_SoundArchivePlayer.h>

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const int SoundActor::ActorPlayerCount );

/*--------------------------------------------------------------------------------*
  Name:         SoundActor

  Description:  コンストラクタ

  Arguments:    soundArchivePlayer - サウンドアーカイブプレイヤー

  Returns:      None.
  *--------------------------------------------------------------------------------*/
SoundActor::SoundActor() NN_NOEXCEPT
: m_pSoundArchivePlayer( NULL )
, m_IsInitialized( false )
, m_IsFinalized( true )
{
    // 何もしない
}
SoundActor::SoundActor( SoundArchivePlayer* pSoundArchivePlayer ) NN_NOEXCEPT
: m_pSoundArchivePlayer( NULL )
, m_IsInitialized( false )
, m_IsFinalized( true )
{
    Initialize( pSoundArchivePlayer );
}

SoundActor::SoundActor( SoundArchivePlayer* pSoundArchivePlayer, ActorPlayer* pActorPlayer ) NN_NOEXCEPT
: m_pSoundArchivePlayer( NULL )
, m_IsInitialized( false )
, m_IsFinalized( true )
{
    Initialize( pSoundArchivePlayer, pActorPlayer );
}

/*--------------------------------------------------------------------------------*
  Name:         ~SoundActor

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
SoundActor::~SoundActor() NN_NOEXCEPT
{
    Finalize();
}

void SoundActor::Initialize( SoundArchivePlayer* pSoundArchivePlayer ) NN_NOEXCEPT
{
    Initialize( pSoundArchivePlayer, nullptr );
}

void SoundActor::Initialize( SoundArchivePlayer* pSoundArchivePlayer, ActorPlayer* pActorPlayer ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSoundArchivePlayer);

    if ( m_IsInitialized ) return;

    m_pSoundArchivePlayer = pSoundArchivePlayer;

    for ( int i = 0; i < ActorPlayerCount; i++ )
    {
        const int playableSoundCount = ( i == 0 ) ? INT_MAX : 1;
        if ( i == 0 && pActorPlayer != NULL )
        {
            m_pActorPlayer[i] = pActorPlayer;
        }
        else
        {
            m_pActorPlayer[i] = &m_ActorPlayer[i];
            m_pActorPlayer[i]->SetPlayableSoundCount( playableSoundCount );
        }
    }

    m_IsInitialized = true;
    m_IsFinalized = false;
}
void SoundActor::Finalize() NN_NOEXCEPT
{
    if ( m_IsFinalized ) return;

    for ( int i = 0; i < ActorPlayerCount; i++ )
    {
        NN_SDK_ASSERT_NOT_NULL( m_pActorPlayer[i] );
        m_pActorPlayer[i]->Finalize( this );
        m_pActorPlayer[i] = NULL;
    }

    m_IsFinalized = true;
    m_IsInitialized = false;
    m_pSoundArchivePlayer = NULL;
    m_ActorParam.Reset();
}

/*--------------------------------------------------------------------------------*
  Name:         StopAllSound

  Description:  全てのサウンドを停止する

  Arguments:    fadeFrames - フェードアウトフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundActor::StopAllSound( int fadeFrames ) NN_NOEXCEPT
{
    for ( int i = 0; i < ActorPlayerCount; i++ )
    {
        NN_SDK_ASSERT_NOT_NULL( m_pActorPlayer[i] );
        m_pActorPlayer[ i ]->StopAllSound( fadeFrames );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundActor::PauseAllSound( bool flag, int fadeFrames ) NN_NOEXCEPT
{
    for ( int i = 0; i < ActorPlayerCount; i++ )
    {
        NN_SDK_ASSERT_NOT_NULL( m_pActorPlayer[i] );
        m_pActorPlayer[ i ]->PauseAllSound( flag, fadeFrames );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数
                pauseMode  - ポーズの設定

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundActor::PauseAllSound( bool flag, int fadeFrames, PauseMode pauseMode ) NN_NOEXCEPT
{
    for ( int i = 0; i < ActorPlayerCount; i++ )
    {
        NN_SDK_ASSERT_NOT_NULL( m_pActorPlayer[i] );
        m_pActorPlayer[ i ]->PauseAllSound( flag, fadeFrames, pauseMode );
    }
}
/*--------------------------------------------------------------------------------*
  Name:         GetPlayingSoundCount

  Description:  現在再生中のサウンド数を取得する

  Arguments:    actorPlayerId - アクタープレイヤーＩＤ

  Returns:      なし
 *--------------------------------------------------------------------------------*/
int SoundActor::GetPlayingSoundCount( int actorPlayerId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( actorPlayerId, 0, ActorPlayerCount );

    NN_SDK_ASSERT_NOT_NULL( m_pActorPlayer[ actorPlayerId ] );
    return m_pActorPlayer[ actorPlayerId ]->GetPlayingSoundCount();
}

/*--------------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  サウンドの最大同時再生数を設定する

  Arguments:    actorPlayerId - アクタープレイヤーＩＤ
                count - サウンドの最大同時再生数

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundActor::SetPlayableSoundCount( int actorPlayerId, int count ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( actorPlayerId, 0, ActorPlayerCount );

    NN_SDK_ASSERT_NOT_NULL( m_pActorPlayer[ actorPlayerId ] );
    m_pActorPlayer[ actorPlayerId ]->SetPlayableSoundCount( count );
}

/*--------------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  サウンドの最大同時再生数を取得する

  Arguments:    actorPlayerId - アクタープレイヤーＩＤ

  Returns:      サウンドの最大同時再生数を返す
  *--------------------------------------------------------------------------------*/
int SoundActor::GetPlayableSoundCount( int actorPlayerId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( actorPlayerId, 0, ActorPlayerCount );

    NN_SDK_ASSERT_NOT_NULL( m_pActorPlayer[ actorPlayerId ] );
    return m_pActorPlayer[ actorPlayerId ]->GetPlayableSoundCount();
}

/*--------------------------------------------------------------------------------*
  Name:         SetupSound [virtual]

  Description:  再生の実装関数

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                startInfo - サウンド再生パラメータ
                setupArg - セットアップ引数

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundActor::SetupSound(
    SoundHandle* handle,
    uint32_t soundId,
    const SoundStartable::StartInfo* startInfo,
    void* setupArg
) NN_NOEXCEPT
{
    return SoundActor::SetupSound(
        handle,
        soundId,
        nullptr,
        startInfo,
        setupArg
    );
}

SoundStartable::StartResult SoundActor::SetupSound(
    SoundHandle* handle,
    uint32_t soundId,
    const char* soundArchiveName,
    const SoundStartable::StartInfo* startInfo,
    void* setupArg
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( setupArg );
    if ( m_IsInitialized == false )
    {
        return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorActorNotInitialized);
    }

    const SetupInfo* setupInfo = reinterpret_cast<SetupInfo*>(setupArg);

    return m_pSoundArchivePlayer->detail_SetupSoundImpl(
        handle,
        soundId,
        NULL,
        this,
        setupInfo->holdFlag,
        soundArchiveName,
        startInfo
    );
}

// AmbientInfoが公開された際には上の関数に統合される予定
// AmbientInfoはStartInfoのメンバで渡される予定
SoundStartable::StartResult SoundActor::detail_SetupSoundWithAmbientInfo(
    SoundHandle* handle,
    uint32_t soundId,
    const char* soundArchiveName,
    const SoundStartable::StartInfo* startInfo,
    detail::BasicSound::AmbientInfo* ambientInfo,
    void* setupArg
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( setupArg );
    if ( m_IsInitialized == false )
    {
        return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_ErrorActorNotInitialized);
    }

    const SetupInfo* setupInfo = reinterpret_cast<SetupInfo*>(setupArg);

    return m_pSoundArchivePlayer->detail_SetupSoundImpl(
        handle,
        soundId,
        ambientInfo,
        this,
        setupInfo->holdFlag,
        soundArchiveName,
        startInfo
    );
}

/*--------------------------------------------------------------------------------*
  Name:         detail_SetupSound [override]

  Description:  SoundStartableインターフェイスの実装

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                holdFlag - ホールドサウンドフラグ
                startInfo - サウンド再生パラメータ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundActor::detail_SetupSound(
    SoundHandle* handle,
    uint32_t soundId,
    bool holdFlag,
    const char* soundArchiveName,
    const SoundStartable::StartInfo* startInfo
) NN_NOEXCEPT
{
    SetupInfo setupInfo;
    setupInfo.holdFlag = holdFlag;

    if (soundArchiveName == nullptr)
    {
        return SetupSound(
            handle,
            soundId,
            startInfo,
            &setupInfo
        );
    }
    else
    {
        return SetupSound(
            handle,
            soundId,
            soundArchiveName,
            startInfo,
            &setupInfo
        );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         detail_GetItemId [override]

  Description:  ラベル文字列からサウンドＩＤへの変換

  Arguments:    label - ラベル文字列

  Returns:      サウンドＩＤを返す
  *--------------------------------------------------------------------------------*/
SoundArchive::ItemId SoundActor::detail_GetItemId( const char* pString ) NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return SoundArchive::InvalidId;
    }
    return m_pSoundArchivePlayer->detail_GetItemId( pString );
}

SoundArchive::ItemId SoundActor::detail_GetItemId( const char* pString, const char* soundArchiveName ) NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return SoundArchive::InvalidId;
    }

    return m_pSoundArchivePlayer->detail_GetItemId( pString, soundArchiveName );
}

} // namespace nn::atk
} // namespace nn

