﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/vfx/vfx_TargetDef.h>

namespace nn {
namespace vfx {
namespace detail {

static const float InverseMaxValue_Float = ( 1.f / 4294967296.f );      //!< 最大値の逆数の float型: 4294967296 == 0x100000000
static const double InverseMaxValue_Double = ( 1.0 / 4294967296.0 );    //!< 最大値の逆数の double型: 4294967296 == 0x100000000

//---------------------------------------------------
//! @brief  乱数発生クラス
//---------------------------------------------------
class RandomGenerator
{
    NN_DISALLOW_COPY( RandomGenerator );
public:
    //----------------------------------------------------------------
    //! @name       コンストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief      引数なしのコンストラクタです。
    //!
    //!             内部でinit()を呼び、乱数生成器を初期化します。
    //---------------------------------------------------------------------------
    RandomGenerator() NN_NOEXCEPT
    {
        Initialize();
    }

    //---------------------------------------------------------------------------
    //! @brief      4バイトの数値をシードとして与えるコンストラクタです。
    //!
    //!             内部でinit(seed)を呼び、乱数生成器を初期化します。
    //!             同じシードを与えたRandomクラスは、同じ乱数列を生成します。
    //!
    //! @param[in]  seed        乱数のシード。
    //---------------------------------------------------------------------------
    explicit RandomGenerator( uint32_t seed ) NN_NOEXCEPT
    {
        Initialize( seed );
    }

    //---------------------------------------------------------------------------
    //! @brief      4バイトの数値を4つシードとして与えるコンストラクタです。
    //!
    //!             内部でinit(seed_0, seed_1, seed_2, seed_3)を呼び、乱数生成器を初期化します。
    //!             同じシードを与えたRandomクラスは、同じ乱数列を生成します。
    //!             全て0の値を与えることはできませんのでご注意ください。アサートで止まります。
    //!             製品版でアサート抜けしたときは、代替の値としてinit(0)で初期化します。
    //!
    //! @param[in]  seed_0      乱数のシード。
    //! @param[in]  seed_1      乱数のシード。
    //! @param[in]  seed_2      乱数のシード。
    //! @param[in]  seed_3      乱数のシード。
    //---------------------------------------------------------------------------
    RandomGenerator( uint32_t seed_0, uint32_t seed_1, uint32_t seed_2, uint32_t seed_3 ) NN_NOEXCEPT
    {
        Initialize( seed_0, seed_1, seed_2, seed_3 );
    }

    //@}

    //----------------------------------------------------------------
    //! @name       初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief      初期化。
    //!
    //!             現状はシード 0 で、乱数生成器を初期化します。
    //---------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT
    {
        Initialize( 0 );
    }

    //---------------------------------------------------------------------------
    //! @brief      引数として与えたシードで乱数生成器を初期化します。
    //!
    //!             XorShiftは初期化するために16バイトの数値を必要としますので、この形式では、
    //!             与えられた4バイトのシードを拡張することで初期化します。
    //!             シードの値は[0-0xffffffff]の任意の値で問題ありませんが、ワンパターンすぎる
    //!             シード（例えば、0,1,2,...を順番に与えるなど）は避けてください。
    //!
    //! @param[in]  seed        乱数のシード。
    //---------------------------------------------------------------------------
    void Initialize( uint32_t seed ) NN_NOEXCEPT
    {
        /**
         *  XorShiftはシードとして4つのuint32_tを必要とする。
         *  この数値は全て0以外であれば何でも問題ないが、シードの決め方
         *  によっては、特に最初に生成する値が一様に分布しないことがある。
         *
         *  ここでは、シードとして0,1,2,...のような線形なパターンの値を
         *  与えたときでも、最初に生成する値がある程度一様に分布するような
         *  拡張の仕方を選んだ。
         *  もちろん、もっとランダムにシードを選ぶのが望ましい。
         */
        static const uint32_t A = 1812433253;
        m_X = ( seed ^ ( seed >> 30 ) ) * A + 1;
        m_Y = ( m_X ^ ( m_X >> 30 ) ) * A + 2;
        m_Z = ( m_Y ^ ( m_Y >> 30 ) ) * A + 3;
        m_W = ( m_Z ^ ( m_Z >> 30 ) ) * A + 4;
    }

    //---------------------------------------------------------------------------
    //! @brief      引数として与えたシードで乱数生成器を初期化します。
    //!
    //!             この形式では、初期化するために必要な16バイトの数値を直接与えます。
    //!             全て0の値を与えることはできませんのでご注意ください。アサートで止まります。
    //!             製品版でアサート抜けしたときは、代替の値としてinit(0)で初期化します。
    //!
    //! @param[in]  seed_0      乱数のシード。
    //! @param[in]  seed_1      乱数のシード。
    //! @param[in]  seed_2      乱数のシード。
    //! @param[in]  seed_3      乱数のシード。
    //---------------------------------------------------------------------------
    void Initialize( uint32_t seed_0, uint32_t seed_1, uint32_t seed_2, uint32_t seed_3 ) NN_NOEXCEPT
    {
        if( seed_0 == 0 && seed_1 == 0 && seed_2 == 0 && seed_3 == 0 )
        {
            //OutputError( "Seeds must not be all zero." );
            // 全て0は使用できない。アサート抜けしたときは別の決まった値で初期化する。
            Initialize( 0 );
        }
        else
        {
            m_X = seed_0;
            m_Y = seed_1;
            m_Z = seed_2;
            m_W = seed_3;
        }
    }
    //@}

    //----------------------------------------------------------------
    //! @name       乱数の生成
    //----------------------------------------------------------------
    //@{

    //---------------------------------------------------------------------------
    //! @brief      乱数をuint32_tの範囲で生成し、返します。
    //!
    //!             記号で表すと [0, 0xffffffff] の整数を返します。
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    uint32_t GetUnsignedInteger() NN_NOEXCEPT
    {
        const uint32_t t = ( m_X ^ ( m_X << 11 ) );
        m_X = m_Y;
        m_Y = m_Z;
        m_Z = m_W;
        m_W = ( m_W ^ ( m_W >> 19 ) ) ^ ( t ^ ( t >> 8 ) );
        return m_W;
    }

    //---------------------------------------------------------------------------
    //! @brief      0以上でceilより小さい整数の乱数を生成し、返します。
    //!
    //!             記号で表すと [0, ceil) の整数を返します。
    //!             ceilが0か1のときは、必ず0を返します。
    //!
    //! @param[in]  ceil    取得する乱数の天井値。
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    uint32_t GetUnsignedInteger( uint32_t ceil ) NN_NOEXCEPT
    {
        return static_cast< uint32_t >( ( static_cast< uint64_t >( GetUnsignedInteger() ) * ceil ) >> 32 );
    }

    //---------------------------------------------------------------------------
    //! @brief      a以上でbより小さい整数の乱数を生成し、返します。
    //!
    //!             記号で表すと [a, b) の整数を返します。
    //!             bはaより大きい値である必要があります。
    //!             b==aのときは、例外的にaを返します。
    //!
    //! @param[in]  a       取得する乱数の最小値
    //! @param[in]  b       取得する乱数の天井値
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    int GetIntegerWithRange( int a, int b ) NN_NOEXCEPT
    {
        //        NN_SDK_ASSERT(b >= a);
        return ( GetUnsignedInteger( b - a ) + a );
    }

    //---------------------------------------------------------------------------
    //! @brief      0以上で1より小さい実数を生成し、float型で返します。
    //!
    //!             記号で表すと [0, 1) の実数を返します。
    //!             ceilが0か1のときは、必ず0を返します。
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    float GetFloat() NN_NOEXCEPT
    {
        // 4294967296 == 0x100000000 である。コンパイルした時点で(1.f / 4294967296.f)が計算されるように、リテラルとして書いている。
        //return GetU32() * VFX_RANDOM_MAX_INVERSE_F32;
        // [1, 2)の単精度浮動小数点を作成し、1を引いて返す
        uint32_t float_binary = ( GetUnsignedInteger() >> 9 ) | 0x3f800000;
        return ( *( reinterpret_cast< float* >( &float_binary ) ) ) - 1.f;
    }

    //---------------------------------------------------------------------------
    //! @brief      0以上で1より小さい実数を生成し、float型で返します。
    //!
    //!             記号で表すと [0, ceil) の実数を返します。
    //!             ceilが0のときは、必ず0を返します。
    //!
    //! @param[in]  ceil    取得する乱数の天井値
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    float GetFloat( float ceil ) NN_NOEXCEPT
    {
        return GetFloat() * ceil;
    }

    //---------------------------------------------------------------------------
    //! @brief      a以上でbより小さい実数の乱数を生成し、float型で返します。
    //!
    //!             記号で表すと [a, b) の実数を返します。
    //!             bはaより大きい値である必要があります。
    //!             b==aのときは、例外的にaを返します。
    //!
    //! @param[in]  a       取得する乱数の最小値
    //! @param[in]  b       取得する乱数の天井値
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    float GetFloatWithRange( float a, float b ) NN_NOEXCEPT
    {
        return ( GetFloat( b - a ) + a );
    }

    //---------------------------------------------------------------------------
    //! @brief      0以上で1より小さい実数を生成し、double型で返します。
    //!
    //!             記号で表すと [0, 1) の実数を返します。
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    double GetDouble() NN_NOEXCEPT
    {
        // 4294967296 == 0x100000000 である。コンパイルした時点で(1.0 / 4294967296.0)が計算されるように、リテラルとして書いている。
        //return GetU32() * VFX_RANDOM_MAX_INVERSE_F64;
        // [1, 2)の倍精度浮動小数点を作成し、1を引いて返す
        uint64_t double_binary = ( static_cast< uint64_t >( GetUnsignedInteger() ) << 20 ) | 0x3ff0000000000000LL;
        return ( *( reinterpret_cast< double* >( &double_binary ) ) ) - 1.0;
    }

    //---------------------------------------------------------------------------
    //! @brief      0以上でceilより小さい実数を生成し、double型で返します。
    //!
    //!             記号で表すと [0, ceil) の実数を返します。
    //!             ceilが0のときは、必ず0を返します。
    //!
    //! @param[in]  ceil    取得する乱数の天井値
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    double GetDouble( double ceil ) NN_NOEXCEPT
    {
        return GetDouble() * ceil;
    }

    //---------------------------------------------------------------------------
    //! @brief      a以上でbより小さい実数の乱数を生成し、double型で返します。
    //!
    //!             記号で表すと [a, b) の実数を返します。
    //!             bはaより大きい値である必要があります。
    //!             b==aのときは、例外的にaを返します。
    //!
    //! @param[in]  a       取得する乱数の最小値
    //! @param[in]  b       取得する乱数の天井値
    //!
    //! @return     乱数。
    //---------------------------------------------------------------------------
    double GetDoubleWithRange( double a, double b ) NN_NOEXCEPT
    {
        //      NN_SDK_ASSERT(b >= a);
        return ( GetDouble( b - a ) + a );
    }

    //---------------------------------------------------------------------------
    //! @brief      -1 または 1 を等確率で返します。
    //!
    //! @return     -1 または 1
    //---------------------------------------------------------------------------
    int GetSign() NN_NOEXCEPT
    {
        return ( GetUnsignedInteger() & 0x2 ) - 1;
    }

    //---------------------------------------------------------------------------
    //! @brief      true または false を等確率で返します。
    //!
    //! @return     true または false
    //---------------------------------------------------------------------------
    bool GetBool() NN_NOEXCEPT
    {
        // 線形合同法と異なり、XorShiftは最下位ビットの短周期の規則性はない。
        // 0または1を直接得ることができるため、最下位ビットを取る実装としている。
        return static_cast< bool >( GetUnsignedInteger() & 0x1 );
    }
    //@}

    //----------------------------------------------------------------
    //! @name       コンテキストの取得
    //----------------------------------------------------------------
    //@{

    //---------------------------------------------------------------------------
    //! @brief      乱数のコンテキストを4つの4バイト数値として取得します。
    //!
    //!             これで取得した値をコンストラクタやinitに与えることで、
    //!             全く同じコンテキストを作成することができます。
    //!
    //! @param[out] pOutValue0  コンテキストの数値を受け取るポインタ
    //! @param[out] pOutValue1  コンテキストの数値を受け取るポインタ
    //! @param[out] pOutValue2  コンテキストの数値を受け取るポインタ
    //! @param[out] pOutValue3  コンテキストの数値を受け取るポインタ
    //---------------------------------------------------------------------------
    void GetContext( uint32_t* pOutValue0, uint32_t* pOutValue1, uint32_t* pOutValue2, uint32_t* pOutValue3 ) const NN_NOEXCEPT
    {
        *pOutValue0 = m_X;
        *pOutValue1 = m_Y;
        *pOutValue2 = m_Z;
        *pOutValue3 = m_W;
    }

    //@}

private:
    uint32_t m_X;     //!< 乱数のコンテキストX
    uint32_t m_Y;     //!< 乱数のコンテキストY
    uint32_t m_Z;     //!< 乱数のコンテキストZ
    uint32_t m_W;     //!< 乱数のコンテキストW
};


//---------------------------------------------------
//! @briefprivate   エフェクトシステムでグローバルに参照されるランダムクラス
//---------------------------------------------------
class Random
{
    NN_DISALLOW_COPY( Random );
public:

    //----------------------------------------------------------------
    //! @name       コンストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  引数なしのコンストラクタです。
    //---------------------------------------------------------------------------
    Random() NN_NOEXCEPT;

    //@}

    //----------------------------------------------------------------
    //! @name       乱数テーブル
    //@{

    //---------------------------------------------------------------------------
    //! @brief      乱数テーブル作成。
    //---------------------------------------------------------------------------
    static void Initialize() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      乱数テーブル破棄。
    //---------------------------------------------------------------------------
    static void Finalize() NN_NOEXCEPT;

    //@}

    //---------------------------------------------------------------------------
    //! @brief      シード設定。
    //! @param[in]  val     乱数シード値
    //---------------------------------------------------------------------------
    void SetSeed( uint32_t val ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      テーブルから Vec3 乱数取得。
    //! @return     乱数を格納したVec3
    //---------------------------------------------------------------------------
    const nn::util::Vector3fType& GetVector3() NN_NOEXCEPT
    {
        return g_Vec3Table[ ( m_Vec3RandomIndex++ ) & RandomTableSettings_RandomTableMaskValue ];
    }

    //---------------------------------------------------------------------------
    //! @brief      テーブルから Normalized Vec3 乱数取得。
    //! @return     乱数を格納したVec3（正規化済）
    //---------------------------------------------------------------------------
    const nn::util::Vector3fType& GetNormalizedVec3() NN_NOEXCEPT
    {
        return g_NormalizedVec3Table[ ( m_NormalizedVec3RandomIndex++ ) & RandomTableSettings_RandomTableMaskValue ];
    }

    //---------------------------------------------------------------------------
    //! @brief      線形合同法のランダム。
    //! @return     乱数値
    //---------------------------------------------------------------------------
    uint32_t GetUnsignedIntegerDirect() NN_NOEXCEPT
    {
        uint32_t ret = m_RandomValue;
        m_RandomValue = m_RandomValue * 1103515245 + 12345;
        return ret;
    }

    //---------------------------------------------------------------------------
    //! @brief      0.0～1.0までのランダム。
    //! @return     乱数値
    //---------------------------------------------------------------------------
    float GetFloat() NN_NOEXCEPT
    {
        return GetUnsignedIntegerDirect() * InverseMaxValue_Float;
    }

    //---------------------------------------------------------------------------
    //! @brief      0～255までのランダム。
    //! @return     乱数値
    //---------------------------------------------------------------------------
    int GetInteger() NN_NOEXCEPT
    {
        return GetUnsignedIntegerDirect() >> 24;
    }

    //---------------------------------------------------------------------------
    //! @brief      0～valまでのランダム。
    //! @param[in]  val     乱数の上限値
    //! @return     乱数値
    //---------------------------------------------------------------------------
    int GetInteger( int val ) NN_NOEXCEPT
    {
        return static_cast< int >( ( static_cast< uint64_t >( GetUnsignedIntegerDirect() ) * val ) >> 32 );
    }

    //---------------------------------------------------------------------------
    //! @brief      グローバルランダム取得関数。
    //! @return     乱数値
    //---------------------------------------------------------------------------
    static RandomGenerator* GetGlobalRandom() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      グローバルシードを利用するかどうか。
    //! @param[in]  enable  グローバルシードの使用可否
    //! @param[in]  seed    乱数シード
    //---------------------------------------------------------------------------
    static void SetGlobalSeed( bool enable, uint32_t seed ) NN_NOEXCEPT
    {
        g_UseGlobalSeed = enable;
        g_GlobalSeed = seed;
    }

private:
    //---------------------------------------------------------------------------
    //! @brief      ランダムテーブルに関する定数
    //---------------------------------------------------------------------------
    enum RandomTableSettings
    {
        RandomTableSettings_RandomTableSize         = 512,                                      //!< ランダムテーブルのサイズ
        RandomTableSettings_RandomTableMaskValue    = RandomTableSettings_RandomTableSize - 1   //!< ランダムテーブルのマスク値
    };

    static RandomGenerator          g_Random;                       //!< グローバルランダム
    static bool                     g_UseGlobalSeed;                //!< グローバルなシードを利用する
    static uint32_t                 g_GlobalSeed;                   //!< グローバルなシード
    static nn::util::Vector3fType*  g_Vec3Table;                    //!< Vec3 ランダムテーブル
    static nn::util::Vector3fType*  g_NormalizedVec3Table;          //!< Normlized Vec3 ランダムテーブル
private:
    uint16_t                        m_Vec3RandomIndex;              //!< Vec3 乱数テーブルの位置
    uint16_t                        m_NormalizedVec3RandomIndex;    //!< Normlized Vec3 乱数テーブルの位置
    uint32_t                        m_RandomValue;                  //!< 線形合同法の為の変数
};

} // namespace detail
} // namespace vfx
} // namespace nn
