﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * :include nn/atk/sndutil_MidiManager.h
 *
 * @file atk_MidiManager.h
 */

#include <nn/atk/atk_SoundThread.h>
#include <nn/atk/atk_NoteOnCallback.h>
#include <nn/atk/atk_MidiStreamParser.h>
#include <nn/atk/atk_MidiSequencePlayer.h>
#include <nn/atk/atk_MidiSequenceTrack.h>
#include <nn/atk/atk_MidiSequenceTrackAllocator.h>
#include <nn/atk/fnd/os/atkfnd_Thread.h>
#include <nn/atk/fnd/os/atkfnd_CriticalSection.h>

namespace nn {
namespace atk {

class SoundDataManager;

//! @internal
//! @brief  MIDI 信号を受診して、指定したバンクで再生させるためのクラスです。
//!
//!         デバッグ以外の用途で使うことは禁止です。
//!
//!         また、試験的な実装のため今後インターフェイスが変わったり、
//!         廃止されたりする可能性があります。
//!
//!         詳しくは、Demo/snd/midi デモをご覧ください
//!
//!         Debug 版、Develop 版で利用可能です。
//!         Release 版ではリンクすることは可能ですが、各関数内で何も処理されません。
class MidiManager :
    public nn::atk::detail::driver::SoundThread::SoundFrameCallback,
    public nn::atk::detail::driver::NoteOnCallback,
    public nn::atk::detail::fnd::Thread::Handler
{
public:
    //! @name コンストラクタ
    //@{
    //! @brief コンストラクタです。
    MidiManager() NN_NOEXCEPT;
    //@}

    //! @name 初期化・終了処理
    //@{

    //! @brief  初期化に必要なメモリサイズを取得します。
    //! @param[in] stackSize 内部で生成される MIDI 受信スレッドのスタックサイズです。
    //! @return 初期化に必要なメモリサイズを返します。
    size_t GetRequiredMemSize( size_t stackSize = StackSize ) const NN_NOEXCEPT;

    //! @brief  MIDI マネージャの初期化を行います。
    //!
    //!         引数の内容を元に、内部で MIDI 受信用のスレッドを起動します。
    //!         MIDI 受信スレッドは、本関数を呼び出したコア上で起動されます。
    //!
    //!
    //!
    //! @param[in] threadPriority   MIDI 受信スレッドの優先度です。
    //! @param[in] stackBase        MIDI 受信スレッドのスタックとして利用されるメモリ領域です。
    //!                             GetRequiredMemSize で取得できたサイズのメモリ領域を渡す必要があります。
    //! @param[in] stackSize        MIDI 受信スレッドのスタックサイズです。
    //!
    //! @return 初期化に成功あるいは初期化済みの場合は true を返します。
    //!         初期化に失敗すると false を返します。
    //!
    //! @see GetRequiredMemSize
    //! @see Finalize
    bool Initialize( int32_t threadPriority, void* stackBase, size_t stackSize ) NN_NOEXCEPT;

    //! @brief  MIDI マネージャを破棄します。
    //!
    //!         内部で起動されたスレッドも破棄されます。
    //!
    //! @see Initialize
    void Finalize() NN_NOEXCEPT;
    //@}

    //! @name データ設定
    //@{
    //! @brief  MIDI 再生に使用されるバンクデータを設定します。
    //!
    //!         本関数では、バンク ID のみ設定されますが、
    //!         指定するバンクおよびバンクに関連付けられた波形アーカイブは、
    //!         ユーザー側でロードしておく必要があります。
    //!
    //! @param[in] bankIds  バンク ID です。最大 4 つまで設定することができます。
    //! @param[in] arc      サウンドアーカイブです。
    //! @param[in] mgr      サウンドデータマネージャです。
    //!                     このサウンドデータマネージャに関連付けられたバンクや波形アーカイブを使って
    //!                     MIDI 再生が行われます。
    void Prepare(
        SoundArchive::ItemId bankIds[],
        const SoundArchive& arc,
        const SoundDataManager& mgr) NN_NOEXCEPT;
    //@}

    //! @name MIDI 通信状況確認
    //@{
    //! @brief  MIDI 信号を受信したかどうかを取得します。
    //! @param[in] portIndex   ポート番号です。
    //! @return MIDI 信号を受信したかどうかを返します。
    //! @see    ResetMidiRecieveFlag
    bool GetMidiRecieveFlag( int portIndex = 0 ) const NN_NOEXCEPT
    {
        return m_MidiPort[portIndex].m_RecieveFlag;
    }

    //! @brief  MIDI 信号の受信状態をリセットします。
    //! @param[in] portIndex   ポート番号です。
    //! @see    GetMidiRecieveFlag
    void ResetMidiRecieveFlag( int portIndex = 0 ) NN_NOEXCEPT
    {
        m_MidiPort[portIndex].m_RecieveFlag = false;
    }
    //@}

    //! @name パラメータ設定・取得
    //@{
    //! @brief  MIDI 再生の音量を設定します。
    //! @param[in] volume   MIDI 再生の音量です。
    //! @see    GetVolume
    void SetVolume( float volume ) NN_NOEXCEPT { m_SequencePlayer.SetVolume( volume ); }

    //! @brief  MIDI 再生の音量を取得します。
    //! @return MIDI 再生の音量を返します。
    //! @see    SetVolume
    float GetVolume() const NN_NOEXCEPT { return m_SequencePlayer.GetVolume(); }

    //! @brief MIDI 再生の出力先を設定します。
    //!
    //!        @ref SoundHandle::SetOutputLine と同じ引数を渡します。
    //! @param[in] lineFlag 出力先のビットフラグです。
    //! @see   SoundHandle::SetOutputLine
    void SetOutputLine( uint32_t lineFlag ) NN_NOEXCEPT { m_SequencePlayer.SetOutputLine( lineFlag ); }
    //@}

    //! @internal
    static void MidiCallback( uint8_t status, uint8_t data1, uint8_t data2, void* arg ) NN_NOEXCEPT;

    //! @brief  MIDI 受信スレッドのデフォルトスタックサイズです。
    //!
    //!         GetRequiredMemSize のデフォルト引数として利用されます。
    //!
    //! @see    GetRequiredMemSize
    static const size_t StackSize = 16 * 1024; // 16KB
private:
    static const int MidiPortCheckIntervalUs = 3000;
    static const int BankIndexMax = 4;
    static const int MidiPortMax = 4;
    static const int MidiBufferSize = 1024;

    struct MidiBuffer
    {
        uint8_t buffer[MidiBufferSize];
        uint32_t size;
    };

    struct MidiPort
    {
        nn::atk::detail::MidiStreamParser m_Parser;
        MidiBuffer m_MidiBuffer[2];
        MidiBuffer* m_pReadBuffer;
        MidiBuffer* m_pWriteBuffer;
        bool m_RecieveFlag;

        MidiPort() NN_NOEXCEPT : m_RecieveFlag( false )
        {
            m_pReadBuffer = &m_MidiBuffer[0];
            m_pWriteBuffer = &m_MidiBuffer[1];
        }
    };

    void RegisterSoundThreadCallback() NN_NOEXCEPT;
    void UnregisterSoundThreadCallback() NN_NOEXCEPT;

    void Prepare( const void* banks[],
            const void* warcs[],
            bool warcIsIndividuals[] ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( banks );
        NN_SDK_ASSERT_NOT_NULL( warcs );
        NN_SDK_ASSERT_NOT_NULL( warcIsIndividuals );
        m_SequencePlayer.Prepare( banks, warcs, warcIsIndividuals );
    }

    virtual uint32_t Run(void* param) NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::atk::detail::driver::Channel* NoteOn(
            nn::atk::detail::driver::SequenceSoundPlayer* pSequencePlayer,
            uint8_t bankIndex,
            const nn::atk::detail::driver::NoteOnInfo& noteOnInfo
            ) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnBeginSoundFrame() NN_NOEXCEPT NN_OVERRIDE;

    nn::atk::detail::fnd::Thread m_RecieveThread;
    MidiPort m_MidiPort[MidiPortMax];
    nn::atk::detail::driver::MidiSequencePlayer m_SequencePlayer;
    nn::atk::detail::driver::MidiSequenceTrack m_SequenceTrack[16];
    nn::atk::detail::driver::MidiSequenceTrackAllocator m_SequenceTrackAllocator;

    const nn::atk::SoundArchivePlayer* m_pSoundArchivePlayer;
    detail::fnd::CriticalSection m_CriticalSection;
    volatile bool m_FinalizeFlag;
    bool m_InitializeThreadFlag;
};

}} // namespace nn::atk

