﻿using System;
using System.Collections.Generic;
using Microsoft.Win32;

namespace SdkEnvironmentCheckerLibrary
{
    /// <summary>
    /// インストールされているプログラム情報を提供します。
    /// </summary>
    public static class InstalledPrograms
    {
        private const string UninstallSubKeyName = @"Software\Microsoft\Windows\CurrentVersion\Uninstall";
        private const string DisplayNameValueName = @"DisplayName";
        private const string DisplayVersionValueName = @"DisplayVersion";

        /// <summary>
        /// インストールの種別です。
        /// </summary>
        public enum InstallType
        {
            /// <summary>
            /// すべてのユーザー向けにインストールされたプログラムを示します。
            /// LocalMachine レジストリからプログラムを検索します。
            /// </summary>
            LocalMachine,

            /// <summary>
            /// 現在のユーザー向けにインストールされたプログラムを示します。
            /// CurrentUser レジストリからプログラムを検索します。
            /// </summary>
            CurrentUser,
        }

        /// <summary>
        /// インストールされいているプログラムの情報を列挙します。
        /// </summary>
        /// <param name="installType">インストールの種別を指定します。</param>
        /// <returns>該当するプログラム情報の列挙子を返します。</returns>
        public static IEnumerable<ProgramInstallInfo> EnumerateProgramInfos(InstallType installType)
        {
            var targetKey = GetRegistryKeyFromInstallType(installType);
            if (targetKey == null) throw new ArgumentException($"Unsupported InstallType {installType}", nameof(installType));

            var uninstallKey = targetKey.OpenSubKey(UninstallSubKeyName);
            if (uninstallKey == null) throw new InvalidOperationException($"Failed to read '{UninstallSubKeyName}' registry keys.");

            return EnumerateProgramInfos(uninstallKey);
        }

        private static IEnumerable<ProgramInstallInfo> EnumerateProgramInfos(RegistryKey key)
        {
            foreach (var subKeyName in key.GetSubKeyNames())
            {
                var installedProgramKey = key.OpenSubKey(subKeyName);

                if (installedProgramKey == null)
                {
                    throw new InvalidOperationException($"failed to read '{subKeyName}' registry keys.");
                }

                yield return CreateProgramInfo(installedProgramKey);
            }
        }

        private static RegistryKey GetRegistryKeyFromInstallType(InstallType installType)
        {
            switch (installType)
            {
                case InstallType.LocalMachine:
                    return Registry.LocalMachine;

                case InstallType.CurrentUser:
                    return Registry.CurrentUser;

                default:
                    return null;
            }
        }

        private static ProgramInstallInfo CreateProgramInfo(RegistryKey key)
        {
            return new ProgramInstallInfo(
                GetSubKeyName(key.Name),
                (string)key.GetValue(DisplayNameValueName),
                (string)key.GetValue(DisplayVersionValueName));
        }

        /// <summary>
        /// キー名からサブキー名を取得します。
        /// キー名は、'\' で終わらないことを前提とします。
        /// </summary>
        /// <param name="keyName">キー名を指定します。</param>
        /// <returns>サブキー名を返します。</returns>
        private static string GetSubKeyName(string keyName)
        {
            var separatorIndex = keyName.LastIndexOf(@"\");
            return separatorIndex <= 0 ? keyName : keyName.Substring(separatorIndex + 1);
        }
    }
}
