﻿using System;
using System.IO;

namespace SdkEnvironmentCheckerLibrary
{
    /// <summary>
    /// 比較可能な単一のディレクトリを示すパスです。
    /// </summary>
    public class DirectoryPath : IEquatable<DirectoryPath>, IEquatable<string>
    {
        /// <summary>
        /// 空のディレクトリパスを示します。
        /// </summary>
        public static readonly DirectoryPath Empty = new DirectoryPath();

        /// <summary>
        /// ディレクトリパスから <see cref="DirectoryPath"/> クラスのインスタンスを作成します。
        /// ディレクトリパスが null, 空, 空白文字列である場合、DirectoryPath.Empty を返します。
        /// </summary>
        /// <param name="directoryPath">ディレクトリパス</param>
        /// <returns><see cref="DirectoryPath"/> インスタンス</returns>
        /// <exception cref="System.ArgumentException">
        /// <para>- directoryPath にパスとして不正な文字が含まれています。</para>
        /// <para>- 環境変数が指定されていますが、展開に失敗しました。</para>
        /// </exception>
        public static DirectoryPath Create(string directoryPath) => string.IsNullOrWhiteSpace(directoryPath) ? Empty : new DirectoryPath(directoryPath);

        /// <summary>
        /// ディレクトリパスから <see cref="FilePath"/> クラスのインスタンスの作成を試みます。
        /// ディレクトリパスが null, 空, 空白文字列である場合、DirectoryPath.Empty を返します。
        /// </summary>
        /// <param name="directoryPath">ディレクトリパス</param>
        /// <param name="instance"><see cref="DirectoryPath"/> インスタンス</param>
        /// <returns>インスタンス作成に成功すれば true</returns>
        public static bool TryCreate(string directoryPath, out DirectoryPath instance)
        {
            try
            {
                instance = Create(directoryPath);
                return true;
            }
            catch
            {
                instance = null;
                return false;
            }
        }

        private readonly Lazy<DirectoryPath> parent;

        private DirectoryPath()
        {
            parent = new Lazy<DirectoryPath>(() =>
            {
                if (string.IsNullOrEmpty(FullPath)) return null;

                var parent = Path.GetDirectoryName(FullPath);
                if (string.IsNullOrWhiteSpace(parent)) return null;
                return new DirectoryPath(parent);
            });
        }

        /// <summary>
        /// 指定されたディレクトリパスで DirectoryPath の新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="directoryPath">単一のディレクトリを示すパスです。</param>
        /// <exception cref="System.ArgumentException">
        /// <para>- directoryPath が null, 空, または空白文字列です。</para>
        /// <para>- directoryPath にパスとして不正な文字が含まれています。</para>
        /// <para>- 環境変数が指定されていますが、展開に失敗しました。</para>
        /// </exception>
        public DirectoryPath(string directoryPath) : this()
        {
            if (string.IsNullOrWhiteSpace(directoryPath)) throw new ArgumentException(nameof(directoryPath));

            FullPath = PathUtility.GetFullPath(directoryPath);
            NormalizedPath = PathUtility.NormalizePath(directoryPath);
        }

        /// <summary>
        /// 環境変数を展開したフルパスを取得します。パスの末尾にディレクトリ文字を含みません。
        /// </summary>
        public string FullPath { get; } = string.Empty;

        /// <summary>
        /// 比較用に正規化されたパスを取得します。パスの末尾にディレクトリ文字を含みません。
        /// </summary>
        public string NormalizedPath { get; } = string.Empty;

        /// <summary>
        /// ディレクトリが存在するかどうかを取得します。
        /// </summary>
        public bool Exists => Directory.Exists(NormalizedPath);

        /// <summary>
        /// ディレクトリ名を取得します。
        /// </summary>
        public string DirectoryName => Path.GetFileName(FullPath);

        /// <summary>
        /// 親ディレクトリのディレクトリパスを取得します。
        /// 自身が空のディレクトリパス、またはルートディレクトリである場合、null を返します。
        /// </summary>
        public DirectoryPath ParentDirectory => parent.Value;

        /// <summary>
        /// ディレクトリが存在し、かつ読み取り専用どうかを返します。
        /// </summary>
        public bool IsReadOnly
        {
            get
            {
                try
                {
                    return Exists && (File.GetAttributes(NormalizedPath) & FileAttributes.ReadOnly) != 0;
                }
                catch
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// 空のディレクトリパスであるかどうかを返します。
        /// </summary>
        public bool IsEmpty => Equals(Empty);

        /// <summary>
        /// フルパスを取得します。
        /// </summary>
        /// <returns>フルパスを取得します。</returns>
        public override string ToString() => FullPath;

        #region equality

        /// <summary>
        /// DirectoryPath を比較します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして同一であれば true</returns>
        public static bool operator ==(DirectoryPath lhs, DirectoryPath rhs) => ReferenceEquals(null, lhs) ? ReferenceEquals(null, rhs) : lhs.Equals(rhs);

        /// <summary>
        /// DirectoryPath を比較します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして異なっていれば true</returns>
        public static bool operator !=(DirectoryPath lhs, DirectoryPath rhs) => !(lhs == rhs);

        /// <summary>
        /// DirectoryPath を比較します。
        /// </summary>
        /// <param name="other">比較するパス</param>
        /// <returns>パスとして等しいかどうか</returns>
        public bool Equals(DirectoryPath other) => !ReferenceEquals(null, other) && NormalizedPath == other.NormalizedPath;

        /// <summary>
        /// 入力文字列がパスとして等しいかどうかを比較します。
        /// </summary>
        /// <param name="path">比較するファイルパス</param>
        /// <returns>パスとして等しいかどうか</returns>
        public bool Equals(string path)
        {
            if (path == null) return false;
            if (string.IsNullOrWhiteSpace(path)) return IsEmpty;

            try
            {
                return NormalizedPath == PathUtility.NormalizePath(path);
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// インスタンスを比較します。
        /// </summary>
        /// <param name="obj">比較するパス</param>
        /// <returns>パスとして等しいかどうか</returns>
        public override bool Equals(object obj)
        {
            if (obj is DirectoryPath directoryPath) return Equals(directoryPath);
            if (obj is string stringPath) return Equals(stringPath);
            return false;
        }

        public override int GetHashCode() => NormalizedPath.GetHashCode();

        #endregion equality

        #region string operators

        /// <summary>
        /// DirectoryPath を文字列に変換します。
        /// </summary>
        /// <param name="path">パスのインスタンス</param>
        public static implicit operator string(DirectoryPath path) => path?.FullPath;

        /// <summary>
        /// DirectoryPath と文字列がパスとして等しいかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス文字列</param>
        /// <returns>パスとして等しいかどうか</returns>
        public static bool operator ==(DirectoryPath lhs, string rhs) => ReferenceEquals(null, lhs) ? rhs == null : lhs.Equals(rhs);

        /// <summary>
        /// DirectoryPath と文字列がパスとして異なるかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス文字列</param>
        /// <returns>パスとして等しくないかどうか</returns>
        public static bool operator !=(DirectoryPath lhs, string rhs) => !(lhs == rhs);

        /// <summary>
        /// DirectoryPath と文字列がパスとして等しいかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス文字列</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして等しいかどうか</returns>
        public static bool operator ==(string lhs, DirectoryPath rhs) => rhs == lhs;

        /// <summary>
        /// DirectoryPath と文字列がパスとして異なるかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス文字列</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして等しくないかどうか</returns>
        public static bool operator !=(string lhs, DirectoryPath rhs) => rhs != lhs;

        #endregion string operators
    }
}
