﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace QCIT_Maker
{
    internal class FsUtil
    {
        /// <summary>
        /// 作業フォルダの絶対パスを取得
        /// </summary>
        /// <returns>作業フォルダの絶対パス文字列</returns>
        public static string GetDirFullPath(string inSrcPath)
        {
            string fullPath = Path.GetFullPath(inSrcPath);
            if (fullPath.EndsWith(@"\") == true)
            {
                return fullPath;
            }
            else
            {
                return fullPath + @"\";
            }
        }

        /// <summary>
        /// 作業フォルダの絶対パスを取得
        /// </summary>
        /// <returns>作業フォルダの絶対パス文字列</returns>
        public static List<string> GetNspFileList(string inTargetDirPath)
        {
            var pathList = new List<string>();
            // ディレクトリ内部の nsp ファイルパスをリストに詰める
            var nspDirInfo = new DirectoryInfo(inTargetDirPath);
            foreach (var nspFileInfo in nspDirInfo.GetFiles("*.nsp"))
            {
                pathList.Add(nspFileInfo.FullName);
            }
            return pathList;
        }

        /// <summary>
        /// ファイルの移動処理
        /// 移動先に指定したファイルが存在した場合は削除する
        /// </summary>
        /// <param name="inSrcPath">移動元ファイルパス</param>
        /// <param name="inDstPath">移動先ファイルパス</param>
        public static void MoveFile(string inSrcPath, string inDstPath)
        {
            try
            {
                if (File.Exists(inDstPath) == true)
                {
                    File.Delete(inDstPath);
                }
                File.Move(inSrcPath, inDstPath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.Message);
                Console.Error.WriteLine("MoveFile() Failed：src={0}, dst={1}", inSrcPath, inDstPath);
                throw err;
            }
        }

        /// <summary>
        /// ディレクトリのコピー処理
        /// </summary>
        /// <param name="inSrcDir">コピー元ディレクトリパス</param>
        /// <param name="inDstDir">コピー先ディレクトリパス</param>
        public static void CopyDirectory(string inSrcDir, string inDstDir)
        {
            try
            {
                var srcDir = new DirectoryInfo(inSrcDir);
                var dstDir = new DirectoryInfo(inDstDir);

                if (dstDir.Exists == false)
                {
                    dstDir.Create();
                    dstDir.Attributes = srcDir.Attributes;
                }

                foreach (var fileInfo in srcDir.GetFiles())
                {
                    fileInfo.CopyTo(dstDir.FullName + @"\" + fileInfo.Name, true);
                }

                foreach (var dirInfo in srcDir.GetDirectories())
                {
                    FsUtil.CopyDirectory(dirInfo.FullName, dstDir.FullName + @"\" + dirInfo.Name);
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.Message);
                Console.Error.WriteLine("CopyDirectory() Failed：src={0}, dst={1}", inSrcDir, inDstDir);
                throw err;
            }
        }

        /// <summary>
        /// 引数文字列を書き込んだテキストファイルを作成します
        /// (UTF8のBOM付き形式で保存する)
        /// </summary>
        /// <param name="inFilePath">作成するファイルパス</param>
        /// <param name="inWriteValue">書き込む文字列</param>
        public static void CreateRecordFile(string inFilePath, string inWriteValue)
        {
            // 引数文字列をファイルに書き込み (UTF8のBOM付き形式で保存)
            using (var sw = new StreamWriter(inFilePath, false, Encoding.UTF8))
            {
                sw.Write(inWriteValue);
            }
        }
    }
}
