<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Create asset data for retail interactive display of Switch.

    .DESCRIPTION
        Create asset data for retail interactive display of Switch using AssetDataCreator.exe.
        Playable time is optional and its defalut value is 900 seconds.
        Exit message is optional and it is disabled only when "disabled" is specified to -exitImage option.
#>

Param(
    [string]$outputPath,
    [string]$workDir,
    [string]$exitMessage,
    [string]$exitImage,
    [parameter(mandatory=$true)]$inputNsp,
    [int]$index,
    $playableTime
)

$scriptPath = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
$assetDataCreatorPath = Join-Path $scriptDirectoryPath AssetDataCreator.exe
$outputDirectoryName = Get-Date -Format "__ou\tpu\t_yyyyMMdd_HHmmss"
$outputDirDefault = Join-Path $scriptDirectoryPath $outputDirectoryName
$inputFileInfo = New-Object System.IO.FileInfo($inputNsp)

function CheckArguments
{
    Param([ref]$arguments, [ref]$outputPath, [ref]$workDir, [ref]$exitMessage, [ref]$exitImage, [ref]$index, $inputFileInfo, $playableTime)

    $arguments.Value = "-s -a $inputFileInfo"

    if(([string]::IsNullOrEmpty($outputPath.Value)))
    {
        $outputPath.Value = Join-Path $outputDirDefault AssetData.nsp
        New-Item -Path $outputDirDefault -ItemType directory 2>&1 > $null
    }
    $arguments.Value += (" -o " + $outputPath.Value)
    echo ("Output nsp file: " + $outputPath.Value)

    if(([string]::IsNullOrEmpty($workDir.Value)))
    {
        $workDir.Value = Join-Path $scriptDirectoryPath __work
        New-Item -Path $workDir.Value -ItemType directory 2>&1 > $null
    }
    $arguments.Value += (" -w " + $workDir.Value)
    echo ("Working Directory: " + $workDir.Value)

    if($index.Value)
    {
       if($index.Value -lt 50)
       {
           echo "[Warning] Invalid index is specified. Index must be [50-1999]. Index has been changed to 1001 forcibly."
           $index.Value = 1001
       }
    }
    else
    {
        $index.Value = 1001
    }
    $arguments.Value += (" -i " + $index.Value)

    if(-not [string]::IsNullOrEmpty($exitImage.Value))
    {
        if(Test-Path $exitImage.Value)
        {
            $arguments.Value += (" -y " + $exitImage.Value)
            echo ("Exit Image File Path: " + $exitImage.Value)
        }
        else
        {
            echo ("[Warning] Exit image file is not found.(" + $exitImage.Value + ")")
        }
    }

    if(-not [string]::IsNullOrEmpty($exitMessage.Value))
    {
        if($exitMessage.Value -eq "enabled")
        {
            $arguments.Value += " -x"
            echo ("Exit Message: Enabled")
        }
        elseif($exitMessage.Value -eq "disabled")
        {
            $arguments.Value += " -X"
            echo "Exit Message: Disabled"
        }
        else
        {
            $arguments.Value += " -x"
            echo ("[Warning] Exit Message: Invalid value is specified(" + $exitMessage.Value + "). `"enabled`" is applied as default.")
        }
    }
    else
    {
        $arguments.Value += " -x"
        echo "Exit Message: Unspecified. `"enabled`" is applied as default."
    }

    if($playableTime)
    {
        $arguments.Value += (" -t " + $playableTime)
        echo "Playable Time: $playableTime seconds"
    }
    else
    {
        echo "Playable Time: Unspecified. 900 seconds is applied as default."
    }
}

Write-Host "------------------------------"
Write-Host "CreateAssetData.ps1 start"
Write-Host "------------------------------"

if(-not(Test-Path $assetDataCreatorPath))
{
    throw "Execute file is not found.($assetDataCreatorPath)"
}

echo "Input nsp file: $inputFileInfo"

$arguments = $null

CheckArguments ([ref]$arguments) ([ref]$outputPath) ([ref]$workDir)([ref]$exitMessage) ([ref]$exitImage) ([ref]$index) ($inputFileInfo) ($playableTime)

$processInfo = New-Object System.Diagnostics.ProcessStartInfo
$processInfo.FileName = $assetDataCreatorPath
$processInfo.RedirectStandardError = $false
$processInfo.RedirectStandardOutput = $false
$processInfo.UseShellExecute = $false
$processInfo.Arguments = $arguments
$processInfo.Verb = "runas"

$process = New-Object System.Diagnostics.Process
$process.StartInfo = $processInfo

Write-Host "------------------------------"
Write-Host "Execute AssetDataCreator.exe"
Write-Host "------------------------------"

$process.Start() | Out-Null
$process.WaitForExit()

Remove-Item -Path $workDir -Recurse > $null

Write-Host "end"

exit 0