﻿using System;
using System.Collections.Generic;
using System.Text;
using Nintendo.Foundation.IO;
using System.IO;
using CommandUtility;
using System.Globalization;
using System.Threading;

namespace ReplacePreinstallAppWriter
{
    public class ReplacePreinstallAppWriter
    {
        [CommandLineOption('o', "output", Description = "output nsp image path", IsRequired = false)]
        public string OutputImagePath { get; set; }

        [CommandLineOption('i', "input", Description = "input nsp path", IsRequired = false)]
        public string InputNspPath { get; set; }

        [CommandLineOption("rule", Description = "convert rule YAML file path", IsRequired = false)]
        public string RulePath { get; set; }

        [CommandLineOption("desc", Description = "desc file path", IsRequired = false)]
        public string DescPath { get; set; }

        [CommandLineOption("outdir", Description = "output directory path. Cannot use with output and outputxml option.", IsRequired = false)]
        public string OutputDirectoryPath { get; set; }

        [CommandLineOption("outputxml", Description = "output XML file path", IsRequired = false)]
        public string OutputXmlPath { get; set; }

        [CommandLineOption("xmlschema", Description = "preinstallInfo XML schema path", IsRequired = false)]
        public string XmlSchemaPath { get; set; }

        [CommandLineOption("force", Description = "overwrite output nsp", IsRequired = false)]
        public bool Force { get; set; }

        public void SetDefaultParameter1()
        {
            if (InputNspPath == null)
            {
                InputNspPath = Path.Combine(SdkPath.FindSdkRoot(), "Programs/Iris/Outputs/NX-NXFP2-a64/TargetTools/PreinstallAppWriter/Release/PreinstallAppWriter.nsp");
                Console.WriteLine("--input option is not specified. Use default parameter {0}", InputNspPath);
            }
            if (RulePath == null)
            {
                RulePath = Path.Combine(SdkPath.FindSdkRoot(), "Integrate/Scripts/NX/Manufacturing/PreinstallAppWriter/PreinstallAppWriter_ReplaceRule.yml");
                Console.WriteLine("--rule option is not specified. Use default parameter {0}", RulePath);
            }
            if (DescPath == null)
            {
                DescPath = Path.Combine(SdkPath.FindSdkRoot(), "Programs/Iris/Sources/TargetTools/PreinstallAppWriter/PreinstallAppWriter.autogen.desc");
                Console.WriteLine("--desc option is not specified. Use default parameter {0}", DescPath);
            }
            if (OutputImagePath != null && OutputDirectoryPath != null)
            {
                throw new Exception("Both output and outdir option are specified.");
            }
            if (OutputXmlPath != null && OutputDirectoryPath != null)
            {
                throw new Exception("Both outputxml and outdir option are specified.");
            }
            if (XmlSchemaPath == null)
            {
                XmlSchemaPath = Path.Combine(SdkPath.FindSdkRoot(), "Programs/Iris/Sources/Tools/ReplacePreinstallAppWriter/ReplacePreinstallAppWriter/preinstallInfo.xsd");
                Console.WriteLine("--xmlschema option is not specified. Use default parameter {0}", XmlSchemaPath);
            }
        }

        public void SetDefaultParameter2(string version)
        {
            if (OutputDirectoryPath == null)
            {
                OutputDirectoryPath = SdkPath.FindSdkRoot();
            }
            else
            {
                System.IO.Directory.CreateDirectory(OutputDirectoryPath);
            }
            if (OutputImagePath == null)
            {
                OutputImagePath = Path.Combine(OutputDirectoryPath, "PreinstallAppWriter_" + version + ".nsp");
                Console.WriteLine("--output option is not specified. Use parameter {0}", OutputImagePath);
            }
            if (OutputXmlPath == null)
            {
                OutputXmlPath = Path.Combine(OutputDirectoryPath, "preinstallInfo_" + version + ".xml");
                Console.WriteLine("--outputxml option is not specified. Use parameter {0}", OutputXmlPath);
            }
        }

        public int Run()
        {
            SetDefaultParameter1();

            using (var tempHolder = new TemporaryFileHolder("ReplacePreinstallAppWriter"))
            {
                var converter = new Converter();
                var config = converter.Parse(FileUtility.MakeExistedFileInfo("convert rule", RulePath));
                SetDefaultParameter2(config.Version);
                ReplaceNsp(config.Application, config.Version, tempHolder);
                ReplaceNspForRepair(tempHolder);

                PreinstallInfo.OutputXml(OutputXmlPath, config.Application.ToArray(), config.Version, tempHolder.CreateTemporaryDirectory("AuthoringWorkingDir"));
                PreinstallInfo.Validate(OutputXmlPath, XmlSchemaPath);
            }

            return 0;
        }

        public void ReplaceNsp(List<string> application, string version, TemporaryFileHolder holder)
        {
            var versionFilePath = holder.CreateTemporaryFilePath(Path.GetRandomFileName()).FullName;
            File.WriteAllBytes(versionFilePath, Encoding.UTF8.GetBytes(version));

            var targetFile = "PreinstallVersion.txt";
            var sourceFile = versionFilePath;

            for (int i = 0; i < application.Count; i++)
            {
                var targetStr = Path.GetExtension(application[i]) == ".nspu" ?
                "TargetApplication" + i.ToString() + ".nspu" : "TargetPatch" + i.ToString() + ".nsp";
                targetFile = targetFile + "|" + string.Join("|", targetStr);
            }
            sourceFile = sourceFile + "|" + string.Join("|", application);


            if (Force)
            {
                File.Delete(OutputImagePath);
            }

            SdkTool.Execute("ReplaceFileInNsp.exe",
                "-i", InputNspPath,
                "--target", targetFile,
                "--source", sourceFile,
                "--desc", DescPath,
                "--output", OutputImagePath
                );
        }

        public void ReplaceNspForRepair(TemporaryFileHolder holder)
        {
            var repairSettingFilePath = holder.CreateTemporaryFilePath(Path.GetRandomFileName()).FullName;
            var value = new byte[] { 0x01 };
            File.WriteAllBytes(repairSettingFilePath, value);

            var outputNsp = OutputImagePath.Replace(".nsp", "_Repair.nsp");

            if (Force)
            {
                File.Delete(outputNsp);
            }

            SdkTool.Execute("ReplaceFileInNsp.exe",
                "-i", OutputImagePath,
                "--target", "RepairSetting.bin",
                "--source", repairSettingFilePath,
                "--desc", DescPath,
                "--output", outputNsp
                );
        }

    }

    class Program
    {
        static void Main(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);
            try
            {
                SingleCommandRunner<ReplacePreinstallAppWriter>.Run(args);
            }
            catch
            {
                Environment.Exit(1);
            }

        }
    }
}
