﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;

namespace Nintendo.NX.IrsensorNfcFirmwareUpdater
{
    class Config
    {
        /// <summary>
        /// 最近使用したファイルの最大数
        /// </summary>
        const int RecentFileCountMax = 1;

        /// <summary>
        /// 唯一のインスタンス
        /// </summary>
        private static Config instance = null;

        /// <summary>
        /// 設定ファイル名
        /// </summary>
        private const string ConfigFileName = "config.xml";

        /// <summary>
        /// 一時設定ファイル名
        /// </summary>
        private const string TempConfigFileName = "config.$$$";

        /// <summary>
        /// 最後に使用したポート番号
        /// </summary>
        public int LastPortNumber { get; set; }

        /// <summary>
        /// 直近に使用したファイルリスト
        /// </summary>
        public List<string> RecentFileList { get; private set; }

        /// <summary>
        /// ベリファイ有効フラグ
        /// </summary>
        public bool VerifyEnabled { get; set; }

        /// <summary>
        /// 設定ファイルのパス
        /// </summary>
        private string ConfigFilePath
        {
            get { return Path.Combine(Application.StartupPath, ConfigFileName); }
        }

        /// <summary>
        /// 一時設定ファイルのパス
        /// </summary>
        private string TempConfigFilePath
        {
            get { return Path.Combine(Application.StartupPath, TempConfigFileName); }
        }

        private Config()
        {
            LastPortNumber = 0;
            RecentFileList = new List<string>();
            VerifyEnabled = false;
        }

        /// <summary>
        /// インスタンスの取得
        /// </summary>
        /// <returns></returns>
        public static Config GetInstance()
        {
            if (instance == null)
            {
                instance = new Config();
            }

            return instance;
        }

        private void CopyFrom(Config config)
        {
            LastPortNumber = config.LastPortNumber;
            RecentFileList.Clear();
            RecentFileList.AddRange(config.RecentFileList);
            VerifyEnabled = config.VerifyEnabled;
        }

        /// <summary>
        /// 直近に使用したファイルの追加登録
        /// </summary>
        /// <param name="path"></param>
        public void RegisterRecentFile(string path)
        {
            for (int i = 0; i < RecentFileList.Count; i++)
            {
                if (string.Equals(RecentFileList[i], path, StringComparison.OrdinalIgnoreCase))
                {
                    // 既に登録済みなら削除
                    RecentFileList.RemoveAt(i);
                    break;
                }
            }

            RecentFileList.Insert(0, path);

            // 溢れた分を削除
            while (RecentFileList.Count > RecentFileCountMax)
            {
                RecentFileList.RemoveAt(RecentFileList.Count - 1);
            }
        }

        /// <summary>
        /// 設定の読み込み
        /// </summary>
        public void Load()
        {
            if (!File.Exists(ConfigFilePath))
            {
                return;
            }

            var tempConfig = new Config();

            var document = new XmlDocument();
            document.Load(ConfigFilePath);

            var root = document.DocumentElement;
            for (int i = 0; i < root.ChildNodes.Count; i++)
            {
                var node = root.ChildNodes[i];
                switch (node.Name)
                {
                case "LastPortNumber":
                    {
                        int configValue;
                        if (int.TryParse(node.InnerText, out configValue))
                        {
                            tempConfig.LastPortNumber = configValue;
                        }
                    }
                    break;
                case "RecentFiles":
                    if (node.HasChildNodes)
                    {
                        for (int j = 0; j < node.ChildNodes.Count; j++)
                        {
                            var fileNode = node.ChildNodes[j];
                            tempConfig.RecentFileList.Add(fileNode.InnerText);
                        }
                    }
                    break;
                case "Verify":
                    {
                        bool configValue;
                        if (bool.TryParse(node.InnerText, out configValue))
                        {
                            tempConfig.VerifyEnabled = configValue;
                        }
                    }
                    break;
                }
            }

            CopyFrom(tempConfig);
        }

        /// <summary>
        /// 設定の保存
        /// </summary>
        public void Save()
        {
            var settings = new XmlWriterSettings();
            settings.Encoding = Encoding.UTF8;

            XmlWriter writer = null;
            using (writer = XmlWriter.Create(TempConfigFilePath, settings))
            {
                writer.WriteStartDocument();
                writer.WriteStartElement("IrsensorNfcFirmwareUpdater");

                writer.WriteElementString("LastPortNumber", LastPortNumber.ToString());

                writer.WriteStartElement("RecentFiles");
                foreach (var file in RecentFileList)
                {
                    writer.WriteElementString("File", file);
                }
                writer.WriteEndElement();

                writer.WriteElementString("Verify", VerifyEnabled.ToString());

                writer.WriteEndElement();
                writer.WriteEndDocument();
                writer.Close();
            }

            // 保存に成功したら一時ファイルを本ファイルに上書き
            File.Copy(TempConfigFilePath, ConfigFilePath, true);
            File.Delete(TempConfigFilePath);
        }
    }
}
