﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition.Hosting;
    using System.Linq;
    using System.Net.Sockets;
    using System.Reflection;
    using System.Threading;
    using System.Threading.Tasks;
    using Services;

    /// <summary>
    /// セッションを管理します。
    /// </summary>
    internal sealed class Session : IDisposable
    {
        /// <summary>
        /// メッセージのバイト数です。
        /// </summary>
        internal const int MessageSize = 12;

        private const ulong InvalidSessionId = 0;

        private readonly object stateSyncObject = new object();

        private State state = State.Disconnected;

        private ulong sessionId = 0;

        private TcpClient client = null;

        private readonly object serviceSyncObject = new object();

        private readonly IEnumerable<ISessionService> services;

        private readonly CaptureService captureService;

        private readonly PingPongService pingPongService;

        private readonly object startingSyncObject = new object();

        private readonly ManualResetEvent connectionEvent;

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        private readonly string ipAddress;

        private readonly int port;

        /// <summary>
        /// Session クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="info">セッションの情報です。</param>
        internal Session(SessionInfo info)
        {
            this.Name = info.Name;
            this.ipAddress = info.IpAddress;
            this.port = info.Port;

            Assembly assembly = Assembly.GetExecutingAssembly();
            using (var catalog = new AssemblyCatalog(assembly))
            using (var container = new CompositionContainer(catalog))
            {
                this.services = container.GetExportedValues<ISessionService>();
            }

            foreach (ISessionService service in this.services)
            {
                service.Info = info;
            }

            this.captureService = this.GetService<CaptureService>();

            this.pingPongService = this.GetService<PingPongService>();

            this.connectionEvent = new ManualResetEvent(false);

            this.StartMonitoring();
        }

        ~Session()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// セッションの対向となる開発機の名前を返します。
        /// </summary>
        internal string Name { get; private set; }

        /// <summary>
        /// セッションが接続状態にあるか否かを表す値を返します。
        /// </summary>
        internal bool IsConnected
        {
            get
            {
                lock (this.stateSyncObject)
                {
                    return (this.state == State.Connected);
                }
            }
        }

        /// <summary>
        /// タイムアウトが発生したか否かを表す値を返します。
        /// </summary>
        internal bool IsTimedOut
        {
            get
            {
                lock (this.serviceSyncObject)
                {
                    return this.pingPongService.IsTimedOut && this.IsConnected;
                }
            }
        }

        /// <summary>
        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Capture サービスのセッションを取得します。
        /// </summary>
        /// <returns>Capture サービスのセッションです。</returns>
        internal ICaptureSession GetCaptureSession()
        {
            lock (this.serviceSyncObject)
            {
                return this.captureService.GetCaptureSession();
            }
        }

        /// <summary>
        /// セッションを開始します。
        /// </summary>
        internal void Start()
        {
            lock (this.stateSyncObject)
            {
                if (this.state != State.Disconnected)
                {
                    return;
                }

                this.state = State.Connecting;
            }

            Task.Run(() =>
            {
                lock (this.startingSyncObject)
                {
                    lock (this.stateSyncObject)
                    {
                        if (this.state != State.Connecting)
                        {
                            return;
                        }
                    }

                    TcpClient localClient = null;

                    try
                    {
                        localClient = new TcpClient(this.ipAddress, this.port);
                    }
                    catch
                    {
                        lock (this.stateSyncObject)
                        {
                            if (this.state != State.Disposed)
                            {
                                this.state = State.Disconnected;
                            }
                        }
                    }

                    lock (this.serviceSyncObject)
                    {
                        var isConnected = false;

                        lock (this.stateSyncObject)
                        {
                            if (this.state == State.Connecting)
                            {
                                this.state = State.Connected;
                                this.sessionId += 1;
                                this.client = localClient;

                                localClient = null;

                                isConnected = true;

                                this.connectionEvent.Set();
                            }
                        }

                        if (isConnected)
                        {
                            var writer = new Writer(this.client, new object());

                            foreach (ISessionService service in this.services)
                            {
                                service.Start(writer);
                            }
                        }
                    }

                    if (localClient != null)
                    {
                        localClient.Close();
                    }
                }
            });
        }

        /// <summary>
        /// セッションを停止します。
        /// </summary>
        internal void Stop()
        {
            TcpClient localClient = null;

            lock (this.serviceSyncObject)
            {
                lock (this.stateSyncObject)
                {
                    if (this.state == State.Disposed)
                    {
                        return;
                    }

                    localClient = this.client;

                    this.state = State.Disconnected;
                    this.client = null;
                }

                if (localClient != null)
                {
                    foreach (ISessionService service in this.services)
                    {
                        service.Stop();
                    }
                }
            }

            if (localClient != null)
            {
                localClient.Close();
            }
        }

        private void Dispose(bool isDisposing)
        {
            bool isDisposed = true;

            TcpClient localClient = null;

            lock (this.serviceSyncObject)
            {
                lock (this.stateSyncObject)
                {
                    if (this.state != State.Disposed)
                    {
                        isDisposed = false;

                        localClient = this.client;

                        this.state = State.Disposed;
                        this.client = null;
                    }
                }

                if (localClient != null)
                {
                    foreach (ISessionService service in this.services)
                    {
                        service.Stop();
                    }
                }
            }

            if (localClient != null)
            {
                localClient.Close();
            }

            if (!isDisposed)
            {
                if (isDisposing)
                {
                    this.StopMonitoring();
                }
            }
        }

        private T GetService<T>() where T : class, ISessionService
        {
            return this.services.Select(x => x as T).First(x => x != null);
        }

        private void Monitor(CancellationToken token)
        {
            var buffer = new byte[MessageSize];

            while (!token.IsCancellationRequested)
            {
                ulong localSessionId = InvalidSessionId;

                TcpClient localClient = null;

                lock (this.stateSyncObject)
                {
                    if (this.state == State.Connected)
                    {
                        localSessionId = this.sessionId;

                        localClient = this.client;
                    }
                    else
                    {
                        this.connectionEvent.Reset();
                    }
                }

                if (localClient == null)
                {
                    WaitHandle.WaitAny(new[]
                    {
                        token.WaitHandle, this.connectionEvent
                    });

                    continue;
                }

                NetworkStream stream = localClient.GetStream();

                try
                {
                    while (true)
                    {
                        int count = 0;

                        using (Task<int> task =
                            stream.ReadAsync(buffer, 0, buffer.Length, token))
                        {
                            task.Wait();

                            count = task.GetAwaiter().GetResult();
                        }

                        byte[] message = buffer.Take(count).ToArray();

                        lock (this.serviceSyncObject)
                        {
                            if (this.IsConnected)
                            {
                                foreach (
                                    ISessionService service in this.services)
                                {
                                    service.AcceptMessage(message);
                                }
                            }
                        }
                    }
                }
                catch
                {
                    lock (this.serviceSyncObject)
                    {
                        lock (this.stateSyncObject)
                        {
                            if (this.state != State.Connected ||
                                this.sessionId != localSessionId)
                            {
                                localClient = null;
                            }
                            else
                            {
                                this.state = State.Disconnected;
                                this.client = null;
                            }
                        }

                        if (localClient != null)
                        {
                            foreach (ISessionService service in this.services)
                            {
                                service.Cancel();
                            }
                        }
                    }

                    if (localClient != null)
                    {
                        localClient.Close();
                    }
                }
            }
        }

        private void StartMonitoring()
        {
            this.tokenSource = new CancellationTokenSource();

            CancellationToken token = this.tokenSource.Token;

            this.task = Task.Run(() => this.Monitor(token));
        }

        private void StopMonitoring()
        {
            this.tokenSource.Cancel();

            this.Stop();

            this.task.Wait();
            this.task.Dispose();
            this.task = null;

            this.tokenSource.Dispose();
            this.tokenSource = null;
        }

        private enum State
        {
            Disposed,
            Disconnected,
            Connecting,
            Connected,
        }

        private sealed class Writer : ISessionWriter
        {
            private readonly TcpClient client;

            private readonly object syncObject;

            internal Writer(TcpClient client, object syncObject)
            {
                this.client = client;

                this.syncObject = syncObject;
            }

            public void Write(
                byte[] buffer, int offset, int count, CancellationToken token)
            {
                NetworkStream stream = this.client.GetStream();

                lock (this.syncObject)
                {
                    using (Task task =
                        stream.WriteAsync(buffer, offset, count, token))
                    {
                        task.Wait();
                    }
                }
            }
        }
    }
}
