﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// メモリ上にマップされた入出力ポートを管理します。
    /// </summary>
    internal sealed class IoPortManager : IDisposable
    {
        private static readonly TimeSpan Interval =
            TimeSpan.FromMilliseconds(8);

        private readonly object syncObject = new object();

        private readonly Dictionary<string, uint> portMap =
            new Dictionary<string, uint>();

        private bool isDisposed = false;

        private bool isStarted = false;

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        /// <summary>
        /// IoPortManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal IoPortManager()
        {
        }

        ~IoPortManager()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// ポーリングが行われた際に発生します。
        /// </summary>
        internal event EventHandler<HidShellPortAccessor>
            Polling = delegate { };

        /// <summary>
        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// ポートをアタッチします。
        /// </summary>
        /// <param name="name">ポートの名前です。</param>
        /// <returns>ポートのアタッチに成功したか否かを表す値です。</returns>
        internal bool AttachPort(string name)
        {
            bool isAttached = false;

            using (var accessor = new HidShellPortAccessor())
            {
                isAttached = accessor.AttachPort(name);

                if (isAttached)
                {
                    uint port = HidShellPortAccessor.DefaultPort;

                    isAttached = accessor.GetPort(out port, name);

                    lock (this.syncObject)
                    {
                        this.portMap[name] = port;
                    }
                }
            }

            return isAttached;
        }

        /// <summary>
        /// ポートをデタッチします。
        /// </summary>
        /// <param name="name">ポートの名前です。</param>
        internal void DetachPort(string name)
        {
            using (var accessor = new HidShellPortAccessor())
            {
                lock (this.syncObject)
                {
                    this.portMap.Remove(name);
                }

                accessor.DetachPort(name);
            }
        }

        /// <summary>
        /// ポートを返します。
        /// </summary>
        /// <param name="name">ポートの名前です。</param>
        /// <returns>ポートです。</returns>
        internal uint GetPort(string name)
        {
            lock (this.syncObject)
            {
                uint port = HidShellPortAccessor.DefaultPort;

                if (this.portMap.TryGetValue(name, out port))
                {
                    return port;
                }
                else
                {
                    return HidShellPortAccessor.DefaultPort;
                }
            }
        }

        /// <summary>
        /// ポーリングを開始します。
        /// </summary>
        internal void Start()
        {
            this.tokenSource = new CancellationTokenSource();

            CancellationToken token = this.tokenSource.Token;

            this.task = Task.Run(() => this.Monitor(token));

            this.isStarted = true;
        }

        /// <summary>
        /// ポーリングを停止します。
        /// </summary>
        internal void Stop()
        {
            this.isStarted = false;

            this.tokenSource.Cancel();

            this.task.Wait();

            this.task.Dispose();
            this.task = null;

            this.tokenSource.Dispose();
            this.tokenSource = null;
        }

        private void Dispose(bool isDisposing)
        {
            if (!this.isDisposed)
            {
                this.isDisposed = true;

                if (isDisposing)
                {
                    if (this.isStarted)
                    {
                        this.Stop();
                    }
                }
            }
        }

        private void Monitor(CancellationToken token)
        {
            var stopwatch = new Stopwatch();

            while (!token.IsCancellationRequested)
            {
                stopwatch.Start();

                using (var accessor = new HidShellPortAccessor())
                {
                    this.Polling(this, accessor);
                }

                stopwatch.Stop();

                if (stopwatch.Elapsed < Interval)
                {
                    token.WaitHandle.WaitOne(Interval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }
    }
}
