﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace InputCapture.Communication
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Net.Sockets;
    using System.Text;
    using System.Threading;
    using InputCapture.Binary;

    /// <summary>
    /// 通信処理クラスです。
    /// </summary>
    public sealed class Connecter : IDisposable
    {
        /// <summary>
        /// 接続状態を表す列挙子です。
        /// </summary>
        public enum ConnectionState
        {
            Disconnected,           // 切断中です。
            StartConnection,        // 接続を開始しています。
            Connected               // 接続中です。
        }

        private readonly string LocalHost = "localhost";
        private const int HostPortNo = 6003;
        private readonly string ChannelName = "NW_INPUT_CAP";
        private int InputCapturePortNo = -1;
        private TcpClient _client = null;

        Thread _connectionThread = null;
        Thread _receiveMessageThread = null;

        uint _checkAliveInterval = UserSettings.DefaultCheckAliveTimeout;
        int _pollingInterval = (int)UserSettings.DefaultPollingInterval;
        const int Timeout = 3000;
        const int ConnectInterval = 1000;
        const int CheckAliveTrials = 2;
        bool _checkAliveWaiting = false;
        bool _recvAck = false;

        ConnectionState _connectionState = ConnectionState.Disconnected;
        bool _autoConnectEnabled = true;
        bool _autoDisconnectEnabled = true;
        bool _manualDisconnect = false;

        bool _isReading = false;
        bool _doneRead = false;
        int _readSize = 0;
        const int MaxMsgReadSize = MessageSender.MsgSize * 8;
        MemoryStream _memRead = new MemoryStream(MaxMsgReadSize);

        bool _mouseOut = false;
        Point _mouseOutPos = new Point();

        readonly byte[] _pingTypeMsg = new byte[4] { (byte)MessageType.Ping, 0, 0, 0 };
        readonly byte[] _pingMsg = new byte[8] { (byte)'p', (byte)'i', (byte)'n', (byte)'g', 0, 0, 0, 0 };
        readonly byte[] _ackMsg = new byte[8] { (byte)'a', (byte)'c', (byte)'k', (byte)' ', 0, 0, 0, 0 };

        //-----------------------------------------------------------------

        public bool AutoConnectEnabled
        {
            get { return this._autoConnectEnabled; }
            set { this._autoConnectEnabled = value; }
        }

        public bool AutoDisconnectEnabled
        {
            get { return this._autoDisconnectEnabled; }
            set { this._autoDisconnectEnabled = value; }
        }

        public uint CheckAliveInterval
        {
            get { return this._checkAliveInterval; }
            set { this._checkAliveInterval = value; }
        }

        public int PollingInterval
        {
            get { return this._pollingInterval; }
            set { this._pollingInterval = value; }
        }

        /// <summary>
        /// 手動で切断したことを表すフラグです。
        /// </summary>
        public bool ManualDisconnect
        {
            set { this._manualDisconnect = value; }
        }

        /// <summary>
        /// 接続状態を取得します。
        /// </summary>
        public ConnectionState GetConnectionState()
        {
            return _connectionState;
        }

        public bool IsMouseOut
        {
            get { return this._mouseOut; }
        }

        public void  ResetMouseOut()
        {
            this._mouseOut = false;
        }

        public Point MouseOutPos
        {
            get { return this._mouseOutPos; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Connecter()
        {
            _connectionThread = new Thread(new ThreadStart(this.ConnectionThreadProc));
            _connectionThread.Start();

            _receiveMessageThread = new Thread(new ThreadStart(this.ReceiveMessageThreadProc));
            _receiveMessageThread.Start();
        }

        public void Dispose()
        {
            _connectionThread.Abort();
            _connectionThread.Join();
            _connectionThread = null;
            _receiveMessageThread.Abort();
            _receiveMessageThread.Join();
            _receiveMessageThread = null;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 接続状態確認用スレッドプロシージャです。
        /// </summary>
        private void ConnectionThreadProc()
        {
            try
            {
                Stopwatch stopWatch = new Stopwatch();
                stopWatch.Start();
                int _numTrials = 0;

                while (true)
                {
                    if (_connectionState != ConnectionState.Connected && _connectionState != ConnectionState.StartConnection)
                    {
                        if (_autoConnectEnabled && !_manualDisconnect)
                        {
                            StartConnect();
                        }

                        _numTrials = 0;

                        stopWatch.Reset();
                        stopWatch.Start();
                    }

                    if (_autoDisconnectEnabled && _connectionState == ConnectionState.Connected)
                    {
                        if (stopWatch.ElapsedMilliseconds > _checkAliveInterval * 1000 / CheckAliveTrials)
                        {
                            // 生存確認をします。
                            {
                                bool bAlive = true;

                                if (_checkAliveWaiting && !ReceiveAck())
                                {
                                    bAlive = false;
                                }

                                if (!_checkAliveWaiting && !SendPing())
                                {
                                    bAlive = false;
                                }

                                if (bAlive)
                                {
                                    _numTrials = 0;
                                }
                                else
                                {
                                    ++_numTrials;
                                }
                            }

                            // CheckAliveTrials 回生存確認をしてすべて失敗した場合に切断します。
                            if (_numTrials >= CheckAliveTrials)
                            {
                                Disconnect();

                                Debug.WriteLine("Disconnected due to timeout.");
                            }

                            stopWatch.Reset();
                            stopWatch.Start();
                        }
                    }

                    Thread.Sleep(ConnectInterval);
                }
            }
            catch (System.Threading.ThreadAbortException)
            {
                Debug.WriteLine("Thread Abort: ConnectionThreadProc");
            }
        }

        /// <summary>
        /// 実機から送られてくるメッセージを処理するスレッドプロシージャです。
        /// </summary>
        private void ReceiveMessageThreadProc()
        {
            try
            {
                while (true)
                {
                    ReceiveMessage();

                    Thread.Sleep(_pollingInterval);
                }
            }
            catch (System.Threading.ThreadAbortException)
            {
                Debug.WriteLine("Thread Abort: ReceiveMessageThreadProc");
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 接続を開始します。
        /// </summary>
        public void StartConnect()
        {
            if (_connectionState == ConnectionState.StartConnection ||
                _connectionState == ConnectionState.Connected)
            {
                return;
            }

            _connectionState = ConnectionState.StartConnection;

            Thread connectingThread = new Thread(new ThreadStart(() =>
            {
                if (!Connect())
                {
                    _connectionState = ConnectionState.Disconnected;
                    return;
                }

                _connectionState = ConnectionState.Connected;
            }));

            connectingThread.Start();
        }

        /// <summary>
        /// 通信を停止します。
        /// </summary>
        public void StopConnect()
        {
            Disconnect();
        }

        /// <summary>
        /// 接続します。
        /// </summary>
        private bool Connect()
        {
            _isReading = false;
            _doneRead = false;

            try
            {
                // Host Bridge に接続。
                TcpClient client = new TcpClient(LocalHost, HostPortNo);
                client.ReceiveTimeout = Timeout;
                client.SendTimeout = Timeout;

                if (client != null)
                {
                    // InputCapture のためのポート番号を取得します。
                    InputCapturePortNo = this.GetTargetPort(client, ChannelName);

                    client.Close();
                }
            }
            catch (SocketException)
            {
                Debug.WriteLine("HostBridge failure.");
            }

            if (InputCapturePortNo == -1)
            {
                return false;
            }

            try
            {
                // InputCapture に接続。
                _client = new TcpClient(LocalHost, InputCapturePortNo);
                _client.ReceiveTimeout = Timeout;
                _client.SendTimeout = Timeout;

                if (_client != null)
                {
                    Debug.WriteLine("Connection success.");
                    return true;
                }
            }
            catch
            {
                Debug.WriteLine("Connection failure.");
            }

            return false;
        }

        /// <summary>
        /// 切断します。
        /// </summary>
        private void Disconnect()
        {
            _client.Close();
            _client = null;
            _connectionState = ConnectionState.Disconnected;

            Debug.WriteLine("Disconnection success.");
        }

        /// <summary>
        /// ネットワーク上のストリームからポート情報を読み出します。
        /// </summary>
        /// <param name="stream">ポート情報を取得するためのストリームです。</param>
        /// <param name="channelName">ポート情報を取得するチャンネル名です。</param>
        /// <returns>NW_INPUT_CAP のポート番号を返します。</returns>
        private int GetTargetPort(TcpClient client, string channelName)
        {
            // Host Bridge ドライバ（ポート6003）から次の形式でデータが来ます。
            //    "+<Channel Name>:<Port Number>" ... チャンネル開始
            //    "-<Channel Name>:<Port Number>" ... チャンネル終了

            MemoryStream memory = new MemoryStream();
            NetworkStream stream = client.GetStream();

            try
            {
                int portNo = -1;

                int readBytes = 0;
                byte[] buffer = new byte[256];

                do
                {
                    readBytes = stream.Read(buffer, 0, buffer.Length);

                    if (readBytes > 0)
                    {
                        memory.Write(buffer, 0, readBytes);
                    }

                } while (client.Available > 0);

                string infoString = Encoding.ASCII.GetString(memory.ToArray());

                IList<string> channels = this.SplitChannelInfos(infoString);

                foreach (string channel in channels)
                {
                    string[] tokens = channel.Split(new char[] { ':' }, 2);

                    if (tokens.Length != 2)
                    {
                        continue;
                    }

                    string chName = tokens[0];
                    int port = int.Parse(tokens[1]);

                    if (0 != chName.Substring(1).CompareTo(channelName))
                    {
                        continue;
                    }

                    if (chName[0] == '+')
                    {
                        return port;
                    }

                    portNo = port;
                }

                return portNo;
            }
            catch (IOException)
            {
                return -1;
            }
            finally
            {
                stream.Close();
            }
        }

        /// <summary>
        /// 文字列をチャンネル情報の配列に分割します。
        /// </summary>
        /// <param name="message">全チャンネル情報の入った文字列です。</param>
        /// <returns>チャンネル情報単位に分割して配列を返します。</returns>
        private IList<string> SplitChannelInfos(string message)
        {
            List<string> stringList = new List<string>();

            for (int index = 0; index < message.Length; )
            {
                if (message[index] == '+' || message[index] == '-')
                {
                    int i = index + 1;
                    for (; i < message.Length; ++i)
                    {
                        if (message[i] == '+' || message[i] == '-')
                        {
                            break;
                        }
                    }

                    stringList.Add(message.Substring(index, i - index));
                    index = i;
                }
                else
                {
                    ++index;
                }
            }

            return stringList;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 非同期でストリームの書き込みを行います。
        /// </summary>
        public bool WriteStream(MemoryStream memStream)
        {
            if (_connectionState == ConnectionState.Connected)
            {
                try
                {
                    NetworkStream stream = _client.GetStream();
                    stream.BeginWrite(memStream.GetBuffer(), 0, (int)memStream.Length, WriteAsyncCallback, stream);
                    return true;
                }
                catch
                {
                    Debug.WriteLine("WriteStream failure.");
                    _connectionState = ConnectionState.Disconnected;
                }
            }

            return false;
        }

        /// <summary>
        /// 非同期書き込みのコールバック関数です。
        /// </summary>
        private void WriteAsyncCallback(IAsyncResult ar)
        {
            NetworkStream state = (NetworkStream)ar.AsyncState;

            try
            {
                state.EndWrite(ar);
            }
            catch
            {
                Debug.WriteLine("WriteAsyncCallback failure.");
                _connectionState = ConnectionState.Disconnected;
            }
        }

        /// <summary>
        /// 非同期でストリームの読み込みを行います。
        /// </summary>
        public bool ReadStream(MemoryStream memStream, int size)
        {
            if (_connectionState == ConnectionState.Connected)
            {
                if (!_doneRead && !_isReading)
                {
                    try
                    {
                        _isReading = true;
                        NetworkStream stream = _client.GetStream();
                        stream.BeginRead(memStream.GetBuffer(), 0, size, ReadAsyncCallback, stream);
                    }
                    catch
                    {
                        Debug.WriteLine("ReadStream failure.");
                        _connectionState = ConnectionState.Disconnected;
                        return false;
                    }
                }

                return true;
            }

            return false;
        }

        /// <summary>
        /// 非同期読み込みのコールバック関数です。
        /// </summary>
        private void ReadAsyncCallback(IAsyncResult ar)
        {
            NetworkStream state = (NetworkStream)ar.AsyncState;

            try
            {
                _readSize = state.EndRead(ar);
                _isReading = false;
                _doneRead = true;
            }
            catch
            {
                Debug.WriteLine("ReadAsyncCallback failure.");
                _connectionState = ConnectionState.Disconnected;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 送信テストを行います。
        /// </summary>
        private bool SendPing()
        {
            bool result = false;

            if (_connectionState == ConnectionState.Connected)
            {
                MemoryStream memStream = new MemoryStream();
                BinaryWriter bw = new BinaryWriter(memStream);
                bw.Write(_pingTypeMsg);
                bw.Write(_pingMsg);
                bw.Flush();

                if (WriteStream(memStream))
                {
                    _checkAliveWaiting = true;
                    result = true;
                    // Debug.WriteLine("Sending Ping OK.");
                }
                else
                {
                    Debug.WriteLine("Sending Ping NG!");
                }
            }

            return result;
        }

        /// <summary>
        /// 受信テストを行います。
        /// </summary>
        private bool ReceiveAck()
        {
            bool result = false;

            if (_connectionState == ConnectionState.Connected)
            {
                if (_checkAliveWaiting)
                {
                    if (_recvAck)
                    {
                        result = true;
                        // Debug.WriteLine("Ack OK.");
                    }
                    else
                    {
                        Debug.WriteLine("Ack NG!");
                    }
                }

                _checkAliveWaiting = false;
                _recvAck = false;
            }

            return result;
        }

        /// <summary>
        /// 実機アプリからのメッセージを受信します。
        /// </summary>
        private void ReceiveMessage()
        {
            if (!_isReading && !_doneRead)
            {
                ReadStream(_memRead, MaxMsgReadSize);
            }
            else if (_doneRead)
            {
                if (_readSize > 0)
                {
                    BinaryReader reader = new NetBinaryReader(new MemoryStream(_memRead.GetBuffer()));
                    reader.BaseStream.Position = 0;

                    for (int i = 0; i < _readSize / MessageSender.MsgSize; ++i)
                    {
                        byte type = reader.ReadByte();
                        reader.ReadBytes(3);

                        switch (type)
                        {
                            case (byte)MessageType.Ack:
                                // 実機アプリからの応答。
                                {
                                    byte[] readMsg = reader.ReadBytes(8);

                                    if (System.Linq.Enumerable.SequenceEqual(readMsg, _ackMsg))
                                    {
                                        if (_checkAliveWaiting)
                                        {
                                            _recvAck = true;
                                        }
                                    }
                                }
                                break;

                            case (byte)MessageType.CaptureStop:
                                // キャプチャ停止。（実機からマウスカーソルが出たとき。）
                                {
                                    reader.ReadBytes(4);  // 不必要な情報を空読みします。
                                    _mouseOut = true;
                                    _mouseOutPos = new Point(reader.ReadInt16(), reader.ReadInt16());
                                }
                                break;

                            default:
                                Debug.WriteLine("Received unknown message!!");
                                break;
                        }
                    }
                }

                _doneRead = false;
            }
        }
    }
}
