﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include "Common.h"
#include "../Encoder/Encoder.h"

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
//! @brief エンコーダのクラスです。
//=============================================================================
class REncoder
{
protected:
    std::string m_DllPath; //!< エンコーダの DLL ファイルのパスです。
    HMODULE m_hDll; //!< エンコーダの DLL のインスタンスハンドルです。

public:
    nn::gfx::tool::texenc::GetDataSizeFunction GetDataSize; //!< フォーマットに応じたデータサイズ（バイト数）を取得する関数です。
    nn::gfx::tool::texenc::ConvertFormatFunction ConvertFormat; //!< 画像のフォーマットを変換する関数です。
    nn::gfx::tool::texenc::IsGpuEncodingAvailableFunction IsGpuEncodingAvailable; //!< 現在の環境で GPU によるエンコーディングが可能なら true を返す関数です。

public:
    //! @biref コンストラクタです。
    REncoder()
    : m_hDll(nullptr),
      GetDataSize(nullptr),
      ConvertFormat(nullptr),
      IsGpuEncodingAvailable(nullptr)
    {
    }

    //! @biref デストラクタです。
    ~REncoder()
    {
        Finalize();
    }

    //! @biref エンコーダの DLL ファイルのパスを設定します。
    //!
    //! @param[in] dllPath エンコーダの DLL ファイルのパスです。
    //!
    void SetDllPath(const char* dllPath)
    {
        m_DllPath = dllPath;
    }

    //! @biref エンコーダの DLL ファイルのパスを返します。
    std::string GetDllPath() const
    {
        return m_DllPath;
    }

    //! @biref 初期化します。
    //!
    //! @param[in] dllPath エンコーダの DLL ファイルのパスです。
    //!                    nullptr なら SetDllPath で設定したパスを使用します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus Initialize(const char* dllPath);

    //! @biref 終了します。
    void Finalize();

    //! @biref 初期化済みなら true を返します。
    bool IsInitialized() const
    {
        return (m_hDll != nullptr);
    }

    //! @biref エンコーダ用のイメージの次元を取得します（static 関数）。
    //!
    //! @param[in] dimension 次元です。
    //!
    //! @return エンコーダ用のイメージの次元を返します。
    //!
    static nn::gfx::tool::texenc::ImageDimension GetImageDimension(const FtxDimension dimension);
};

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

