﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DecodePng OutputPngFile
// CreateFramePixels

//=============================================================================
// include
//=============================================================================
#include "ImageConvert.h"
#include <atlbase.h> // CComPtr に必要

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================
const int PngFileSignatureBytes = 8; //!< ファイルシグネチャのバイト数です。
const int PngImageHeaderBytes = 0x19; //!< イメージヘッダのバイト数です。

//-----------------------------------------------------------------------------
//! @brief PNG ファイルが指定したタイプのチャンクを持つなら、
//!        チャンク先頭へのポインタを返します。
//!
//! @param[in] fileBuf PNG ファイルをリードしたバッファです。
//! @param[in] fileSize PNG ファイルのサイズです。
//! @param[in] chunkType 検索するチャンクのタイプです。
//!
//! @return チャンク先頭へのポインタ（見つからなければ nullptr）を返します。
//-----------------------------------------------------------------------------
const uint8_t* FindPngChunk(
    const void* fileBuf,
    const size_t fileSize,
    const std::string& chunkType
)
{
    const int SizeBytes = 4;
    const int TypeBytes = 4;
    const int CrcBytes = 4;

    const uint8_t* pData = reinterpret_cast<const uint8_t*>(fileBuf);
    const uint8_t* pEnd = pData + fileSize - (SizeBytes + TypeBytes);
    pData += PngFileSignatureBytes;
    while (pData < pEnd)
    {
        const uint32_t size = (pData[0] << 24) | (pData[1] << 16) | (pData[2] << 8) | pData[3];
        const std::string type = std::string(reinterpret_cast<const char*>(pData + 4), 4);
        //cerr << "type: " << type << ": " << hex << size << dec << endl;
        if (type == chunkType)
        {
            return pData;
        }
        pData += SizeBytes + TypeBytes + size + CrcBytes;
    }
    return nullptr;
}

//-----------------------------------------------------------------------------
//! @brief 16 bit/チャンネルのイメージデータを float 型に変換します。
//!
//! @param[in,out] pImageData float 型 RGBA を格納できるサイズのイメージデータへのポインタです。
//!                           先頭から 16 bit/チャンネルのイメージデータを格納しておきます。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//-----------------------------------------------------------------------------
void ConvertBit16ToFloat(void* pImageData, const int imageW, const int imageH)
{
    const size_t valueCount = static_cast<size_t>(R_RGBA_COUNT) * imageW * imageH;
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pImageData) +
        (valueCount - 1) * sizeof(uint16_t);
    float* pDst = reinterpret_cast<float*>(pImageData) + (valueCount - 1);
    for (size_t valueIdx = 0; valueIdx < valueCount; ++valueIdx)
    {
        const int value = (pSrc[1] << 8) | pSrc[0];
        *pDst-- = static_cast<float>(value) / 0xffff;
        pSrc -= sizeof(uint16_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief float 型の値を 16 bit/チャンネルのピクセル値に変換して設定します。
//!
//! @param[out] pDst 変換先ピクセルデータへのポインタです。
//! @param[in] fv float 型の値です。
//-----------------------------------------------------------------------------
void SetBit16PixelValue(uint8_t* pDst, const float fv)
{
    const int value = RClampValue(0x0000, 0xffff, RRound(fv * 0xffff));
    pDst[0] = static_cast<uint8_t>((value     ) & 0xff);
    pDst[1] = static_cast<uint8_t>((value >> 8) & 0xff);
}

//-----------------------------------------------------------------------------
//! @brief フレームの 1 ピクセルのデータを設定します。
//!
//! @param[in,out] ppDst 変換先ピクセルデータへのポインタへのポインタです。
//!                      この関数内でインクリメントされます。
//! @param[in,out] ppSrc 変換元ピクセルデータへのポインタへのポインタです。
//!                      この関数内でインクリメントされます。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//! @param[in] isFloat 変換元のデータ型が浮動小数点数なら true、整数なら false です。
//-----------------------------------------------------------------------------
void SetOneFramePixel(
    uint8_t** ppDst,
    const uint8_t** ppSrc,
    const bool hasAlpha,
    const bool isFloat
)
{
    uint8_t*& pDst = *ppDst;
    const uint8_t*& pSrc = *ppSrc;
    if (!isFloat)
    {
        // 8 bit/チャンネルの場合は BGR(A) の順に格納します。
        *pDst++ = pSrc[2]; // B
        *pDst++ = pSrc[1]; // G
        *pDst++ = pSrc[0]; // R
        if (hasAlpha)
        {
            *pDst++ = pSrc[3]; // A
        }
        pSrc += R_RGBA_COUNT;
    }
    else
    {
        // 16 bit/チャンネルの場合は RGB(A) の順に格納します。
        const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
        SetBit16PixelValue(pDst + 0, pSrcF32[0]); // R
        SetBit16PixelValue(pDst + 2, pSrcF32[1]); // G
        SetBit16PixelValue(pDst + 4, pSrcF32[2]); // B
        pDst += 6;
        if (hasAlpha)
        {
            SetBit16PixelValue(pDst, pSrcF32[3]); // A
            pDst += 2;
        }
        pSrc += R_RGBA_FLOAT_BYTES;
    }
}

//-----------------------------------------------------------------------------
//! @brief フレーム用のピクセルデータを作成します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] image 画像データです。
//! @param[in] isCubeVC 垂直十字キューブマップにするなら true、
//!                     水平十字キューブマップにするなら false を指定します。
//! @param[in] dstW 変換先の幅です。
//! @param[in] dstPixelBytes 変換先のピクセルあたりのバイト数です。
//! @param[in] dstFaceBytes 変換先のフェースあたりのバイト数です。
//-----------------------------------------------------------------------------
void CreateFramePixels(
    uint8_t* pDstPixels,
    const RImage& image,
    const bool isCubeVC,
    const int dstW,
    const size_t dstPixelBytes,
    const size_t dstFaceBytes
)
{
    memset(pDstPixels, 0xff, dstFaceBytes);
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(image.GetImageData());
    for (int iz = 0; iz < image.GetImageD(); ++iz)
    {
        int iFace = iz % RImage::CUBE_FACE_COUNT;
        int dstIx = 0;
        int dstIy = image.GetImageH() * iz;
        if (image.IsCubeMap())
        {
            // 元のイメージデータには +X, -X, +Y, -Y, -Z, +Z の順に格納されているので
            // -Z と +Z を入れ替えます。
            if (iFace == RImage::CUBE_FACE_PZ)
            {
                iFace = RImage::CUBE_FACE_NZ;
            }
            else if (iFace == RImage::CUBE_FACE_NZ)
            {
                iFace = RImage::CUBE_FACE_PZ;
            }
            const int* cubePoss = (!isCubeVC) ?
                RImage::CubeHcPositions[iFace] : RImage::CubeVcPositions[iFace];
            dstIx = cubePoss[0] * image.GetImageW();
            dstIy = cubePoss[1] * image.GetImageH();
            dstIy += (iz / RImage::CUBE_FACE_COUNT) * image.GetImageH() *
                ((!isCubeVC) ? RImage::CUBE_HC_COUNT_V : RImage::CUBE_VC_COUNT_V);
        }
        uint8_t* pDstTop = pDstPixels + (dstIx + dstIy * dstW) * dstPixelBytes;
        uint8_t* pDst = pDstTop;
        for (int iy = 0; iy < image.GetImageH(); ++iy)
        {
            for (int ix = 0; ix < image.GetImageW(); ++ix)
            {
                SetOneFramePixel(&pDst, &pSrc, image.HasAlpha(), image.IsFloat());
            }
            pDst += (dstW - image.GetImageW()) * dstPixelBytes;
        }

        // 垂直十字キューブマップは +Z 方向を上下左右反転します。
        if (image.IsCubeMap() && isCubeVC && iFace == RImage::CUBE_FACE_PZ)
        {
            RFlipFaceHv(pDstTop, image.GetImageW(), image.GetImageH(),
                dstW * dstPixelBytes, dstPixelBytes);
        }
    }
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief PNG ファイルのデータをデコードします。
//-----------------------------------------------------------------------------
RStatus RImage::DecodePng(WicUtility* pWicUtility, const uint8_t* fileBuf, const size_t fileSize)
{
    //-----------------------------------------------------------------------------
    // ファイルシグネチャをチェックします。
    if (fileSize < PngFileSignatureBytes ||
        fileBuf[0x00] != 0x89 ||
        fileBuf[0x01] != 0x50 ||
        fileBuf[0x02] != 0x4e ||
        fileBuf[0x03] != 0x47 ||
        fileBuf[0x04] != 0x0d ||
        fileBuf[0x05] != 0x0a ||
        fileBuf[0x06] != 0x1a ||
        fileBuf[0x07] != 0x0a)
    {
        return RStatus(RStatus::FAILURE, "PNG file is wrong: " + m_FilePath + " (file signature)"); // RShowError
    }

    //-----------------------------------------------------------------------------
    // イメージヘッダから情報を取得します。
    if (fileSize < PngFileSignatureBytes + PngImageHeaderBytes)
    {
        return RStatus(RStatus::FAILURE, "PNG file is wrong: " + m_FilePath + " (image header)"); // RShowError
    }
    const int bitSize = (fileBuf[0x18] == 0x10) ? 16 : 8;
    const int colorType = fileBuf[0x19];
    m_HasAlpha = ((colorType & 0x04) != 0 ||
        FindPngChunk(fileBuf, fileSize, "tRNS") != nullptr);
    m_IsFloat = (bitSize == 16);

    //-----------------------------------------------------------------------------
    // デコーダを作成します。
    RStatus status;
    IWICImagingFactory* pFactory = pWicUtility->GetFactory(&status);
    RCheckStatus(status);

    HRESULT hr;
    CComPtr<IWICStream> pStream;
    hr = pFactory->CreateStream(&pStream);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot create stream: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    hr = pStream->InitializeFromMemory(const_cast<uint8_t*>(fileBuf), static_cast<DWORD>(fileSize));
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot initialize stream: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    CComPtr<IWICBitmapDecoder> pDecorder;
    hr = pFactory->CreateDecoderFromStream(pStream, nullptr,
        WICDecodeMetadataCacheOnLoad, &pDecorder);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot create decoder: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    //-----------------------------------------------------------------------------
    // フレームを取得して、プレーンな RGBA フォーマットに変換します。
    CComPtr<IWICBitmapFrameDecode> pFrame;
    hr = pDecorder->GetFrame(0, &pFrame);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot get frame: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    CComPtr<IWICFormatConverter> pConverter;
    hr = pFactory->CreateFormatConverter(&pConverter);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot create converter: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    const WICPixelFormatGUID planeFormat = (bitSize == 8) ?
        GUID_WICPixelFormat32bppRGBA :
        GUID_WICPixelFormat64bppRGBA;
    hr = pConverter->Initialize(pFrame, planeFormat,
        WICBitmapDitherTypeNone, nullptr, 0.0, WICBitmapPaletteTypeMedianCut);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot initialize converter: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    //-----------------------------------------------------------------------------
    // コンバータから情報を取得します。
    UINT imageW;
    UINT imageH;
    pConverter->GetSize(&imageW, &imageH);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot get size from converter: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    m_ImageW = static_cast<int>(imageW);
    m_ImageH = static_cast<int>(imageH);

    //-----------------------------------------------------------------------------
    // イメージデータを取得します。
    const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    m_ImageDataSize = pixelBytes * m_ImageW * m_ImageH;
    m_pImageData = new uint8_t[m_ImageDataSize];

    const size_t srcPixelBytes = R_RGBA_COUNT * bitSize / 8;
    const size_t srcLineBytes = srcPixelBytes * imageW;
    const size_t srcFaceBytes = srcLineBytes * imageH;
    hr = pConverter->CopyPixels(nullptr,
        static_cast<UINT>(srcLineBytes), static_cast<UINT>(srcFaceBytes),
        reinterpret_cast<BYTE*>(m_pImageData));
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot copy pixels: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 16 bit/チャンネルなら float 型に変換します。
    if (m_IsFloat)
    {
        ConvertBit16ToFloat(m_pImageData, m_ImageW, m_ImageH);
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief PNG ファイルを出力します。
//-----------------------------------------------------------------------------
RStatus RImage::OutputPngFile(
    WicUtility* pWicUtility,
    const std::string& filePath,
    const bool isCubeVC
) const
{
    //-----------------------------------------------------------------------------
    // エンコーダを作成します。
    RStatus status;
    IWICImagingFactory* pFactory = pWicUtility->GetFactory(&status);
    RCheckStatus(status);

    HRESULT hr;
    CComPtr<IWICBitmapEncoder> pEncoder;
    hr = pFactory->CreateEncoder(GUID_ContainerFormatPng, nullptr, &pEncoder);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot create encoder: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 出力ストリームを作成します。
    CComPtr<IWICStream> pStream;
    hr = pFactory->CreateStream(&pStream);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot create stream: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    hr = pStream->InitializeFromFilename(RGetUnicodeFromShiftJis(filePath).c_str(), GENERIC_WRITE);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot initialize stream: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    hr = pEncoder->Initialize(pStream, WICBitmapEncoderNoCache);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot initialize encoder: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    //-----------------------------------------------------------------------------
    // フレームを作成します。
    CComPtr<IWICBitmapFrameEncode> pFrame;
    CComPtr<IPropertyBag2> pPropertyBag;
    hr = pEncoder->CreateNewFrame(&pFrame, &pPropertyBag);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot create new frame: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    //-----------------------------------------------------------------------------
    // エンコードオプションを設定します。
    // 現在はデフォルト設定を使用するので、何もしません。
    // デフォルト設定は「インターレースなし」「フィルタ指定なし」です。

    //-----------------------------------------------------------------------------
    // PNG ファイルの幅と高さを決定します。
    int dstW = m_ImageW;
    int dstH = m_ImageH;
    if (IsCubeMap())
    {
        if (!isCubeVC)
        {
            dstW *= CUBE_HC_COUNT_H;
            dstH *= CUBE_HC_COUNT_V;
        }
        else
        {
            dstW *= CUBE_VC_COUNT_H;
            dstH *= CUBE_VC_COUNT_V;
        }
        dstH *= RMax(static_cast<int>(m_ImageD / CUBE_FACE_COUNT), 1);
    }
    else if (m_Dimension == FtxDimension_3d      ||
             m_Dimension == FtxDimension_1dArray ||
             m_Dimension == FtxDimension_2dArray)
    {
        dstH *= m_ImageD;
    }

    //-----------------------------------------------------------------------------
    // フレームを初期化します。
    hr = pFrame->Initialize(pPropertyBag);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot initialize frame: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    hr = pFrame->SetSize(dstW, dstH);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot set frame size: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    const int bitSize = (m_IsFloat) ? 16 : 8;
    WICPixelFormatGUID frameFormat = (bitSize == 8) ?
        ((m_HasAlpha) ? GUID_WICPixelFormat32bppRGBA : GUID_WICPixelFormat24bppRGB) :
        ((m_HasAlpha) ? GUID_WICPixelFormat64bppRGBA : GUID_WICPixelFormat48bppRGB);
    hr = pFrame->SetPixelFormat(&frameFormat);
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot set pixel format: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    //-----------------------------------------------------------------------------
    // フレーム用のピクセルデータを作成します。
    const size_t dstCompCount = (m_HasAlpha) ? R_RGBA_COUNT : R_RGB_COUNT;
    const size_t dstPixelBytes = dstCompCount * bitSize / 8;
    const size_t dstLineBytes = dstPixelBytes * dstW;
    const size_t dstFaceBytes = dstLineBytes * dstH;
    uint8_t* pDstPixels = new uint8_t[dstFaceBytes];
    CreateFramePixels(pDstPixels, *this, isCubeVC, dstW, dstPixelBytes, dstFaceBytes);

    //-----------------------------------------------------------------------------
    // フレームにピクセルデータを設定して、コミットします。
    hr = pFrame->WritePixels(dstH, static_cast<UINT>(dstLineBytes),
        static_cast<UINT>(dstFaceBytes), pDstPixels);
    delete[] pDstPixels;
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot write pixels: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    hr = pFrame->Commit();
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot commit frame: " + GetWindowsSystemMessage(hr)); // RShowError
    }
    hr = pEncoder->Commit();
    if (FAILED(hr))
    {
        return RStatus(RStatus::FAILURE, "Cannot commit encoder: " + GetWindowsSystemMessage(hr)); // RShowError
    }

    return status;
}

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

