﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// 定数 FtxFormat
// RStatus OptionArgInfo RTimeMeasure RXMLElement
// RDataStream RBinDataStream
// RUserData
// RUnicodeCArray
// RFileBuf
// RCropRect

//=============================================================================
// include
//=============================================================================
#include <cassert>
#include <cmath>
#include <cstdint>
#include <ctime>
#include <iomanip>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <map>

#include <windows.h>
#include <shlwapi.h>

#include "Argument.h"

#ifndef NN_IMPLICIT
    #define NN_IMPLICIT
#endif

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
// 定数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// color
const int R_RGB_COUNT  = 3; //!< RGB の成分数です。
const int R_RGBA_COUNT = 4; //!< RGBA の成分数です。

const int R_RGB_BYTES  = 3; //!< RGB 各 8 ビット整数のピクセルのバイト数です。
const int R_RGBA_BYTES = 4; //!< RGBA 各 8 ビット整数のピクセルのバイト数です。

const int R_RGB_FLOAT16_BYTES  = 3 * sizeof(uint16_t); //!< RGB 各 16 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT16_BYTES = 4 * sizeof(uint16_t); //!< RGBA 各 16 ビット浮動小数点数のピクセルのバイト数です。

const int R_RGB_FLOAT_BYTES  = 3 * sizeof(float); //!< RGB 各 32 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT_BYTES = 4 * sizeof(float); //!< RGBA 各 32 ビット浮動小数点数のピクセルのバイト数です。

//-----------------------------------------------------------------------------
// math
const double R_SAME_TOLERANCE   = 1.0e-6;  //!< double 型の値を比較する際の許容値です。
const float  R_SAME_TOLERANCE_F = 1.0e-6f; //!< float 型の値を比較する際の許容値です。

const double R_M_PI = 3.14159265358979323846; //!< 円周率です。
const double R_M_RAD_TO_DEG = 180.0 / R_M_PI; //!< ラジアンを度数に変換するためのスケールです。
const double R_M_DEG_TO_RAD = R_M_PI / 180.0; //!< 度数をラジアンに変換するためのスケールです。

const int R_XYZ_COUNT  = 3; //!< XYZ の成分数です。
const int R_XYZW_COUNT = 4; //!< XYZW の成分数です。

//-----------------------------------------------------------------------------
// ファイルの拡張子
extern const std::string TgaExtension; //!< TGA ファイルの拡張子です。
extern const std::string PngExtension; //!< PNG ファイルの拡張子です。
extern const std::string DdsExtension; //!< DDS ファイルの拡張子です。
extern const std::string ExrExtension; //!< EXR ファイルの拡張子です。
extern const std::string FtxaExtension; //!< テクスチャー中間ファイル（アスキー形式）の拡張子です。
extern const std::string FtxbExtension; //!< テクスチャー中間ファイル（バイナリー形式）の拡張子です。
extern const std::string BntxExtension; //!< テクスチャーバイナリーファイルの拡張子です。

//-----------------------------------------------------------------------------
// output
#define R_ENDL "\r\n"

const int R_FLOAT_OUT_PRECISION = 9; //!< std::ostream::precision で指定する精度です。

const int R_BINARY_ALIGNMENT = 32; //!< バイナリー形式の中間ファイルのアライメントです。

//-----------------------------------------------------------------------------
// テクスチャー

//! @brief テクスチャーの次元を表す列挙型です。
enum FtxDimension
{
    FtxDimension_1d, //!< 一次元です。
    FtxDimension_2d, //!< 二次元です。
    FtxDimension_3d, //!< 三次元です。
    FtxDimension_CubeMap, //!< キューブマップイメージです。
    FtxDimension_1dArray, //!< 一次元配列です。
    FtxDimension_2dArray, //!< 二次元配列です。
    FtxDimension_2dMultisample, //<! マルチサンプルイメージです。
    FtxDimension_2dMultisampleArray, //!< マルチサンプルイメージ配列です。
    FtxDimension_CubeMapArray, //!< キューブマップイメージ配列です。

    FtxDimension_End
};

//! @brief テクスチャーフォーマットを表す列挙型です。
enum FtxFormat
{
    FtxFormat_Invalid,

    FtxFormat_Unorm_8,
    FtxFormat_Snorm_8,
    FtxFormat_Uint_8,
    FtxFormat_Sint_8,
    FtxFormat_Unorm_16,
    FtxFormat_Snorm_16,
    FtxFormat_Uint_16,
    FtxFormat_Sint_16,
    FtxFormat_Uint_32,
    FtxFormat_Sint_32,

    FtxFormat_Unorm_4_4,
    FtxFormat_Unorm_8_8,
    FtxFormat_Snorm_8_8,
    FtxFormat_Uint_8_8,
    FtxFormat_Sint_8_8,
    FtxFormat_Unorm_16_16,
    FtxFormat_Snorm_16_16,
    FtxFormat_Uint_16_16,
    FtxFormat_Sint_16_16,
    FtxFormat_Uint_32_32,
    FtxFormat_Sint_32_32,

    FtxFormat_Unorm_5_6_5,

    FtxFormat_Unorm_5_5_5_1,
    FtxFormat_Unorm_4_4_4_4,

    FtxFormat_Unorm_8_8_8_8,
    FtxFormat_Snorm_8_8_8_8,
    FtxFormat_Srgb_8_8_8_8,
    FtxFormat_Uint_8_8_8_8,
    FtxFormat_Sint_8_8_8_8,
    FtxFormat_Unorm_10_10_10_2 ,
    FtxFormat_Uint_10_10_10_2,
    FtxFormat_Unorm_16_16_16_16,
    FtxFormat_Snorm_16_16_16_16,
    FtxFormat_Uint_16_16_16_16,
    FtxFormat_Sint_16_16_16_16,
    FtxFormat_Uint_32_32_32_32,
    FtxFormat_Sint_32_32_32_32,

    FtxFormat_Float_16,
    FtxFormat_Float_32,
    FtxFormat_Float_16_16,
    FtxFormat_Float_32_32,
    FtxFormat_Float_11_11_10,
    FtxFormat_Float_16_16_16_16,
    FtxFormat_Float_32_32_32_32,

    FtxFormat_Unorm_Bc1,
    FtxFormat_Srgb_Bc1,
    FtxFormat_Unorm_Bc2,
    FtxFormat_Srgb_Bc2,
    FtxFormat_Unorm_Bc3,
    FtxFormat_Srgb_Bc3,
    FtxFormat_Unorm_Bc4,
    FtxFormat_Snorm_Bc4,
    FtxFormat_Unorm_Bc5,
    FtxFormat_Snorm_Bc5,

    FtxFormat_Ufloat_Bc6,
    FtxFormat_Float_Bc6,
    FtxFormat_Unorm_Bc7,
    FtxFormat_Srgb_Bc7,

    FtxFormat_Unorm_Etc1,
    FtxFormat_Unorm_Etc2,
    FtxFormat_Srgb_Etc2,
    FtxFormat_Unorm_Etc2_Mask,
    FtxFormat_Srgb_Etc2_Mask,
    FtxFormat_Unorm_Etc2_Alpha,
    FtxFormat_Srgb_Etc2_Alpha,

    FtxFormat_Unorm_Eac_11,
    FtxFormat_Snorm_Eac_11,
    FtxFormat_Unorm_Eac_11_11,
    FtxFormat_Snorm_Eac_11_11,

    FtxFormat_Unorm_Pvrtc1_2Bpp,
    FtxFormat_Srgb_Pvrtc1_2Bpp,
    FtxFormat_Unorm_Pvrtc1_4Bpp,
    FtxFormat_Srgb_Pvrtc1_4Bpp,
    FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp,
    FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp,
    FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp,
    FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp,
    FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp,
    FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp,
    FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp,
    FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp,

    FtxFormat_Unorm_Astc_4x4,
    FtxFormat_Srgb_Astc_4x4,
    FtxFormat_Unorm_Astc_5x4,
    FtxFormat_Srgb_Astc_5x4,
    FtxFormat_Unorm_Astc_5x5,
    FtxFormat_Srgb_Astc_5x5,
    FtxFormat_Unorm_Astc_6x5,
    FtxFormat_Srgb_Astc_6x5,
    FtxFormat_Unorm_Astc_6x6,
    FtxFormat_Srgb_Astc_6x6,
    FtxFormat_Unorm_Astc_8x5,
    FtxFormat_Srgb_Astc_8x5,
    FtxFormat_Unorm_Astc_8x6,
    FtxFormat_Srgb_Astc_8x6,
    FtxFormat_Unorm_Astc_8x8,
    FtxFormat_Srgb_Astc_8x8,
    FtxFormat_Unorm_Astc_10x5,
    FtxFormat_Srgb_Astc_10x5,
    FtxFormat_Unorm_Astc_10x6,
    FtxFormat_Srgb_Astc_10x6,
    FtxFormat_Unorm_Astc_10x8,
    FtxFormat_Srgb_Astc_10x8,
    FtxFormat_Unorm_Astc_10x10,
    FtxFormat_Srgb_Astc_10x10,
    FtxFormat_Unorm_Astc_12x10,
    FtxFormat_Srgb_Astc_12x10,
    FtxFormat_Unorm_Astc_12x12,
    FtxFormat_Srgb_Astc_12x12,

    FtxFormat_Uint_32_32_32,
    FtxFormat_Sint_32_32_32,
    FtxFormat_Float_32_32_32,

    FtxFormat_End
};

//! @brief テクスチャーの成分選択の成分を表す列挙型です。
enum FtxComponent
{
    FtxComponent_Red,   //!< R 成分です。
    FtxComponent_Green, //!< G 成分です。
    FtxComponent_Blue,  //!< B 成分です。
    FtxComponent_Alpha, //!< A 成分です。
    FtxComponent_Zero,  //!< 固定値 0 です。
    FtxComponent_One,   //!< 固定値 1 です。
    FtxComponent_Count  //!< 成分数です。
};

//! @brief テクスチャーの成分選択を表す型定義です。
typedef uint32_t FtxCompSel;

const int FtxCompSelShiftR = 24; //!< 成分選択値の R 成分のシフト数です。
const int FtxCompSelShiftG = 16; //!< 成分選択値の G 成分のシフト数です。
const int FtxCompSelShiftB =  8; //!< 成分選択値の B 成分のシフト数です。
const int FtxCompSelShiftA =  0; //!< 成分選択値の A 成分のシフト数です。
#define FTX_GET_COMP_SEL(r, g, b, a) (\
    (((r) & 0xff) << FtxCompSelShiftR) |\
    (((g) & 0xff) << FtxCompSelShiftG) |\
    (((b) & 0xff) << FtxCompSelShiftB) |\
    (((a) & 0xff) << FtxCompSelShiftA))

const FtxCompSel FtxCompSelNone = FTX_GET_COMP_SEL(FtxComponent_Zero, FtxComponent_Zero , FtxComponent_Zero, FtxComponent_Zero );
const FtxCompSel FtxCompSelRgba = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Green, FtxComponent_Blue, FtxComponent_Alpha);
const FtxCompSel FtxCompSelRrr1 = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Red  , FtxComponent_Red , FtxComponent_One  );
const FtxCompSel FtxCompSelRrrg = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Red  , FtxComponent_Red , FtxComponent_Green);
const FtxCompSel FtxCompSelRa01 = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Alpha, FtxComponent_Zero, FtxComponent_One  );

//! @brief テクスチャーのタイルモードを表す列挙型です。
enum FtxTileMode
{
    FtxTileMode_Linear,  //!< リニアーなタイルモードです。
    FtxTileMode_NX,      //!< NX 向けのタイルモードです。
    FtxTileMode_Invalid, //!< 不正なタイルモードです。
};

//! @brief テクスチャーのタイリング最適化を表す列挙型です。
enum FtxTileOptimize
{
    FtxTileOptimize_Performance, //!< パフォーマンス優先のタイリングです。
    FtxTileOptimize_Size,        //!< サイズ優先のタイリングです。
    FtxTileOptimize_SizeAuto,    //!< データ削減率に応じてサイズ優先のタイリングを選択します。
    FtxTileOptimize_Invalid
};

const int FtxDefaultTileSizeThreshold = 75; //!< タイリング最適化時のデータ削減率のしきい値のデフォルト値です。

//=============================================================================
//! @brief 未使用変数の警告を抑止するためのマクロです。
//=============================================================================
#define R_UNUSED_VARIABLE(Variable) (void)(&Variable);

//=============================================================================
//! @brief 配列型の定義です。
//=============================================================================

//! @brief char 型配列の定義です。
typedef std::vector<char> RCharArray;

//! @brief unsigned char 型配列の定義です。
typedef std::vector<unsigned char> RUCharArray;

//! @brief short 型配列の定義です。
typedef std::vector<short> RShortArray;

//! @brief unsigned short 型配列の定義です。
typedef std::vector<unsigned short> RUShortArray;

//! @brief int 型配列の定義です。
typedef std::vector<int> RIntArray;

//! @brief unsigned int 型配列の定義です。
typedef std::vector<unsigned int> RUIntArray;

//! @brief size_t 型配列の定義です。
typedef std::vector<size_t> RSizeTypeArray;

//! @brief float 型配列の定義です。
typedef std::vector<float> RFloatArray;

//! @brief double 型配列の定義です。
typedef std::vector<double> RDoubleArray;

//! @brief 文字列型配列の定義です。
typedef std::vector<std::string> RStringArray;

//! @brief ユニコード文字列型配列の定義です。
typedef std::vector<std::wstring> RWStringArray;

//! @brief bool 型配列の定義です。
typedef std::vector<bool> RBoolArray;

//=============================================================================
// デバッグ関連の関数です。
//=============================================================================

//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//!
void NoteTrace(const char* format, ...);

//=============================================================================
// 一般的な数学関連の関数です。
//=============================================================================

//! @brief [0, 360) の範囲に正規化した角度を返します。
//!
//! @param[in] ang 角度 [degree] です。
//!
//! @return [0, 360) の範囲に正規化した角度を返します。
//!
template <typename T>
inline T RNormalizeAngle(T ang)
{
    if (ang >= (T)360)
    {
        const int ishift = static_cast<int>(ang / (T)360);
        ang -= ishift * (T)360;
    }
    else if (ang < (T)0)
    {
        const int ishift = static_cast<int>(-ang / (T)360) + 1;
        ang += ishift * (T)360;
    }
    return ang;
}

//! @brief [-360, 360] の範囲にシフトした角度を返します。
//!
//! @param[in] ang 角度 [degree] です。
//!
//! @return [-360, 360] の範囲にシフトした角度を返します。
//!
template <typename T>
inline T RShiftAngle(T ang)
{
    if (ang < (T)(-360) || ang > (T)360)
    {
        int ishift = static_cast<int>(ang / (T)360);
        ang -= ishift * (T)360;
    }
    return ang;
}

inline bool RIsSame(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (a == b)
    {
        return true;
    }
    const double c = a - b;
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSame(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (a == b)
    {
        return true;
    }
    const float c = a - b;
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameRelative(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (a == b)
    {
        return true;
    }
    double c = a - b;
    if (a != 0.0 && b != 0.0)
    {
        c /= a;
    }
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameRelative(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (a == b)
    {
        return true;
    }
    float c = a - b;
    if (a != 0.0f && b != 0.0f)
    {
        c /= a;
    }
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameAngle(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (RIsSame(a, b, tolerance))
    {
        return true;
    }
    const double na = RNormalizeAngle(a);
    const double nb = RNormalizeAngle(b);
    return
        RIsSame(na, nb, tolerance) ||
        RIsSame( a, nb, tolerance) ||
        RIsSame(na,  b, tolerance);
}

inline bool RIsSameAngle(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (RIsSame(a, b, tolerance))
    {
        return true;
    }
    const float na = RNormalizeAngle(a);
    const float nb = RNormalizeAngle(b);
    return
        RIsSame(na, nb, tolerance) ||
        RIsSame( a, nb, tolerance) ||
        RIsSame(na,  b, tolerance);
}

//! @brief 絶対値を返します。
template <typename T>
inline T RAbs(const T x)
{
    return (x >= (T)0) ? x : -x;
}

//! @brief 四捨五入した値を返します。
template <typename T>
inline int RRound(const T x)
{
    return (x >= (T)0) ? static_cast<int>(x + (T)0.5) : static_cast<int>(x - (T)0.5);
}

//! @brief 小さいほうの値を返します。
template <typename T>
inline T RMin(const T a, const T b)
{
    return (a < b) ? a : b;
}

//! @brief 大きいほうの値を返します。
template <typename T>
inline T RMax(const T a, const T b)
{
    return (a > b) ? a : b;
}

//! @brief double 型の 0 に近い値を 0 にした値を返します。
inline double RSnapToZero(const double v)
{
    if (-R_SAME_TOLERANCE < v && v < R_SAME_TOLERANCE)
    {
        return 0.0;
    }
    else
    {
        return v;
    }
}

//! @brief float 型の 0 に近い値を 0 にした値を返します。
inline float RSnapToZero(const float v)
{
    if (-R_SAME_TOLERANCE_F < v && v < R_SAME_TOLERANCE_F)
    {
        return 0.0f;
    }
    else
    {
        return v;
    }
}

//! @brief 指定した範囲にクランプした値を返します。
template <typename T>
inline T RClampValue(const T valMin, const T valMax, const T val)
{
    if (val < valMin)
    {
        return valMin;
    }
    else if (val > valMax)
    {
        return valMax;
    }
    else
    {
        return val;
    }
}

//! @brief アライメントした値を返します。
template <typename T>
inline T RAlignValue(T val, const int align)
{
    return (T)((val + align - 1) & (~(align - 1)));
}

//! @brief アライメントしたポインタを返します。
template <typename T>
inline T* RAlignPointer(T* val, const int align)
{
    #ifdef _WIN64
    return (T*)(((unsigned __int64)val + align - 1) & (~(align - 1)));
    #else
    return (T*)(((unsigned __int32)val + align - 1) & (~(align - 1)));
    #endif
}

//! @brief 2 つの値を交換します。
template <typename T>
inline void RSwapValue(T& r1, T& r2)
{
    T tmpVal = r1;
    r1 = r2;
    r2 = tmpVal;
}

//! @brief 2 のべき乗の数なら true を返します。
//!
//! @param[in] number 判定する数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 2 のべき乗の数なら true を返します。
//!
bool RIsPower2Number(const int number, const bool includeOne = true);

//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//!
//! @param[in] number 数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 指定した数以上で最小の 2 のべき乗の数を返します。
//!
int RGetPower2Number(const int number, const bool includeOne = true);

//=============================================================================
// 文字列関連の関数です。
//=============================================================================

//! @brief 文字が半角スペースかタブなら true を返します。
inline bool RIsSpace(const char c)
{
    return (c == ' ' || c == '\t');
}

bool RIsShiftJisFirstByte(const unsigned char val);
std::string RGetUtf8FromShiftJis(const std::string& src);
std::string RGetShiftJisFromUtf8(const std::string& src);
std::wstring RGetUnicodeFromShiftJis(const std::string& src);
std::string RGetShiftJisFromUnicode(const std::wstring& src);

//! @brief ユニコード C 配列を RStringArray に変換します。
//!
//! @param[out] dsts 変換先の文字列配列です。
//! @param[in] srcs ユニコード C 配列です。最後に nullptr を格納します。
//!
//! @return 文字列数を返します。
//!
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[]);

std::string RGetLowerCaseString(const std::string& src);
std::string RGetUpperCaseString(const std::string& src);
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2);
std::string RTrimLeftString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimRightString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimString(const std::string& src, const char* targets = " \t\r\n");
std::string RDequoteString(const std::string& src, const char quote = '\"');

//! @brief Shift-JIS 文字列の改行コード CR + LF を LF に置換します。
std::string RReplaceCrLfToLf(const std::string& src);

//! @brief 文字列をトークンに分解します。
//!
//! @param[out] tokens トークン配列を格納します。
//! @param[in] input 文字列です。
//! @param[in] delims 区切り記号です。
//!
//! @return トークン数を返します。
//!
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims = " \t\r\n");

//! @brief int 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。nullptr なら "%d" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const int num, const char* format = nullptr);

//! @brief uint32_t 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。nullptr なら "%u" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const uint32_t num, const char* format = nullptr);

//! @brief ポインタを文字列に変換します。
//!
//! @param[in] ptr ポインタです。
//! @param[in] format フォーマット文字列です。nullptr なら "%p" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetPointerString(const void* ptr, const char* format = nullptr);

//! @brief 文字列の末尾の整数値を取得します。
//!
//! @param[out] pNumber 整数値を格納します。文字列の末尾が数値でない場合は -1 を格納します。
//! @param[in] str 文字列です。
//!
//! @return 文字列の末尾の数値を削除した文字列を返します。
//!
std::string RGetSuffixNumberFromString(int* pNumber, const std::string& str);

//! @brief Shift-JIS 文字列を XML 用にエンコードします。
std::string REncodeXmlString(const std::string& src);

//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
std::string RDecodeXmlString(const std::string& src);

//! @brief uint8_t 型のデータ文字列をデコードします。
//!
//! @param[out] pDst データを格納するバッファの先頭アドレスです。
//! @param[in] str スペースで 16 進数値を区切ったデータ文字列です。
//! @param[in] count デコードする値の個数です。
//!
void RDecodeDataStringU8(void* pDst, const std::string& str, const size_t count);

//! @brief float 型のデータ文字列をデコードします。
//!
//! @param[out] pDst データを格納するバッファの先頭アドレスです。
//! @param[in] str スペースで f32 値を区切ったデータ文字列です。
//! @param[in] count デコードする値の個数です。
//!
void RDecodeDataStringF32(void* pDst, const std::string& str, const size_t count);

//=============================================================================
//! @brief ユニコード C 配列のクラスです。
//=============================================================================
class RUnicodeCArray
{
protected:
    RWStringArray m_WStrings; //!< ユニコード文字列配列です。
    const wchar_t** m_CArray; //!< ユニコード C 配列です。

public:
    //! コンストラクタです（RStringArray から生成）。
    explicit RUnicodeCArray(const RStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_WStrings.push_back(RGetUnicodeFromShiftJis(srcs[iStr]));
        }
        CreateCArray();
    }

    //! コンストラクタです（RWStringArray から生成）。
    explicit RUnicodeCArray(const RWStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_WStrings.push_back(srcs[iStr]);
        }
        CreateCArray();
    }

    //! ユニコード C 配列を作成します。
    void CreateCArray()
    {
        const int strCount = static_cast<int>(m_WStrings.size());
        m_CArray = new const wchar_t*[strCount + 1];
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_CArray[iStr] = m_WStrings[iStr].c_str();
        }
        m_CArray[strCount] = nullptr; // 最終要素を nullptr にします。
    }

    //! デストラクタです。
    virtual ~RUnicodeCArray()
    {
        //std::cerr << "~RUnicodeCArray(): " << m_WStrings.size() << std::endl;
        delete[] m_CArray;
    }

    //! ユニコード C 配列を返します。
    const wchar_t** GetCArray() const
    {
        return m_CArray;
    }
};

//=============================================================================
// ファイルパス関連の関数です。
//=============================================================================
std::string RGetWindowsFilePath(const std::string& path);
std::string RGetUnixFilePath(const std::string& path);
std::string RGetFileNameFromFilePath(const std::string& path);
std::string RGetFolderFromFilePath(const std::string& path);
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower = true);
std::string RGetNoExtensionFilePath(const std::string& path);
bool RIsFullFilePath(const std::string& path);
std::string RGetFullFilePath(const std::string& path, const bool isUnix);

//! @brief 末尾にスラッシュを追加したファイルパスを返します。
std::string RGetFilePathWithEndingSlash(const std::string& path);

//! @brief 末尾にバックスラッシュを追加したファイルパスを返します。
std::string RGetFilePathWithEndingBackslash(const std::string& path);

//=============================================================================
// ファイル関連の関数です。
//=============================================================================
bool RFileExists(const std::string& path);
bool RFolderExists(const std::string& path);

//=============================================================================
// ファイル出力関連の関数です。
//=============================================================================
void RInitOutStreamFormat(std::ostream& os);
void ROutUtf8Bom(std::ostream& os);
void ROutUtf16Bom(std::ostream& os);
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr);
const char* RTab(const int size = 1);
const char* RBoolStr(const bool value);

//! @brief パディングを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] size サイズ（バイト数）です。
//! @param[in] value パディングに使用する値です。
//!
void ROutPadding(std::ostream& os, const size_t size, const char value = '\0');

//! @brief C 言語のソース形式でバイト型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 値のインデントに必要なタブの数です。
//! @param[in] array バイト型のデータ値配列です。
//! @param[in] column 列数です。
//!
void ROutCSourceData(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
);

//=============================================================================
// 配列関連の関数です。
//=============================================================================

//! @brief std::vector 配列から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
template <typename T>
int RFindValueInArray(const std::vector<T>& array, const T& value)
{
    const int size = static_cast<int>(array.size());
    for (int index = 0; index < size; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//=============================================================================
// メモリー関連の関数です。
//=============================================================================

//! @brief オブジェクトへのポインタが nullptr でなければ delete して nullptr を代入します。
//!
//! @param[in,out] ptr オブジェクトへのポインタです。
//!
template <typename T>
inline void RFreeAndClearObject(T*& ptr)
{
    if (ptr != nullptr)
    {
        delete ptr;
        ptr = nullptr;
    }
}

//! @brief 配列へのポインタが nullptr でなければ delete[] して nullptr を代入します。
//!
//! @param[in,out] ptr 配列へのポインタです。
//!
template <typename T>
inline void RFreeAndClearArray(T*& ptr)
{
    if (ptr != nullptr)
    {
        delete[] ptr;
        ptr = nullptr;
    }
}

//! @brief メモリーからリトルエンディアンの値を取得します。
template <typename T>
void RGetMemLittle(T& value, const void* pMemory)
{
    memcpy(&value, pMemory, sizeof(T));
    #if 0
    // swap for big endian
    char* p1 = reinterpret_cast<char*>(&value);
    char* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int iByte = 0; iByte < byteSizeHalf; ++iByte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリーからビッグエンディアンの値を取得します。
template <typename T>
void RGetMemBig(T& value, const void* pMemory)
{
    memcpy(&value, pMemory, sizeof(T));
    #if 1
    // swap for little endian
    char* p1 = reinterpret_cast<char*>(&value);
    char* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int iByte = 0; iByte < byteSizeHalf; ++iByte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリーからリトルエンディアンの short 値を取得します。
inline short RGetMemLittleShort(const void* pMemory)
{
    short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからリトルエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemLittleUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからリトルエンディアンの int 値を取得します。
inline int RGetMemLittleInt(const void* pMemory)
{
    int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからリトルエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemLittleUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの short 値を取得します。
inline short RGetMemBigShort(const void* pMemory)
{
    short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemBigUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの int 値を取得します。
inline int RGetMemBigInt(const void* pMemory)
{
    int value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemBigUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemBig(value, pMemory);
    return value;
}

//=============================================================================
// Base64 関連の関数です。
//=============================================================================

//! @brief Base64 形式にエンコードした際のサイズを取得します。
//!
//! @param[in] dataSize 元データのサイズです。
//!
//! @return Base64 形式にエンコードした際のサイズを返します。
//!
size_t GetBase64EncodedSize(const size_t dataSize);

//! @brief データを Base64 形式にエンコードします。
//!        Base64 形式データの末尾に \0 は含まれません。
//!
//! @param[out] base64Buf Base64 形式データを格納するバッファーへのポインターです。
//!                       GetBase64EncodedSize が返すサイズ以上で確保されている必要があります。
//! @param[in] dataBuf 元データへのポインターです。
//! @param[in] dataSize 元データのサイズです。
//!
//! @return エンコード後のサイズを返します。
//!
size_t EncodeToBase64(void* base64Buf, const void* dataBuf, const size_t dataSize);

//! @brief Base64 形式のデータをデコードした際のサイズを取得します。
//!
//! @param[in] base64Buf Base64 形式データへのポインターです。
//! @param[in] base64Size Base64 形式のサイズです。
//!
//! @return Base64 形式のデータをデコードした際のサイズを返します。
//!
size_t GetBase64DecodedSize(const void* base64Buf, const size_t base64Size);

//! @brief Base64 形式のデータをデコードします。
//!
//! @param[out] dataBuf デコードしたデータを格納するバッファーへのポインターです。
//!                     GetBase64DecodedSize が返すサイズ以上で確保されている必要があります。
//! @param[in] base64Buf Base64 形式データへのポインターです。
//! @param[in] base64Size Base64 形式のサイズです。
//!
//! @return デコード後のサイズを返します。
//!
size_t DecodeFromBase64(void* dataBuf, const void* base64Buf, const size_t base64Size);

//=============================================================================
// 一般的な Windows 関連の関数です。
//=============================================================================

//! @brief 環境変数の値を取得します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数の値を返します。環境変数が定義されていない場合は空文字を返します。
//!
std::string RGetEnvVar(const char* name);

//! @brief PC のユーザー名を返します。
std::string RGetUserName();

//! @brief PC のコンピュータ名を返します。
std::string RGetComputerName();

//! @brief Windows のシステムメッセージ文字列を取得します。
//!
//! @param[in] messageId メッセージ ID です。
//!
//! @return Windows のシステムメッセージ文字列を返します。
//!
std::string GetWindowsSystemMessage(const int messageId);

//! @brief Windows の最後のエラーのメッセージ文字列を取得します。
//!
//! @return Windows の最後のエラーのメッセージ文字列を返します。
//!
std::string GetWindowsLastErrorMessage();

//! @brief SYSTEMTIME から time_t を取得して返します。
time_t RGetTimeT(const SYSTEMTIME& st);

//! @brief ISO 8601 に準拠した書式で現在の日時を返します。
std::string RGetDateTimeString();

//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//!
//! @param[in] hModule モジュールのハンドルです
//!                    nullptr なら現在のプロセスを作成するために使われたファイルのパスを取得します。
//!
//! @return 実行ファイルのフルパスを返します。
//!
std::string RGetModuleFileName(HMODULE hModule);

//! @brief ユーザーが使用できるアドレス空間の空き容量を返します。
size_t RGetAvailableVirtualMemorySize();

//=============================================================================
//! @brief 処理結果ステータスのクラスです。
//=============================================================================
class RStatus
{
public:
    //! 処理結果ステータスコードを表す列挙型です。
    enum RStatusCode
    {
        SUCCESS = 0, //!< 成功です。
        FAILURE      //!< 失敗です。
    };

private:
    RStatusCode m_Code; //!< 処理結果ステータスコードです。
    std::string m_Message; //!< 失敗メッセージ文字列です。

public:
    RStatus() : m_Code(SUCCESS) {}
    NN_IMPLICIT RStatus(const RStatusCode code) : m_Code(code) {}
    RStatus(const RStatusCode code, const std::string& message) :
        m_Code(code), m_Message(message) {}
    NN_IMPLICIT operator bool() const
    {
        return (m_Code == SUCCESS);
    }
    bool operator==(const RStatus& other) const
    {
        return (m_Code == other.m_Code);
    }
    bool operator!=(const RStatus& other) const
    {
        return (m_Code != other.m_Code);
    }
    bool operator==(const RStatusCode code) const
    {
        return (m_Code == code);
    }
    bool operator!=(const RStatusCode code) const
    {
        return (m_Code != code);
    }
    std::string GetMessage() const
    {
        return m_Message;
    }
};

//-----------------------------------------------------------------------------
//! @brief 処理結果ステータスが成功でなければ return するマクロです。
//-----------------------------------------------------------------------------
#define RCheckStatus(status)        \
{                                   \
    if (status != RStatus::SUCCESS) \
    {                               \
        return status;              \
    }                               \
}

//=============================================================================
//! @brief ジョブ引数の構造体です。
//=============================================================================
struct JobArgument
{
    RStringArray options; //!< オプション配列です。
    RStringArray filePaths; //!< 入力ファイルパス配列です。
};

//=============================================================================
//! @brief コンバーター引数の構造体です。
//=============================================================================
struct ConverterArgument
{
    RStringArray globalOptions; //!< グローバルオプション配列です。
    std::vector<JobArgument> jobArgs; //!< ジョブ引数配列です。
};

//=============================================================================
//! @brief オプション引数情報の構造体です。
//=============================================================================
struct OptionArgInfo
{
public:
    char shortName; //!< ショートネームです。ショートネームなしなら空白です。
    const char* longName; //!< ロングネームです。nullptr なら配列の最後を表します。末尾の # は整数値を表します。
    bool requiresValue; //!< 値を必要とするなら true です。
};

//=============================================================================
// 引数関連の関数です。
//=============================================================================

//! @brief テキストファイルの 1 行から引数のトークンを取得します。
//!
//! @param[in,out] tokens 取得した引数のトークンを追加する配列です。
//! @param[in] lineStr 1 行の文字列です。
//!
void GetArgTokensFromLine(RStringArray& tokens, const std::string& lineStr);

//! @brief テキストファイルから引数のトークンを取得します。
//!
//! @param[in,out] tokens 取得した引数のトークンを追加する配列です。
//! @param[in] filePath テキストファイルのパスです。
//!
//! @return 処理結果を返します。
//!
RStatus GetArgTokensFromFile(RStringArray& tokens, const std::string& filePath);

//! @brief 引数配列からオプション値を取得します。
//!
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//!
//! @return 処理結果を返します。
//!
RStatus GetOptionArgValue(
    std::string* pValue,
    int* pArgIdx,
    const RStringArray& args
);

//! @brief 引数ファイルが指定されていればファイル内の引数を展開します。
//!
//! @param[out] pDstArgs 引数配列を格納します。
//! @param[in] srcArgs 展開前の引数配列です。
//!
//! @return 処理結果を返します。
//!
RStatus ExpandArguments(RStringArray* pDstArgs, const RStringArray& srcArgs);

//! @brief ジョブリストファイルを解析します。
//!
//! @param[in,out] pConvArg コンバーター引数へのポインタです。
//! @param[in] commonJobArg 共通ジョブ引数です。
//! @param[in] jobListFilePath ジョブリストファイルのパスです。
//! @param[in] isGlobalOptionArgFunc 引数がグローバルオプションなら true を返す関数です。
//! @param[in] isNextArgJobOptionValueFunc 指定した引数の次の引数がジョブオプションの値なら true を返す関数です。
//!
//! @return 処理結果を返します。
//!
RStatus ParseJobListFile(
    ConverterArgument* pConvArg,
    const JobArgument& commonJobArg,
    const std::string& jobListFilePath,
    IsGlobalOptionArgFunc isGlobalOptionArgFunc,
    IsNextArgJobOptionValueFunc isNextArgJobOptionValueFunc
);

//! @brief 引数からオプション引数情報へのポインタを取得します。
//!
//! @param[out] pIsLongName ロングネームなら true、ショートネームなら false を格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @return オプション引数情報へのポインタを返します。
//!         該当するオプション引数情報がなければ nullptr を返します。
//!
const OptionArgInfo* GetOptionArgInfo(
    bool* pIsLongName,
    int* pSuffixNum,
    const OptionArgInfo* optionArgInfos,
    const std::string& arg
);

//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//!
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//!
bool IsNextArgOptionValue(const OptionArgInfo* optionArgInfos, const std::string& arg);

//! @brief オプション引数を解析します。
//!
//! @param[out] ppInfo オプション引数情報へのポインタを格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//! @param[in] optionArgInfos オプション引数情報配列です。
//!
//! @return 処理結果を返します。
//!
RStatus ParseOptionArg(
    const OptionArgInfo** ppInfo,
    int* pSuffixNum,
    std::string* pValue,
    int* pArgIdx,
    const RStringArray& args,
    const OptionArgInfo* optionArgInfos
);

//=============================================================================
//! @brief 文字列と値のクラスです。
//=============================================================================
template <typename T>
class RStringAndValue
{
public:
    const char* m_String; //!< 文字列へのポインタです。
    T m_Value; //!< 値です。
};

//! @brief 値に対応する文字列を取得します。
//!
//! @param[in] svs 文字列と値配列です。配列の最後の文字列は nullptr である必要があります。
//! @param[in] value 値です。
//! @param[in] alt 値に対応する文字列が存在しなかった場合に返す文字列です。
//!
//! @return 文字列を返します。
//!
template <typename T>
std::string RGetStringFromValue(const RStringAndValue<T>* svs, const T value, const char* alt = "")
{
    for (const RStringAndValue<T>* pSv = svs; pSv->m_String != nullptr; ++pSv)
    {
        if (value == pSv->m_Value)
        {
            return pSv->m_String;
        }
    }
    return alt;
}

//! @brief 文字列に対応する値を取得します。
//!
//! @param[in] svs 文字列と値配列です。配列の最後の文字列は nullptr である必要があります。
//! @param[in] str 文字列です。
//! @param[in] alt 文字列に対応する値が存在しなかった場合に返す値です。
//!
//! @return 値を返します。
//!
template <typename T>
T RGetValueFromString(const RStringAndValue<T>* svs, const std::string& str, const T alt)
{
    for (const RStringAndValue<T>* pSv = svs; pSv->m_String != nullptr; ++pSv)
    {
        if (str == pSv->m_String)
        {
            return pSv->m_Value;
        }
    }
    return alt;
}

//=============================================================================
//! @brief 時間計測のクラスです。
//=============================================================================
class RTimeMeasure
{
protected:
    std::clock_t m_StartClock; //!< 計測開始時のクロックです。

public:
    //! @brief コンストラクタです。
    RTimeMeasure()
    {
        Reset();
    }

    //! 計測開始時間を現在のクロックに設定します。
    void Reset()
    {
        m_StartClock = std::clock();
    }

    //! 計測開始時からの秒数を取得します。
    float GetSec() const
    {
        return static_cast<float>(std::clock() - m_StartClock) / CLOCKS_PER_SEC;
    }

    //! 計測開始時からのミリ秒数を取得します。
    float GetMilliSec() const
    {
        return GetSec() * 1000.0f;
    }
};

//=============================================================================
//! @brief XML 要素のクラスです。
//=============================================================================
class RXMLElement
{
public:
    typedef std::map<std::string, std::string> AttrMap;
    typedef void (*ErrorFunc)(const char* message, void* pUserData);

    std::string name;
    AttrMap attributes;
    std::string text;
    std::vector<RXMLElement> nodes;

    ErrorFunc m_ErrorFunc; //!< エラー表示関数です。
    void* m_pErrorUserData; //!< エラー表示関数に渡すユーザーデータのポインタです。

private:
    //! @brief 属性を解析してノードに記録します。
    //!
    //! @param[in] src 要素の文字列です。
    //! @param[in] end 要素の終端の文字です。
    //!
    //! @return 要素の終端までの文字数を返します。
    //!
    int LoadAttributes(const char* src, const char end);

    //! @brief エラーを表示します。
    //!
    //! @param[in] message エラーメッセージです。
    //!
    void DisplayError(const std::string& message) const;

public:
    //! コンストラクタです。
    RXMLElement(ErrorFunc errorFunc = nullptr, void* pErrorUserData = nullptr)
    : m_ErrorFunc(errorFunc),
      m_pErrorUserData(pErrorUserData)
    {
    }

    //! コピーコンストラクタです。
    RXMLElement(const RXMLElement &r)
    {
        operator=(r);
    }

    //! 再初期化します。
    void Clear()
    {
        name = "";
        text = "";
        attributes.clear();
        nodes.clear();
    }

    //! 代入演算子です。
    RXMLElement& operator=(const RXMLElement& r);

    //! @brief 下層ノードを新規作成します。
    //!
    //! @return 新規作成した要素のポインタを返します。
    //!
    RXMLElement* CreateElement()
    {
        nodes.push_back(RXMLElement(m_ErrorFunc, m_pErrorUserData));
        return &nodes[nodes.size() - 1];
    }

    //! @brief 指定のパスに該当する最初の要素のポインタを取得します。
    //!
    //! @param[in] path パスです。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 要素のポインタを返します。見つからなかった場合は nullptr を返します。
    //!
    const RXMLElement* FindElement(const std::string& path, const bool showError = true) const
    {
        std::vector<const RXMLElement*> list;
        FindElements(list, path, showError, 1);
        return (list.size() > 0) ? list[0] : nullptr;
    }

    //! @brief 指定のパスに該当する要素を指定個数分検索します。
    //!
    //! @param[out] list 要素のポインタ配列です。
    //! @param[in] path パスです。
    //! @param[in] maxCount 取得する最大要素数です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    void FindElements(
        std::vector<const RXMLElement*>& list,
        const std::string& path,
        const bool showError = true,
        const int maxCount = 0x7fffffff
    ) const;

    //! @brief 属性の値を取得します。
    //!
    //! @param[in] tag 属性名です。
    //! @param[in] alt 属性が見つからなかった場合に返す値です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 属性の値を返します。
    //!
    std::string GetAttribute(
        const std::string& tag,
        const char* alt = "",
        const bool showError = true
    ) const
    {
        AttrMap::const_iterator it = attributes.find(tag);
        if (showError && it == attributes.end())
        {
            DisplayError("Attribute is not found: " + tag + " in <" + name + ">");
        }
        return (it != attributes.end()) ? it->second : alt;
    }

    //! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
    //!
    //! @param[in,out] decl 宣言要素です。宣言要素の属性を取得しないなら nullptr を指定します。
    //! @param[in] xmlDoc XML 文書です。
    //!
    void LoadDocument(RXMLElement* decl, const std::string& xmlDoc);

    //! @brief XML 文書を取得します。
    //!        要素、属性、平データ、下層ノードの内容を再帰的に文字列に出力します。
    //!
    //! @param[in] tabCount 要素のインデントに必要なタブの数です。
    //! @param[in] lf 改行コードです。
    //!
    //! @return XML 文書の文字列を返します。
    //!
    std::string GetDocument(const int tabCount = 0, const char* lf = "\n") const;

    //! @brief エラー表示関数を設定します。
    //!
    //! @param[in] func エラー表示関数です。
    //! @param[in] pUserData エラー表示関数に渡すユーザーデータのポインタです。
    //!
    void SetErrorFunc(ErrorFunc func, void* pUserData)
    {
        m_ErrorFunc = func;
        m_pErrorUserData = pUserData;
    }
};

//=============================================================================
// NintendoSDK 関連の関数です。
//=============================================================================

//! @brief グラフィックスツールの exe および dll ファイルのフルパスを返します。
//!        指定したモジュールのフォルダーにファイル名を追加して返します。
//!        32bit 版の場合、モジュールのフォルダーの下の x86 フォルダーに
//!        ファイルが存在すればそのパスを優先的に返します。
//!
//! @param[in] fileName exe (dll) ファイル名です。
//! @param[in] hModule モジュールのハンドルです
//!                    nullptr なら現在のプロセスを作成するために使われた exe ファイルのフォルダーを参照します。
//!
//! @return exe (dll) ファイルのフルパスを返します。
//!
std::string RGetGraphicsToolsExeDllFullPath(const std::string& fileName, HMODULE hModule);

//! @brief 中間ファイルバージョンを整数（major * 100 + minor）で返します。
//!        1.0.2 -> 100
//!        1.3.4 -> 103
//!
//! @return 中間ファイルバージョンを整数で返します。
//!
int RGetNw4f3difVerAsInt(const std::string& value);

//=============================================================================
// @brief データ列のクラスです。
//=============================================================================
class RDataStream
{
public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, BYTE, STRING, WSTRING };

    //! バイナリー形式のヘッダサイズです。
    enum { HEADER_SIZE = 0x20 };

    Type m_Type; //!< 値の型です。
    int m_Column; //!< アスキー出力時の列数です。
    RFloatArray m_FloatValues; //!< 浮動小数点数値の配列です。
    RIntArray m_IntValues; //!< 整数値の配列です。
    RUCharArray m_ByteValues; //!< バイト値の配列です。
    std::string m_StringValue; //!< 文字列値（Shift-JIS）です。

public:
    //! @brief 浮動小数点数値用のコンストラクタです。
    //!
    //! @param[in] array 浮動小数点数値の配列です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const RFloatArray& array, const int column = 4)
    : m_Type(FLOAT),
      m_Column(column)
    {
        m_FloatValues = array;
    }

    //! @brief 整数値用のコンストラクタです。
    //!
    //! @param[in] array 整数値の配列です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const RIntArray& array, const int column = 4)
    : m_Type(INT),
      m_Column(column)
    {
        m_IntValues = array;
    }

    //! @brief バイト値用のコンストラクタです。
    //!
    //! @param[in] array 配列の先頭の値へのポインタです。
    //! @param[in] bytes 配列のバイト数です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const uint8_t* array, const size_t bytes, const int column = 4);

    //! @brief 文字列値のコンストラクタです。
    //!
    //! @param[in] str 文字列値（Shift-JIS）です。
    //! @param[in] isWstring ASCII 文字列として出力するなら false、
    //!                      Unicode 文字列として出力するなら true を指定します。
    //!
    RDataStream(const std::string& str, const bool isWstring = false)
    : m_Column(0)
    {
        m_Type = (isWstring) ? WSTRING : STRING;
        m_StringValue = str;
    }

    //! @brief Unicode 文字列値のコンストラクタです。
    //!
    //! @param[in] wstr Unicode 文字列値です。
    //!
    explicit RDataStream(const std::wstring& wstr)
    : m_Type(WSTRING),
      m_Column(0)
    {
        m_StringValue = RGetShiftJisFromUnicode(wstr);
    }

    //! バイト値を追加します。
    //!
    //! @param[in] array 配列の先頭の値へのポインタです。
    //! @param[in] bytes 配列のバイト数です。
    //!
    void Append(const uint8_t* array, const size_t bytes);

    //! バイト値のパディングを追加します。
    //!
    //! @param[in] size パディングのバイト数です。
    //! @param[in] value パディングに使用する値です。
    //!
    void AppendPadding(const size_t bytes, const char value = '\0')
    {
        if (bytes > 0)
        {
            uint8_t* buf = new uint8_t[bytes];
            memset(buf, value, bytes);
            Append(buf, bytes);
            delete[] buf;
        }
    }

    //! @brief データへのポインターを取得します。
    //!
    //! @return データへのポインターを返します。
    //!
    const void* GetDataPtr() const;

    //! 値の個数を取得します。
    const size_t GetCount() const
    {
        return
            (m_Type == FLOAT) ? m_FloatValues.size() :
            (m_Type == INT  ) ? m_IntValues.size()   :
            (m_Type == BYTE ) ? m_ByteValues.size()  :
            1; // STRING, WSTRING
    }

    //! バイナリー形式の実データサイズ（バイト数）を取得します。
    const size_t GetSize() const
    {
        return
            (m_Type == FLOAT ) ? m_FloatValues.size() * sizeof(float) :
            (m_Type == INT   ) ? m_IntValues.size() * sizeof(int)     :
            (m_Type == BYTE  ) ? m_ByteValues.size()                  :
            (m_Type == STRING) ? m_StringValue.size() + 1             : // 終端文字を含むので +1 します。
            (RGetUnicodeFromShiftJis(m_StringValue).size() + 1) * sizeof(wchar_t); // WSTRING
                                                                                   // 終端文字を含むので +1 します。
    }

    //! ヘッダを含むバイナリー形式のアライメントしたデータサイズ（バイト数）を取得します。
    const size_t GetBinarySize() const
    {
        return RAlignValue(HEADER_SIZE + GetSize(), R_BINARY_ALIGNMENT);
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <stream> 要素のインデントに必要なタブの数です。
    //! @param[in] index <stream_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! @brief バイナリー形式で出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //!
    void OutBinary(std::ostream& os) const;
};

//! @brief データ列配列の定義です。
typedef std::vector<RDataStream> RDataStreamArray;

//! @brief データ列配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] dataStreams データ列配列です。
//!
void ROutDataStreams(std::ostream& os, const RDataStreamArray& array);

//! @brief データ列配列をバイナリー形式で出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] dataStreams データ列配列です。
//! @param[in] outZeroPadding バイナリー形式の後半部の前のゼロパディングを出力するなら true を指定します。
//!
void ROutBinaryDataStreams(
    std::ostream& os,
    const RDataStreamArray& dataStreams,
    const bool outZeroPadding
);

//! @brief 中間ファイルの先頭からバイナリーデータまでのオフセットを取得します。
//!
//! @param[in] fileBuf 中間ファイルをリードしたメモリーのポインタです。
//! @param[in] fileSize 中間ファイルのサイズ（バイト数）です。
//!
//! @return バイナリーデータまでのオフセットを返します。
//!         バイナリーデータがない場合は fileSize を返します。
//!
size_t RGetBinaryOffset(const void* fileBuf, const size_t fileSize);

//=============================================================================
// @brief バイナリーデータ列のクラスです。
//=============================================================================
class RBinDataStream
{
public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, BYTE, STRING, WSTRING };

    char m_Identifier[8]; //!< 識別子です。
    int m_Type; //!< 値の型です。
    int m_Count; //!< 値の個数です。
    int m_Column; //!< アスキー出力時の列数です。
    int m_Size; //!< データサイズ（バイト数）です。文字列の場合は終端文字を含んだバイト数です。
    int m_Padding[2]; //!< 実データの先頭アドレスを 32 バイトアライメントにするためのパディングです。
    int m_Data[1]; //!< 実データです。実際は可変長です。

public:
    //! @brief バイナリーデータ列へのポインタを取得します（static 関数）。
    //!
    //! @param[in] binTop 中間ファイルのバイナリー部分の先頭アドレスです。
    //! @param[in] streamIdx バイナリーデータ列のインデックスです。
    //!
    //! @return バイナリーデータ列へのポインタを返します。
    //!         インデックスが不正な場合は nullptr を返します。
    //!
    static const RBinDataStream* Get(const void* binTop, const int streamIdx);
};

//=============================================================================
// @brief ユーザーデータのクラスです。
//=============================================================================
class RUserData
{
public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, STRING, WSTRING, STREAM };

    std::string m_Name; //!< 名前です。
    Type m_Type; //!< 値の型です。
    int m_NumberValueCount; //!< 浮動小数点数値または整数値の値の数です。
    std::string m_NumberValuesText; //!< 浮動小数点数値または整数値の値の配列の中間ファイル上の文字列です。
    RStringArray m_StringValues; //!< 文字列値（Shift-JIS）の配列です。
    RUCharArray m_StreamValues; //!< ストリーム型値の配列です。

    RStringArray m_BinaryStringValues; //!< bntx ファイルに格納する文字列値（UTF-8、LF）の配列です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] userDataElem <user_data> 要素です。
    //! @param[in] streamsElem <stream_array> 要素です（バイナリー形式なら nullptr を指定します）。
    //! @param[in] binTop 中間ファイルのバイナリー部分の先頭アドレスです（アスキー形式なら nullptr を指定します）。
    //!
    RUserData(
        const RXMLElement* userDataElem,
        const RXMLElement* streamsElem,
        const void* binTop
    );

    //! @brief コンストラクタです。
    //!
    //! @param[in] name 名前です。
    //! @param[in] type 値の型です。
    //!
    RUserData(const std::string& name, const Type type)
    : m_Name(name),
      m_Type(type),
      m_NumberValueCount(0)
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount <user_data> 要素のインデントに必要なタブの数です。
    //! @param[in] index <user_data_array> 内でのインデックスです。
    //!
    void Out(
        std::ostream& os,
        RDataStreamArray& dataStreams,
        const int tabCount,
        const int index
    ) const;

    //! @brief 文字列型なら true を返します。
    const bool IsString() const
    {
        return (m_Type == STRING || m_Type == WSTRING);
    }
};

//! @brief ユーザーデータ配列の定義です。
typedef std::vector<RUserData> RUserDataArray;

//-----------------------------------------------------------------------------
//! @brief 配列形式の要素を出力します。テンプレート関数です。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount 要素のインデントに必要なタブの数です。
//! @param[in] array 配列です。
//! @param[in] name 配列の要素名です。
//-----------------------------------------------------------------------------
template <typename T>
void ROutArrayElement(
    std::ostream& os,
    const int tabCount,
    const std::vector<T>& array,
    const char* name
)
{
    const int valCount = static_cast<int>(array.size());
    if (valCount > 0)
    {
        const int& tc = tabCount;
        os << RTab(tc) << "<" << name << " length=\"" << valCount << "\">" << R_ENDL;
        for (int ival = 0; ival < valCount; ++ival)
        {
            array[ival].Out(os, tc + 1, ival);
        }
        os << RTab(tc) << "</" << name << ">" << R_ENDL;
    }
}

//=============================================================================
//! @brief ファイルバッファのクラスです。
//=============================================================================
class RFileBuf
{
protected:
    std::string m_FilePath; //!< ファイルのパスです。
    uint8_t* m_FileBuf; //!< ファイルバッファです。
    size_t m_FileSize; //!< ファイルサイズです。

public:
    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    explicit RFileBuf(const std::string& path)
    : m_FileBuf(nullptr)
    {
        Read(path);
    }

    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    explicit RFileBuf(const std::wstring& path)
    : m_FileBuf(nullptr)
    {
        Read(RGetShiftJisFromUnicode(path));
    }

    //! デストラクタです。
    virtual ~RFileBuf()
    {
        //std::cerr << "~RFileBuf(): " << m_FilePath << std::endl;
        RFreeAndClearArray(m_FileBuf);
    }

    //! メモリーを解放します。
    void Clear()
    {
        RFreeAndClearArray(m_FileBuf);
        m_FilePath.clear();
        m_FileSize = 0;
    }

    //! @biref ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    //! @return リード成功なら true を返します。
    //!
    bool Read(const std::string& path);

    //! リード成功なら true を返します。
    NN_IMPLICIT operator bool() const
    {
        return (m_FileBuf != nullptr);
    }

    //! ファイルのパスを返します。
    std::string GetPath() const
    {
        return m_FilePath;
    };

    //! ファイルバッファを返します。
    const uint8_t* GetBuf() const
    {
        return m_FileBuf;
    };

    //! ファイルバッファを返します（非 const 版）。
    uint8_t* GetBuf()
    {
        return m_FileBuf;
    };

    //! ファイルサイズを返します。
    size_t GetSize() const
    {
        return m_FileSize;
    };

    //! 文字コードが UTF-8 なら true を返します。
    bool IsUtf8() const
    {
        return (m_FileSize >= 3 &&
            m_FileBuf[0] == 0xef &&
            m_FileBuf[1] == 0xbb &&
            m_FileBuf[2] == 0xbf);
    }

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    const uint8_t& operator[](size_t i) const
    {
        return m_FileBuf[i];
    }

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です（非 const 版）。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    uint8_t& operator[](size_t i)
    {
        return m_FileBuf[i];
    }

private:
    RFileBuf(const RFileBuf& other);
    RFileBuf& operator=(const RFileBuf& other);
};

//=============================================================================
//! @brief 切り抜き矩形クラスです。
//=============================================================================
class RCropRect
{
public:
    int m_X; //!< 矩形の左上の X 座標です。
    int m_Y; //!< 矩形の左上の Y 座標です。
    int m_W; //!< 矩形の幅です。
    int m_H; //!< 矩形の高さです。

public:
    //! コンストラクタです。
    RCropRect(
        int x = 0,
        int y = 0,
        int w = 0,
        int h = 0
    )
    : m_X(x),
      m_Y(y),
      m_W(w),
      m_H(h)
    {
    }

    //! デストラクタです。
    virtual ~RCropRect()
    {
    }

    //! 初期化します。
    void Init()
    {
        *this = RCropRect();
    }

    //! 切り抜きが有効なら true を返します。
    bool IsEnabled() const
    {
        return (m_W != 0 && m_H != 0);
    }

    //! @brief 切り抜き矩形が画像全体なら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 切り抜き矩形が画像全体なら true を返します。
    //!
    bool IsEntireImage(const int imageW, const int imageH) const
    {
        return (m_X == 0      &&
                m_Y == 0      &&
                m_W == imageW &&
                m_H == imageH);
    }

    //! @brief 切り抜き矩形が画像の範囲内なら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 切り抜き矩形が画像の範囲内なら true を返します。
    //!
    bool IsWithinImage(const int imageW, const int imageH) const
    {
        const int ex = m_X + m_W;
        const int ey = m_Y + m_H;
        return (0 <= m_X && m_X <  imageW &&
                0 <= m_Y && m_Y <  imageH &&
                1 <= ex  && ex  <= imageW &&
                1 <= ey  && ey  <= imageH);
    }

    //! @brief 切り抜き矩形が画像の範囲内に収まるように調整します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 調整したなら true、調整不要なら false を返します。
    //!
    bool AdjustToImage(const int imageW, const int imageH)
    {
        const int srcX = m_X;
        const int srcY = m_Y;
        const int srcW = m_W;
        const int srcH = m_H;
        int ex = m_X + m_W;
        int ey = m_Y + m_H;
        m_X = RMin(RMax(m_X, 0), imageW - 1);
        m_Y = RMin(RMax(m_Y, 0), imageH - 1);
        ex  = RMin(RMax(ex , 1), imageW    );
        ey  = RMin(RMax(ey , 1), imageH    );
        m_W = (ex > m_X) ? ex - m_X : 1;
        m_H = (ey > m_Y) ? ey - m_Y : 1;
        return (m_X != srcX || m_Y != srcY || m_W != srcW || m_H != srcH);
    }
};

//=============================================================================
//! @brief テクスチャーレイアウトの構造体です。
//=============================================================================
struct FtxTextureLayout
{
    uint32_t value[2]; //!< 値です（内容は対象プラットフォーム依存します）。
};

//=============================================================================
// テクスチャー関連の関数です。
//=============================================================================

//! @brief ftx ファイルのファイルパスなら true を返します。
//!
//! @param[in] filePath ファイルパスです。
//!
//! @return ftx ファイルのファイルパスなら true を返します。
//!
bool IsFtxFilePath(const std::string& filePath);

//! @brief バイナリー形式の ftx ファイルのファイルパスなら true を返します。
//!
//! @param[in] filePath ファイルパスです。
//!
//! @return バイナリー形式の ftx ファイルのファイルパスなら true を返します。
//!
bool IsFtxBinaryFilePath(const std::string& filePath);

//! @brief テクスチャーの次元が配列なら true を返します。
//!
//! @param[in] dimension テクスチャーの次元です。
//!
//! @return テクスチャーの次元が配列なら true を返します。
//!
inline bool RIsArrayDimension(const FtxDimension dimension)
{
    return (
        dimension == FtxDimension_1dArray            ||
        dimension == FtxDimension_2dArray            ||
        dimension == FtxDimension_2dMultisampleArray ||
        dimension == FtxDimension_CubeMapArray);
}

//! @brief テクスチャーの次元がキューブマップまたはキューブマップ配列なら true を返します。
//!
//! @param[in] dimension テクスチャーの次元です。
//!
//! @return テクスチャーの次元がキューブマップまたはキューブマップ配列なら true を返します。
//!
inline bool RIsCubMapDimension(const FtxDimension dimension)
{
    return (
        dimension == FtxDimension_CubeMap      ||
        dimension == FtxDimension_CubeMapArray);
}

//! @brief 中間ファイルの文字列からテクスチャーの次元を取得します。
//!
//! @param[in] str テクスチャーの次元を表す文字列です。
//!
//! @return テクスチャーの次元を返します。
//!
FtxDimension RGetTextureDimensionFromString(const std::string& str);

//! @brief テクスチャーの次元から中間ファイルの文字列を取得します。
//!
//! @param[in] dimension テクスチャーの次元です。
//!
//! @return テクスチャーの次元を表す文字列を返します。
//!
std::string RGetTextureDimensionString(const FtxDimension dimension);

//! @brief 中間ファイルの文字列からテクスチャーフォーマットを取得します。
//!
//! @param[in] str テクスチャーフォーマットを表す文字列です。
//!
//! @return テクスチャーフォーマットを返します。
//!
FtxFormat RGetTextureFormatFromString(const std::string& str);

//! @brief テクスチャーフォーマットから中間ファイルの文字列を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットを表す文字列を返します。
//!
std::string RGetTextureFormatString(const FtxFormat format);

//! @brief テクスチャーフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//!
bool RIsBc123TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが BC6/BC7 フォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが BC6/BC7 フォーマットなら true を返します。
//!
bool RIsBc67TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが BC フォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが BC フォーマットなら true を返します。
//!
bool RIsBcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の unorm / snorm なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが 1 成分あたり 16bit の unorm / snorm なら true を返します。
//!
bool RIsUnormSnorm16TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の uint / sint なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが 1 成分あたり 16bit の uint / sint なら true を返します。
//!
bool RIsUintSint16TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが 1 成分あたり 32bit の uint / sint なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが 1 成分あたり 32bit の uint / sint なら true を返します。
//!
bool RIsUintSint32TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが浮動小数点数型なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが浮動小数点数型なら true を返します。
//!
bool RIsFloatTextureFormat(const FtxFormat format);

//! @brief 実数型として扱うテクスチャーフォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 実数型として扱うテクスチャーフォーマットなら true を返します。
//!
bool RIsRealNumberTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが整数型なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが整数型なら true を返します。
//!
bool RIsIntTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが符号ありなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが符号ありなら true を返します。
//!
bool RIsSignedTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが sRGB フェッチなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが sRGB フェッチなら true を返します。
//!
bool RIsSrgbFetchTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ETC1 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ETC1 なら true を返します。
//!
bool RIsEtc1TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ETC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ETC2 なら true を返します。
//!
bool RIsEtc2TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ETC1 または ETC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ETC1 または ETC2 なら true を返します。
//!
bool RIsEtcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが EAC なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが EAC なら true を返します。
//!
bool RIsEacTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが PVRTC1 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが PVRTC1 なら true を返します。
//!
bool RIsPvrtc1TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが PVRTC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが PVRTC2 なら true を返します。
//!
bool RIsPvrtc2TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが PVRTC1 または PVRTC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが PVRTC1 または PVRTC2 なら true を返します。
//!
bool RIsPvrtcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ASTC なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ASTC なら true を返します。
//!
bool RIsAstcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが圧縮形式なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが圧縮形式なら true を返します。
//!
bool RIsCompressedTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットから成分数を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 成分数を返します。
//!
int RGetComponentCount(const FtxFormat format);

//! @brief テクスチャーフォーマットの 1 ピクセルのビット数を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 1 ピクセルのビット数を返します。
//!         ASTC フォーマットの場合は 0 を返します。
//!
int RGetBitsPerPixel(const FtxFormat format);

//! @brief テクスチャーの圧縮ブロックの幅を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 圧縮ブロックの幅を返します。
//!
int RGetTextureBlockWidth(const FtxFormat format);

//! @brief テクスチャーの圧縮ブロックの高さを取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 圧縮ブロックの高さを返します。
//!
int RGetTextureBlockHeight(const FtxFormat format);

//! @brief テクスチャーのデータサイズ（バイト数）を取得します。
//!
//! @param[out] pLevelOffsets 各ミップマップデータのイメージデータ先頭からのオフセット配列を格納します。
//!                           nullptr なら格納しません。
//! @param[in] format テクスチャーフォーマットです。
//! @param[in] dimension テクスチャーの次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//!
//! @return テクスチャーのデータサイズを返します。
//!
size_t RGetTextureDataSize(
    std::vector<size_t>* pLevelOffsets,
    const FtxFormat format,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
);

//! @brief 中間ファイルの文字列から成分選択を取得します。
//!
//! @param[in] str 成分選択を表す文字列です。
//!
//! @return 成分選択を返します。
//!
FtxCompSel RGetCompSelFromString(const std::string& str);

//! @brief 成分選択から中間ファイルの文字列を取得します。
//!
//! @param[in] compSel 成分選択です。
//!
//! @return 成分選択を表す文字列を返します。
//!
std::string RGetCompSelString(const FtxCompSel compSel);

//! @brief テクスチャーフォーマットに対応したデフォルトの成分選択を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 成分選択を返します。
//!
FtxCompSel RGetDefaultCompSel(const FtxFormat format);

//! @brief 法線用のヒント情報を取得します。
//!
//! @return 法線用のヒント情報を返します。
//!
std::string RGetHintForNormal();

//! @brief テクスチャーフォーマットに対応したデフォルトのヒント情報を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return デフォルトのヒント情報を返します。
//!
std::string RGetDefaultHint(const FtxFormat format);

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

