﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ConvertByPvrTexLib AdjustTextureDataForEncode

//=============================================================================
// include
//=============================================================================
#include "PvrFormat.h"

#pragma warning(push)
    #pragma warning(disable: 4819) // ファイルは、現在のコード ページ (932) で表示できない文字を含んでいます。
    #include <PVRTextureUtilities.h>
#pragma warning(pop)

using namespace std;
using namespace nn::gfx::tool::texenc;
using namespace pvrtexture;

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//-----------------------------------------------------------------------------
//! @brief PVRTexLib に渡すフォーマットを取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return フォーマットを返します。
//-----------------------------------------------------------------------------
PixelType GetPvrFormat(const std::string& formatStr)
{
    return
        (formatStr == "unorm_bc1"                  ) ? ePVRTPF_BC1              :
        (formatStr == "srgb_bc1"                   ) ? ePVRTPF_BC1              :
        (formatStr == "unorm_bc2"                  ) ? ePVRTPF_BC2              :
        (formatStr == "srgb_bc2"                   ) ? ePVRTPF_BC2              :
        (formatStr == "unorm_bc3"                  ) ? ePVRTPF_BC3              :
        (formatStr == "srgb_bc3"                   ) ? ePVRTPF_BC3              :
        (formatStr == "unorm_etc1"                 ) ? ePVRTPF_ETC1             :
        (formatStr == "unorm_etc2"                 ) ? ePVRTPF_ETC2_RGB         :
        (formatStr == "srgb_etc2"                  ) ? ePVRTPF_ETC2_RGB         :
        (formatStr == "unorm_etc2_mask"            ) ? ePVRTPF_ETC2_RGB_A1      :
        (formatStr == "srgb_etc2_mask"             ) ? ePVRTPF_ETC2_RGB_A1      :
        (formatStr == "unorm_etc2_alpha"           ) ? ePVRTPF_ETC2_RGBA        :
        (formatStr == "srgb_etc2_alpha"            ) ? ePVRTPF_ETC2_RGBA        :
        (formatStr == "unorm_eac_11"               ) ? ePVRTPF_EAC_R11          :
        (formatStr == "snorm_eac_11"               ) ? ePVRTPF_EAC_R11          :
        (formatStr == "unorm_eac_11_11"            ) ? ePVRTPF_EAC_RG11         :
        (formatStr == "snorm_eac_11_11"            ) ? ePVRTPF_EAC_RG11         :
        (formatStr == "unorm_pvrtc1_2bpp"          ) ? ePVRTPF_PVRTCI_2bpp_RGB  :
        (formatStr == "srgb_pvrtc1_2bpp"           ) ? ePVRTPF_PVRTCI_2bpp_RGB  :
        (formatStr == "unorm_pvrtc1_4bpp"          ) ? ePVRTPF_PVRTCI_4bpp_RGB  :
        (formatStr == "srgb_pvrtc1_4bpp"           ) ? ePVRTPF_PVRTCI_4bpp_RGB  :
        (formatStr == "unorm_pvrtc1_alpha_2bpp"    ) ? ePVRTPF_PVRTCI_2bpp_RGBA :
        (formatStr == "srgb_pvrtc1_alpha_2bpp"     ) ? ePVRTPF_PVRTCI_2bpp_RGBA :
        (formatStr == "unorm_pvrtc1_alpha_4bpp"    ) ? ePVRTPF_PVRTCI_4bpp_RGBA :
        (formatStr == "srgb_pvrtc1_alpha_4bpp"     ) ? ePVRTPF_PVRTCI_4bpp_RGBA :
        (formatStr == "unorm_pvrtc2_alpha_2bpp"    ) ? ePVRTPF_PVRTCII_2bpp     :
        (formatStr == "srgb_pvrtc2_alpha_2bpp"     ) ? ePVRTPF_PVRTCII_2bpp     :
        (formatStr == "unorm_pvrtc2_alpha_4bpp"    ) ? ePVRTPF_PVRTCII_4bpp     :
        (formatStr == "srgb_pvrtc2_alpha_4bpp"     ) ? ePVRTPF_PVRTCII_4bpp     :
        (formatStr.find("_8" ) != std::string::npos) ? PVRStandard8PixelType    :
        (formatStr.find("_16") != std::string::npos) ? PVRStandard16PixelType   :
        (formatStr.find("_32") != std::string::npos) ? PVRStandard32PixelType   :
        PVRStandard8PixelType;
}

//-----------------------------------------------------------------------------
//! @brief フォーマットから PVRTexLib に渡すチャンネルタイプを取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return チャンネルタイプを返します。
//-----------------------------------------------------------------------------
EPVRTVariableType GetPvrChanTypeFromFormat(const std::string& formatStr)
{
    return
        (formatStr == "snorm_eac_11"     ) ? ePVRTVarTypeSignedByteNorm      :
        (formatStr == "snorm_eac_11_11"  ) ? ePVRTVarTypeSignedByteNorm      :
        (formatStr.find("unorm_8"  ) == 0) ? ePVRTVarTypeUnsignedByteNorm    :
        (formatStr.find("snorm_8"  ) == 0) ? ePVRTVarTypeSignedByteNorm      :
        (formatStr.find("srgb_8"   ) == 0) ? ePVRTVarTypeUnsignedByteNorm    :
        (formatStr.find("uint_8"   ) == 0) ? ePVRTVarTypeUnsignedByte        :
        (formatStr.find("sint_8"   ) == 0) ? ePVRTVarTypeSignedByte          :
        (formatStr.find("unorm_16" ) == 0) ? ePVRTVarTypeUnsignedShortNorm   :
        (formatStr.find("snorm_16" ) == 0) ? ePVRTVarTypeSignedShortNorm     :
        (formatStr.find("uint_16"  ) == 0) ? ePVRTVarTypeUnsignedShort       :
        (formatStr.find("sint_16"  ) == 0) ? ePVRTVarTypeSignedShort         :
        (formatStr.find("unorm_32" ) == 0) ? ePVRTVarTypeUnsignedIntegerNorm :
        (formatStr.find("snorm_32" ) == 0) ? ePVRTVarTypeSignedIntegerNorm   :
        (formatStr.find("uint_32"  ) == 0) ? ePVRTVarTypeUnsignedInteger     :
        (formatStr.find("sint_32"  ) == 0) ? ePVRTVarTypeSignedInteger       :
        (formatStr.find("float_32" ) == 0) ? ePVRTVarTypeSignedFloat         :
        (formatStr.find("ufloat_32") == 0) ? ePVRTVarTypeUnsignedFloat       :
        ePVRTVarTypeUnsignedByteNorm;
}

//-----------------------------------------------------------------------------
//! @brief 圧縮フォーマットなら true を返します。
//!
//! @param[in] format フォーマットです。
//!
//! @return 圧縮フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsCompressed(const PixelType& format)
{
    return (
        format.PixelTypeID != PVRStandard8PixelType.PixelTypeID  &&
        format.PixelTypeID != PVRStandard16PixelType.PixelTypeID &&
        format.PixelTypeID != PVRStandard32PixelType.PixelTypeID);
}

//-----------------------------------------------------------------------------
//! @brief PVRTexLib に渡す品質を取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//! @param[in] qualityStr エンコード品質です。
//!
//! @return 品質を返します。
//-----------------------------------------------------------------------------
ECompressorQuality GetPvrQuality(
    const std::string& formatStr,
    const std::string& qualityStr
)
{
    const int qualityLevel = GetQualityLevel(qualityStr);
    if (formatStr.find("pvrtc") != std::string::npos)
    {
        return
            (qualityStr == "fastest") ? ePVRTCFastest :
            (qualityStr == "fast"   ) ? ePVRTCFast    :
            (qualityStr == "normal" ) ? ePVRTCNormal  :
            (qualityStr == "high"   ) ? ePVRTCHigh    :
            (qualityStr == "best"   ) ? ePVRTCBest    :
            (qualityLevel == 0      ) ? ePVRTCFast    :
            (qualityLevel == 1      ) ? ePVRTCNormal  :
            (qualityLevel == 2      ) ? ePVRTCHigh    :
            (qualityLevel >= 3      ) ? ePVRTCBest    :
            ePVRTCNormal;
    }
    else // ETC / EAC
    {
        const bool isPerceptual = (qualityStr.find("perceptual") != std::string::npos);
        return
            (qualityStr == "fast"           ) ? eETCFast           :
            (qualityStr == "fast_perceptual") ? eETCFastPerceptual :
            (qualityStr == "slow"           ) ? eETCSlow           :
            (qualityStr == "slow_perceptual") ? eETCSlowPerceptual :
            (qualityLevel <= 1) ? ((isPerceptual) ? eETCFastPerceptual : eETCFast) :
            (qualityLevel >= 2) ? ((isPerceptual) ? eETCSlowPerceptual : eETCSlow) :
            eETCFast;
    }
}

//-----------------------------------------------------------------------------
//! @brief PVR テクスチャのデータをエンコードのために調整します。
//!
//! @param[in,out] tex エンコード前の PVR テクスチャです。
//! @param[in] dstFormat エンコード後のフォーマットです。
//-----------------------------------------------------------------------------
void AdjustTextureDataForEncode(CPVRTexture& tex, const PixelType& dstFormat)
{
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(tex.getDataPtr());
    const size_t dataSize = tex.getDataSize();

    if (tex.getChannelType() == ePVRTVarTypeUnsignedByteNorm &&
        dstFormat.PixelTypeID == ePVRTPF_ETC2_RGB_A1)
    {
        //-----------------------------------------------------------------------------
        // 1bit アルファ ETC2 フォーマットにエンコードする際、
        // 半透明アルファが存在すると INVALID ALPHA DATA!! と表示されるので
        // 2 値化（0x00 or 0xff）しておきます。
        const size_t pixelSize = dataSize / EncRgbaBytes;
        for (size_t iPix = 0; iPix < pixelSize; ++iPix)
        {
            pDstU8[3] = (pDstU8[3] >= EncOpaAlphaBorder) ? 0xff : 0x00;
            pDstU8 += EncRgbaCount;
        }
    }
    else if (tex.getChannelType() == ePVRTVarTypeSignedByteNorm &&
        (dstFormat.PixelTypeID == ePVRTPF_EAC_R11 ||
         dstFormat.PixelTypeID == ePVRTPF_EAC_RG11))
    {
        //-----------------------------------------------------------------------------
        // 符号付き 8 ビットの値が 0x80、0xfe、0xff だと
        // EAC フォーマットに正しくエンコードされないので、
        // それぞれ 0x81、0xfd、0x00 に変更します。
        for (size_t iData = 0; iData < dataSize; ++iData)
        {
            if (*pDstU8 == 0x80)
            {
                *pDstU8 = 0x81;
            }
            else if (*pDstU8 == 0xfe)
            {
                *pDstU8 = 0xfd;
            }
            else if (*pDstU8 == 0xff)
            {
                *pDstU8 = 0x00;
            }
            ++pDstU8;
        }
    }
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief PVRTexLib でフォーマットを変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] qualityStr エンコード品質です。
//! @param[in] encodeFlag エンコードフラグです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertByPvrTexLib(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const std::string& qualityStr,
    const int encodeFlag,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    //-----------------------------------------------------------------------------
    // PVRTexLib に渡すフォーマットを取得します。
    const PixelType srcFormat = GetPvrFormat(srcFormatStr);
    const PixelType dstFormat = GetPvrFormat(dstFormatStr);
    const EPVRTVariableType srcChanType = GetPvrChanTypeFromFormat(srcFormatStr);
    const EPVRTVariableType dstChanType = GetPvrChanTypeFromFormat(dstFormatStr);
    const bool encodes = IsCompressed(dstFormat);
    //cerr << "src: " << srcFormat.PixelTypeID << ", " << srcChanType << endl;
    //cerr << "dst: " << dstFormat.PixelTypeID << ", " << dstChanType << ", " << encodes << endl;

    //-----------------------------------------------------------------------------
    // PVR テクスチャを作成します。
    const int numMipMaps = 1;
    const int numFaces = 1;
    const CPVRTextureHeader header(srcFormat.PixelTypeID,
        imageH, imageW, 1, numMipMaps, imageD, numFaces,
        ePVRTCSpacelRGB, srcChanType);
        // ヘッダは高さ、幅、奥行き、ミップマップのレベル数、配列の要素数、フェース数の順に指定します。
        // 現在奥行きを 1 以外にすると Transcode でエラーになるので、
        // imageD を配列の要素数として指定します。
    CPVRTexture tex(header, pSrc);

    //-----------------------------------------------------------------------------
    // PVR テクスチャのデータをエンコードのために調整します。
    if (encodes)
    {
        AdjustTextureDataForEncode(tex, dstFormat);
    }

    //-----------------------------------------------------------------------------
    // 変換します。
    const EPVRTColourSpace colorSpace = ePVRTCSpacelRGB; // リニア変換なし
    const ECompressorQuality quality = GetPvrQuality(dstFormatStr, qualityStr);
    const bool doDither = ((encodeFlag & EncodeFlag_Dither) != 0);
    if (!Transcode(tex, dstFormat, dstChanType, colorSpace, quality, doDither))
    {
        return false;
    }

    memcpy(pDst, tex.getDataPtr(), tex.getDataSize());

    return true;
}

