﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_VariationValueArg.h>
#include <gfxTool_VariationXml.h>

namespace nn {
namespace gfxTool {

void VariationConstantValue::Initialize( nngfxToolShaderCompilerVariationConstantValue* pArg,
        const VariationConstantValueNode* pVariationConstantValueNode,
        const nngfxToolShaderCompilerVariationConstantDefinition* pVariationConstantDefinition )
{
    m_pArg = pArg;
    auto valueText = pVariationConstantValueNode->GetText();
    if( valueText.empty() )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlTextNotFound, "%s has no text.",
            pVariationConstantValueNode->GetId() );
    }
    switch( pVariationConstantDefinition->type )
    {
    case nngfxToolShaderCompilerVariableType_Bool32: NN_FALL_THROUGH;
    case nngfxToolShaderCompilerVariableType_Int32: NN_FALL_THROUGH;
    case nngfxToolShaderCompilerVariableType_Uint32:
        {
            m_ValueUint32 = TextToArray< Custom< std::vector< uint32_t > >::Type >( valueText.data() );
            m_pArg->valueSizeIn32Bit = NumericCastAuto( m_ValueUint32.size() );
            m_pArg->pValue = &m_ValueUint32[ 0 ];
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float32:
        {
            m_ValueFloat32 = TextToArray< Custom< std::vector< float > >::Type >( valueText.data() );
            m_pArg->valueSizeIn32Bit = NumericCastAuto( m_ValueFloat32.size() );
            m_pArg->pValue = &m_ValueFloat32[ 0 ];
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float64:
        {
            m_ValueFloat64 = TextToArray< Custom< std::vector< double > >::Type >( valueText.data() );
            m_pArg->valueSizeIn32Bit = NumericCastAuto( m_ValueFloat64.size() * 2 );
            m_pArg->pValue = &m_ValueFloat64[ 0 ];
        }
        break;
    default:
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
            "Invalid variable type (%d).", NumericCast< int >( pVariationConstantDefinition->type  ) );
    }
}

void VariationValue::Initialize( const StageShaderVariationValueNode* pStageShaderVariationValueNode,
        const nngfxToolShaderCompilerVariationDefinition* pVariationDefinition )
{
    if( auto pVariationConstantValueArrayNode =
        pStageShaderVariationValueNode->GetVariationConstantValueArrayNode() )
    {
        auto length = pVariationConstantValueArrayNode->GetLengthAttribute();
        if( length.empty() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound, "%s::length is not found.",
                pVariationConstantValueArrayNode->GetId() );
        }
        this->variationConstantValueCount = NumericCastAuto( LexicalCast< int >( length.data() ) );
        m_VariationConstantValueArrayRaw.resize( this->variationConstantValueCount );
        m_VariationConstantValueArray.resize( this->variationConstantValueCount );
        decltype( pVariationConstantValueArrayNode->GetNextChild( nullptr ) )
            pVariationConstantValueNode = nullptr;
        for( int idxVariationConstant = 0, variationConstantCount = NumericCastAuto(
            this->variationConstantValueCount ); idxVariationConstant <
            variationConstantCount; ++idxVariationConstant )
        {
            pVariationConstantValueNode =
                pVariationConstantValueArrayNode->GetNextChild( pVariationConstantValueNode );
            if( pVariationConstantValueNode == nullptr )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s is not found",
                    pVariationConstantValueNode->GetId() );
            }
            m_VariationConstantValueArray[ idxVariationConstant ].Initialize(
                &m_VariationConstantValueArrayRaw[ idxVariationConstant ], pVariationConstantValueNode,
                pVariationDefinition->pVariationConstantDefinitionArray + idxVariationConstant );
        }
        this->pVariationConstantValueArray = &m_VariationConstantValueArrayRaw[ 0 ];
    }

    if( auto pPreprocessorDefinitionValueArrayNode =
        pStageShaderVariationValueNode->GetPreprocessorDefinitionValueArrayNode() )
    {
        auto length = pPreprocessorDefinitionValueArrayNode->GetLengthAttribute();
        if( length.empty() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound, "%s::length is not found.",
                pPreprocessorDefinitionValueArrayNode->GetId() );
        }
        this->preprocessorDefinitionValueCount = NumericCastAuto( LexicalCast< int >( length.data() ) );
        m_PreprocessorDefinitionValueArrayRaw.resize( this->preprocessorDefinitionValueCount );
        m_PreprocessorDefinitionValueArray.resize( this->preprocessorDefinitionValueCount );
        decltype( pPreprocessorDefinitionValueArrayNode->GetNextChild( nullptr ) )
            pPreprocessorDefinitionValueNode = nullptr;
        for( int idxPreprocessorDefinition = 0, preprocessorDefinitionCount = NumericCastAuto(
            this->preprocessorDefinitionValueCount ); idxPreprocessorDefinition <
            preprocessorDefinitionCount; ++idxPreprocessorDefinition )
        {
            pPreprocessorDefinitionValueNode =
                pPreprocessorDefinitionValueArrayNode->GetNextChild( pPreprocessorDefinitionValueNode );
            if( pPreprocessorDefinitionValueNode == nullptr )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s is not found.",
                    pPreprocessorDefinitionValueNode->GetId() );
            }

            auto text = pPreprocessorDefinitionValueNode->GetText();
            m_PreprocessorDefinitionValueArray[ idxPreprocessorDefinition ].assign( text.data(), text.length() );
            m_PreprocessorDefinitionValueArrayRaw[ idxPreprocessorDefinition ].pValue =
                m_PreprocessorDefinitionValueArray[ idxPreprocessorDefinition ].data();
            m_PreprocessorDefinitionValueArrayRaw[ idxPreprocessorDefinition ].length =
                NumericCastAuto( m_PreprocessorDefinitionValueArray[ idxPreprocessorDefinition ].length() );
        }
        this->pPreprocessorDefinitionValueArray = &m_PreprocessorDefinitionValueArrayRaw[ 0 ];
    }
}

void VariationValueArg::Initialize( nngfxToolShaderCompilerVariationValueArg* pArg,
        const ShaderVariationValueNode* pShaderVariationValueNode,
        const nngfxToolShaderCompilerVariationDefinitionArg* pVariationDefinitionArg )
{
    m_pArg = pArg;

    static const nngfxToolShaderCompilerVariationValue*
        nngfxToolShaderCompilerVariationValueArg::* pVariationValues[] =
    {
        &nngfxToolShaderCompilerVariationValueArg::pVertexShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pHullShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pDomainShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pGeometryShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pPixelShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pComputeShaderVariationValue
    };
    static const nngfxToolShaderCompilerVariationDefinition*
        nngfxToolShaderCompilerVariationDefinitionArg::* pVariationDefinitions[] =
    {
        &nngfxToolShaderCompilerVariationDefinitionArg::pVertexShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pHullShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pDomainShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pGeometryShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pPixelShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pComputeShaderVariationDefinition
    };
    for( int stage = 0; stage < static_cast< int >( ShaderStage::End ); ++stage )
    {
        auto pStageNode = pShaderVariationValueNode->GetStageShaderVariationValueNode(
            StaticCastAuto( stage ) );
        if( pStageNode )
        {
            m_pVariationValues[ stage ].reset( new VariationValue() );
            m_pVariationValues[ stage ]->Initialize( pStageNode,
                pVariationDefinitionArg->*pVariationDefinitions[ stage ] );
            m_pArg->*pVariationValues[ stage ] = m_pVariationValues[ stage ].get();
        }
    }
}

}
}
