﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/utilTool/utilTool_BinarizerUtil.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_ShaderReflectionBinarizer.h>
#include <gfxTool_ShaderBinarizerContext.h>

namespace nn {
namespace gfxTool {

bool ShaderReflectionStageBinarizer::IsData2Needed( const BinarizationTarget* pTarget ) const
{
    return pTarget->separateTextureCount > 0 || pTarget->separateSamplerCount > 0;
}

void ShaderReflectionStageBinarizer::Initialize( const BinarizationTarget* pTarget, ShaderStage stage )
{
    auto Collect = [ & ]( auto* pDst, const auto* pReflectionArray, int reflectionCount, ShaderStage stage )
    {
        const nngfxToolShaderCompilerReflectionStageReference s_StageBitTable[] =
        {
            nngfxToolShaderCompilerReflectionStageReference_Vertex,
            nngfxToolShaderCompilerReflectionStageReference_Hull,
            nngfxToolShaderCompilerReflectionStageReference_Domain,
            nngfxToolShaderCompilerReflectionStageReference_Geometry,
            nngfxToolShaderCompilerReflectionStageReference_Pixel,
            nngfxToolShaderCompilerReflectionStageReference_Compute
        };

        pDst->reserve( reflectionCount );
        std::for_each( pReflectionArray, pReflectionArray + reflectionCount,
            [ & ]( const decltype( *pReflectionArray )& reflection )
        {
            if( reflection.stages & s_StageBitTable[ static_cast< int >( stage ) ] )
            {
                pDst->push_back( &reflection );
            }
        } );
        m_SlotCount += NumericCast< decltype( m_SlotCount ) >( pDst->size() );
    };

    if( pTarget == nullptr )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }
    m_pTarget = pTarget;
    m_Stage = stage;

    m_SlotCount = 0;

    Collect( &m_ShaderInputs, pTarget->pShaderInputArray, pTarget->shaderInputCount, stage );
    Collect( &m_ShaderOutputs, pTarget->pShaderOutputArray, pTarget->shaderOutputCount, stage );
    Collect( &m_Samplers, pTarget->pSamplerArray, pTarget->samplerCount, stage );
    Collect( &m_ConstantBuffers, pTarget->pConstantBufferArray, pTarget->constantBufferCount, stage );
    Collect( &m_UnorderedAccessBuffers, pTarget->pUnorderedAccessBufferArray,
        pTarget->unorderedAccessBufferCount, stage );
    Collect( &m_Images, pTarget->pImageArray, pTarget->imageCount, stage );
    Collect( &m_SeparateTextures, pTarget->pSeparateTextureArray, pTarget->separateTextureCount, stage );
    Collect( &m_SeparateSamplers, pTarget->pSeparateSamplerArray, pTarget->separateSamplerCount, stage );
}

void ShaderReflectionStageBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    auto AddReflectionBlock = [ & ]( nn::util::MemorySplitter::MemoryBlock* pBlock, const auto& reflections )
    {
        Custom< std::vector< nn::util::string_view > >::Type keys;
        for( auto&& item : reflections )
        {
            keys.push_back( nn::util::string_view( item->name.pValue, item->name.length ) );
        }
        if( keys.size() > 0 )
        {
            pContext->AddResDicMemoryBlock( StaticCastAuto( Section::Common ),
                pBlock, &keys[ 0 ], NumericCastAuto( keys.size() ) );
        }
    };

    if( IsData2Needed( m_pTarget ) )
    {
        pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_Data2Block );
    }

    AddReflectionBlock( &m_ShaderInputDicBlock, m_ShaderInputs );
    AddReflectionBlock( &m_ShaderOutputDicBlock, m_ShaderOutputs );
    AddReflectionBlock( &m_SamplerDicBlock, m_Samplers );
    AddReflectionBlock( &m_ConstantBufferDicBlock, m_ConstantBuffers );
    AddReflectionBlock( &m_UnorderedAccessBufferDicBlock, m_UnorderedAccessBuffers );
    AddReflectionBlock( &m_ImageDicBlock, m_Images );
    AddReflectionBlock( &m_SeparateTextureDicBlock, m_SeparateTextures );
    AddReflectionBlock( &m_SeparateSamplerDicBlock, m_SeparateSamplers );

    pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_SlotBlock );
}

void ShaderReflectionStageBinarizer::CalculateSize()
{
    SetSizeBy< ResTarget >();

    auto SetSize = []( nn::util::MemorySplitter::MemoryBlock* pBlock, int elementCount )
    {
        pBlock->SetSize( elementCount > 0 ? nn::util::ResDic::CalculateSize( elementCount ) : 0 );
    };

    m_Data2Block.SetSizeBy< ResTarget2 >();

    SetSize( &m_ShaderInputDicBlock, NumericCastAuto( m_ShaderInputs.size() ) );
    SetSize( &m_ShaderOutputDicBlock, NumericCastAuto( m_ShaderOutputs.size() ) );
    SetSize( &m_SamplerDicBlock, NumericCastAuto( m_Samplers.size() ) );
    SetSize( &m_ConstantBufferDicBlock, NumericCastAuto( m_ConstantBuffers.size() ) );
    SetSize( &m_UnorderedAccessBufferDicBlock, NumericCastAuto( m_UnorderedAccessBuffers.size() ) );
    SetSize( &m_ImageDicBlock, NumericCastAuto( m_Images.size() ) );
    SetSize( &m_SeparateTextureDicBlock, NumericCastAuto( m_SeparateTextures.size() ) );
    SetSize( &m_SeparateSamplerDicBlock, NumericCastAuto( m_SeparateSamplers.size() ) );

    m_SlotBlock.SetSize( sizeof( int32_t ) * m_SlotCount );
}

void ShaderReflectionStageBinarizer::Link( ShaderBinarizerContext* pContext )
{
    const ResTarget* pResTarget = nullptr;

    pContext->LinkPtr( this, &pResTarget->pShaderInputDic, &m_ShaderInputDicBlock );
    pContext->LinkPtr( this, &pResTarget->pShaderOutputDic, &m_ShaderOutputDicBlock );
    pContext->LinkPtr( this, &pResTarget->pSamplerDic, &m_SamplerDicBlock );
    pContext->LinkPtr( this, &pResTarget->pConstantBufferDic, &m_ConstantBufferDicBlock );
    pContext->LinkPtr( this, &pResTarget->pUnorderedAccessBufferDic,
        &m_UnorderedAccessBufferDicBlock );
    pContext->LinkPtr( this, &pResTarget->pImageDic, &m_ImageDicBlock );
    pContext->LinkPtr( this, &pResTarget->pShaderSlotArray, &m_SlotBlock );

    if( IsData2Needed( m_pTarget ) )
    {
        const ResTarget2* pResTarget2 = nullptr;
        pContext->LinkPtr( this, &pResTarget->pReflectionStageData2, &m_Data2Block );
        pContext->LinkPtr( &m_Data2Block, &pResTarget2->pSeparateTextureDic, &m_SeparateTextureDicBlock );
        pContext->LinkPtr( &m_Data2Block, &pResTarget2->pSeparateSamplerDic, &m_SeparateSamplerDicBlock );
    }
}

void ShaderReflectionStageBinarizer::Convert( ShaderBinarizerContext* pContext )
{
    auto FillStageSlots = []( int32_t* pDst, const auto& reflections, ShaderStage stage ) -> decltype( auto )
    {
        static int32_t const nngfxToolShaderCompilerShaderSlot::* const s_pSlotStages[] =
        {
            &nngfxToolShaderCompilerShaderSlot::vertexShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::hullShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::domainShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::geometryShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::pixelShaderSlot,
            &nngfxToolShaderCompilerShaderSlot::computeShaderSlot
        };

        for( auto&& reflection : reflections )
        {
            *pDst++ = reflection->shaderSlot.*s_pSlotStages[ static_cast< int >( stage ) ];
        }

        return pDst;
    };

    auto FillSlots = [ & ]( int32_t* pDst, const auto& reflections ) -> decltype( auto )
    {
        for( auto&& reflection : reflections )
        {
            *pDst++ = reflection->shaderSlot;
        }

        return pDst;
    };

    auto pSlots = m_SlotBlock.Get< int32_t >( pContext->GetPtr() );
    pSlots = FillSlots( pSlots, m_ShaderInputs );
    pSlots = FillSlots( pSlots, m_ShaderOutputs );
    pSlots = FillStageSlots( pSlots, m_Samplers, m_Stage );
    pSlots = FillStageSlots( pSlots, m_ConstantBuffers, m_Stage );
    pSlots = FillStageSlots( pSlots, m_UnorderedAccessBuffers, m_Stage );
    pSlots = FillStageSlots( pSlots, m_Images, m_Stage );

    auto pResTarget = Get< ResTarget >( pContext->GetPtr() );
    int offset = NumericCastAuto( m_ShaderInputs.size() );

    pResTarget->offsetShaderOutput = offset;
    offset += static_cast<int>( m_ShaderOutputs.size() );
    pResTarget->offsetSampler = offset;
    offset += static_cast<int>( m_Samplers.size() );
    pResTarget->offsetConstantBuffer = offset;
    offset += static_cast<int>( m_ConstantBuffers.size() );
    pResTarget->offsetUnorderedAccessBuffer = offset;
    offset += static_cast<int>( m_UnorderedAccessBuffers.size() );
    pResTarget->offsetImage = offset;
    offset += static_cast<int>( m_Images.size() );

    pResTarget->computeWorkGroupSizeX = m_pTarget->computeWorkGroupSizeX;
    pResTarget->computeWorkGroupSizeY = m_pTarget->computeWorkGroupSizeY;
    pResTarget->computeWorkGroupSizeZ = m_pTarget->computeWorkGroupSizeZ;

    if( IsData2Needed( m_pTarget ) )
    {
        pSlots = FillStageSlots( pSlots, m_SeparateTextures, m_Stage );
        pSlots = FillStageSlots( pSlots, m_SeparateSamplers, m_Stage );

        auto pResTarget2 = m_Data2Block.Get< ResTarget2 >( pContext->GetPtr() );
        pResTarget2->offsetSeparateTexture = offset;
        offset += static_cast<int>( m_SeparateTextures.size() );
        pResTarget2->offsetSeparateSampler = offset;
        offset += static_cast<int>( m_SeparateSamplers.size() );
    }
}

void ShaderReflectionBinarizer::Initialize( const BinarizationTarget* pTarget )
{
    if( pTarget == nullptr )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }
    m_pTarget = pTarget;

    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        m_ReflectionStages[ idxStage ].Initialize( pTarget, StaticCastAuto( idxStage ) );
    }
}

void ShaderReflectionBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        if( !m_ReflectionStages[ idxStage ].IsEmpty() )
        {
            pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), m_ReflectionStages + idxStage );
        }
    }

    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        if( !m_ReflectionStages[ idxStage ].IsEmpty() )
        {
            m_ReflectionStages[ idxStage ].RegisterChild( pContext );
        }
    }
}

void ShaderReflectionBinarizer::CalculateSize()
{
    SetSizeBy< ResTarget >();
}

void ShaderReflectionBinarizer::Link( ShaderBinarizerContext* pContext )
{
    const ResTarget* pResTarget = nullptr;

    static nn::util::BinTPtr< nn::gfx::ResShaderReflectionStageData >
        nn::gfx::ResShaderReflectionData::*s_pReflectionStages[] =
    {
        &nn::gfx::ResShaderReflectionData::pVertexReflection,
        &nn::gfx::ResShaderReflectionData::pHullReflection,
        &nn::gfx::ResShaderReflectionData::pDomainReflection,
        &nn::gfx::ResShaderReflectionData::pGeometryReflection,
        &nn::gfx::ResShaderReflectionData::pPixelReflection,
        &nn::gfx::ResShaderReflectionData::pComputeReflection
    };

    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        if( !m_ReflectionStages[ idxStage].IsEmpty() )
        {
            pContext->LinkPtr( this, &( pResTarget->*s_pReflectionStages[ idxStage ] ),
                m_ReflectionStages + idxStage );
        }
    }
}

void ShaderReflectionBinarizer::Convert( ShaderBinarizerContext* )
{
}

}
}
