﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/gfx/gfx_Enum.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_ShaderCodeBinarizer.h>
#include <gfxTool_ShaderBinarizerContext.h>
#include <gfxTool_ResShaderProgramBinarizer.h>
#include <gfxTool_ResShaderVariationBinarizer.h>
#include <gfxTool_ResShaderContainerBinarizer.h>

namespace nn {
namespace gfxTool {

void ShaderCodeBinarizer::Initialize( const BinarizationTarget* pTarget,
    const ResShaderProgramBinarizer* pParent )
{
    m_pTarget = pTarget;
    m_pParent = pParent;
}

void ShaderCodeBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    static Section s_CodeTypeToSectionTable[ nn::gfx::ShaderCodeType_End ] =
    {
        Section::ShaderBinary,
        Section::ShaderIntermediateLanguage,
        Section::ShaderSource,
        Section::Common
    };

    pContext->AddMemoryBlock( StaticCastAuto( s_CodeTypeToSectionTable[
        m_pParent->GetInfoData()->codeType ] ), &m_CodeBlock );
}

void ShaderCodeBinarizer::CalculateSize()
{
    SetSizeBy< nn::gfx::ShaderCode >();

    m_CodeBlock.SetSize( StaticCastAuto( m_pTarget->codeSize ) );
}

void ShaderCodeBinarizer::Link( ShaderBinarizerContext* pContext )
{
    const BinarizationTarget* pResTarget = nullptr;
    pContext->LinkPtr( this, reinterpret_cast< const nn::util::BinPtr* >( &pResTarget->pCode ), &m_CodeBlock );
}

void ShaderCodeBinarizer::Convert( ShaderBinarizerContext* pContext )
{
    BinarizationTarget* pResTarget = Get< BinarizationTarget >( pContext->GetPtr() );
    pResTarget->codeSize = m_pTarget->codeSize;
    pResTarget->decompressedCodeSize = m_pTarget->decompressedCodeSize;
    memcpy( m_CodeBlock.Get( pContext->GetPtr() ), m_pTarget->pCode, m_pTarget->codeSize );
}

const nn::util::MemorySplitter::MemoryBlock* ShaderCodeBinarizer::GetShaderBinaryBlock() const
{
    return m_pParent->GetInfoData()->codeType == nn::gfx::ShaderCodeType_Binary ? &m_CodeBlock : nullptr;
}


void SourceArrayBinarizer::Initialize( const BinarizationTarget* pTarget,
    const ResShaderProgramBinarizer* pParent, ShaderStage stage )
{
    m_pTarget = pTarget;
    m_pParent = pParent;

    m_Stage = stage;

    m_SourceBlocks.resize( m_pTarget->codeArrayLength );
    std::for_each( m_SourceBlocks.begin(), m_SourceBlocks.end(),
            []( MergeBlock& source ){ source.merge = -1; } );
}

void SourceArrayBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_CodeArrayBlock );
    pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_CodeSizeArrayBlock );
    std::for_each( m_SourceBlocks.begin(), m_SourceBlocks.end(),
        [ pContext ]( MergeBlock& source ) { if( source.merge < 0 ) {
            pContext->AddMemoryBlock( StaticCastAuto( Section::ShaderSource ), &source.block );
        } } );
}

void SourceArrayBinarizer::CalculateSize()
{
    SetSizeBy< nn::gfx::SourceArrayCode >();
    m_CodeArrayBlock.SetSize( sizeof( nn::gfx::detail::Ptr< void > ) * m_pTarget->codeArrayLength );
    m_CodeSizeArrayBlock.SetSize( sizeof( uint32_t ) * m_pTarget->codeArrayLength );
    for( int idxSource = 0, sourceCount = NumericCastAuto( m_pTarget->codeArrayLength );
        idxSource < sourceCount; ++idxSource )
    {
        if( m_SourceBlocks[ idxSource ].merge < 0 )
        {
            m_SourceBlocks[ idxSource ].block.SetSize( m_pTarget->pCodeSizeArray[ idxSource ] + 1 );
        }
    }
}

void SourceArrayBinarizer::Link( ShaderBinarizerContext* pContext )
{
    const nn::gfx::SourceArrayCode* pResTarget = nullptr;
    pContext->LinkPtr( this, reinterpret_cast< const nn::util::BinTPtr< nn::gfx::detail::Ptr<
        const void > >* >( &pResTarget->pCodePtrArray ), &m_CodeArrayBlock );
    pContext->LinkPtr( this, reinterpret_cast< const nn::util::BinTPtr< const uint32_t >* >(
        &pResTarget->pCodeSizeArray ), &m_CodeSizeArrayBlock );

    const nn::gfx::detail::Ptr< const void >* pResCodeArray = nullptr;
    auto pVariationBinarizers = m_pParent->GetParent()->GetParent()->GetVariationBinarizers();
    for( int idxSource = 0, sourceCount = NumericCastAuto( m_pTarget->codeArrayLength );
        idxSource < sourceCount; ++idxSource )
    {
        auto pResPtr = reinterpret_cast< const nn::util::BinTPtr< const void >* >( pResCodeArray + idxSource );
        if( m_SourceBlocks[ idxSource ].merge < 0 )
        {
            pContext->LinkPtr( &m_CodeArrayBlock, pResPtr, &m_SourceBlocks[ idxSource ].block );
        }
        else
        {
            auto pMergeSource = pVariationBinarizers->at( m_SourceBlocks[
                idxSource ].merge ).GetSourceProgramBinarizer(
                    )->GetShaderCodeBinarizer< SourceArrayBinarizer >( m_Stage );
            pContext->LinkPtr( &m_CodeArrayBlock, pResPtr, &pMergeSource->m_SourceBlocks[ idxSource ].block );
        }
    }
}

void SourceArrayBinarizer::Convert( ShaderBinarizerContext* pContext )
{
    auto* pResTarget = Get< nn::gfx::SourceArrayCode >( pContext->GetPtr() );
    pResTarget->codeArrayLength = m_pTarget->codeArrayLength;

    auto* pResCodeSizeArray = m_CodeSizeArrayBlock.Get< uint32_t >( pContext->GetPtr() );
    for( int idxSource = 0, sourceCount = NumericCastAuto( m_pTarget->codeArrayLength );
        idxSource < sourceCount; ++idxSource )
    {
        pResCodeSizeArray[ idxSource ] = m_pTarget->pCodeSizeArray[ idxSource ];
        if( m_SourceBlocks[ idxSource ].merge < 0 )
        {
            void* pDst = m_SourceBlocks[ idxSource ].block.Get( pContext->GetPtr() );
            memcpy( pDst, m_pTarget->pCodePtrArray[ idxSource ], m_pTarget->pCodeSizeArray[ idxSource ] );
            static_cast< char* >( pDst )[ m_pTarget->pCodeSizeArray[ idxSource ] ] = '\0';
        }
    }
}

}
}
