﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/gfx/gfx_ResShader.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_ResShaderContainerBinarizer.h>
#include <gfxTool_ResShaderVariationBinarizer.h>
#include <gfxTool_ShaderBinarizerContext.h>
#include <gfxTool_ResShaderProgramBinarizer.h>
#include <gfxTool_NvnShaderPoolBinarizer.h>

namespace nn {
namespace gfxTool {

ResShaderContainerBinarizer::ResShaderContainerBinarizer()
    : m_pTarget()
    , m_pArg()
{
}

ResShaderContainerBinarizer::~ResShaderContainerBinarizer() /* = default; */
{
}

void ResShaderContainerBinarizer::Reset()
{
    m_pTarget = nullptr;
    m_pArg = nullptr;

    m_Variations.clear();

    this->nn::util::MemorySplitter::MemoryBlock::Initialize();
}

void ResShaderContainerBinarizer::Initialize( const BinarizationTarget* pTarget,
    const nngfxToolShaderConverterConvertArg* pArg )
{
    m_pTarget = pTarget;
    m_pArg = pArg;

    static int s_LowLevelApiTypeTable[] =
    {
        1, // GL
        2, // GX
        4, // NVN
        3, // D3D
        5  // VK
    };

    m_TargetLowLevelApiType = s_LowLevelApiTypeTable[ pArg->pCompileArg->targetLowLevelApiType ];
    m_TargetLowLevelApiVersion = StaticCastAuto( pArg->pCompileArg->targetLowLevelApiVersion );

    m_Variations.resize( pTarget->variationCount );
    for( int idxVariation = 0, variationCount = NumericCastAuto( m_Variations.size() );
        idxVariation < variationCount; ++idxVariation )
    {
        m_Variations[ idxVariation ].Initialize(
            pTarget->pVariationOutputArray + idxVariation, this, idxVariation );
    }

    if( pArg->pCompileArg->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Nvn )
    {
        if( pArg->pCompileArg->targetCodeType == nngfxToolShaderCompilerCodeType_Binary ||
            pArg->pCompileArg->targetCodeType == nngfxToolShaderCompilerCodeType_Binary_Source ||
            pArg->pCompileArg->targetCodeType == nngfxToolShaderCompilerCodeType_Binary_Ir )
        {
            m_pShaderPoolBinarizer.reset( new NvnShaderPoolBinarizer() );
            m_pShaderPoolBinarizer->Initialize( this );
        }
    }
}

void ResShaderContainerBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    std::for_each( m_Variations.begin(), m_Variations.end(),
        [ pContext ]( ResShaderVariationBinarizer& variation )
        { pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &variation ); } );
    std::for_each( m_Variations.begin(), m_Variations.end(),
        [ pContext ]( ResShaderVariationBinarizer& variation )
        { variation.RegisterChild( pContext ); } );

    if( m_pShaderPoolBinarizer.get() )
    {
        pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), m_pShaderPoolBinarizer.get() );
        m_pShaderPoolBinarizer->RegisterChild( pContext );
    }
}

void ResShaderContainerBinarizer::CalculateSize()
{
    SetSizeBy< ResTarget >();
}

void ResShaderContainerBinarizer::Link( ShaderBinarizerContext* pContext )
{
    const ResTarget* pResTarget = nullptr;
    if( m_Variations.size() > 0 )
    {
        pContext->LinkPtr( this, &pResTarget->pShaderVariationArray, &m_Variations[ 0 ] );
    }
    pContext->LinkPtr( this, &pResTarget->pShaderBinaryPool, m_pShaderPoolBinarizer.get() );
}

void ResShaderContainerBinarizer::Convert( ShaderBinarizerContext* pContext )
{
    auto pResTarget = Get< ResTarget >( pContext->GetPtr() );
    pResTarget->blockHeader.signature.SetPacked( nn::gfx::ResShaderContainer::Signature );

    pResTarget->targetApiType = NumericCastAuto( m_TargetLowLevelApiType );
    pResTarget->targetApiVersion = m_TargetLowLevelApiVersion;
    pResTarget->compilerVersion = m_pTarget->compilerVersion;
    pResTarget->shaderVariationCount = m_pTarget->variationCount;
    pResTarget->lowLevelCompilerVersion = m_pTarget->lowLevelCompilerVerison;

    pContext->AddHeader( &pResTarget->blockHeader );
}

}
}
