﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>

#include <nn/util/util_StringView.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_Common.h>
#include <gfxTool_ShaderSourceManager.h>
#include <gfxTool_SimplePreprocess.h>

namespace nn {
namespace gfxTool {

void ShaderSourceManager::Initialize( const nngfxToolShaderCompilerCompileArg* pArg )
{
    // プリプロセスする場合は不要
    for( int idxOption = 0, optionCount = NumericCastAuto( pArg->optionCount );
        idxOption < optionCount; ++idxOption )
    {
        auto& option = pArg->pOptionArray[ idxOption ];
        if( option.optionType == nngfxToolShaderCompilerOptionType_Common )
        {
            if( static_cast< const nngfxToolShaderCompilerCompileOptionCommon* >(
                option.pOption )->isPreprocessEnabled != 0 )
            {
                return;
            }
        }
    }

    ExpandIncludeCallback includeCallback( pArg->pReadIncludeFileCallback,
        pArg->pReadIncludeFileCallbackParam );

    Custom< std::vector< PreprocessorDirectiveCallbackBase* > >::Type callbacks;
    callbacks.reserve( 2 );
    callbacks.push_back( &includeCallback );
    PreprocessorDirectiveMultiCallback callback( &callbacks );

    includeCallback.SetSimplePreprocessCallback( &callback );

    for( int stage = 0; stage < static_cast< int >( ShaderStage::End ); ++stage )
    {
        auto& stageSource = GetStageSource( pArg, StaticCastAuto( stage ) );
        if( stageSource.pValue == nullptr )
        {
            continue;
        }
        auto pStageVariationDefinition = pArg->pVariationDefinitionArg ? GetStageVariationDefinition(
            pArg->pVariationDefinitionArg, StaticCastAuto( stage ) ) : nullptr;

        Custom< std::vector< Custom< std::string >::Type > >::Type macros;
        CommentOutMacroCallback macroCallback( &macros );
        if( pStageVariationDefinition )
        {
            macros.reserve( pStageVariationDefinition->preprocessorDefinitionDefinitionCount );
            for( int idxMacroDefinition = 0, macroDefinitionCount = NumericCastAuto(
                pStageVariationDefinition->preprocessorDefinitionDefinitionCount );
                idxMacroDefinition < macroDefinitionCount; ++idxMacroDefinition )
            {
                auto& macroDefinition =
                    pStageVariationDefinition->pPreprocessorDefinitionDefinitionArray[ idxMacroDefinition ];
                macros.emplace_back( macroDefinition.name.pValue, macroDefinition.name.length );
            }

            callbacks.push_back( &macroCallback );
        }

        m_pSources[ stage ].reset( new ShaderSource() );
        auto& pStageSource = m_pSources[ stage ];
        if( stageSource.pValue )
        {
            nn::util::string_view source( stageSource.pValue, stageSource.length );
            pStageSource->includeExpandedSource = SimplePreprocess( &source, &callback );
        }

        {
            if( pArg->targetLowLevelApiType != nngfxToolShaderCompilerLowLevelApiType_Nvn )
            {
                if( pArg->shaderSourceFormat == nngfxToolShaderCompilerShaderSourceFormat_Glsl )
                {
                    nn::util::string_view includeExpanded( pStageSource->includeExpandedSource.data(),
                        pStageSource->includeExpandedSource.length() );
                    pStageSource->variationBufferSource = CreateVariationBufferSource<
                        nngfxToolShaderCompilerShaderSourceFormat_Glsl >(
                        &pStageSource->beforeVariationBufferView, &pStageSource->afterVariationBufferView,
                        &includeExpanded, pStageVariationDefinition );
                }
                else if( pArg->shaderSourceFormat == nngfxToolShaderCompilerShaderSourceFormat_Hlsl )
                {
                    nn::util::string_view includeExpanded( pStageSource->includeExpandedSource.data(),
                        pStageSource->includeExpandedSource.length() );
                    pStageSource->variationBufferSource = CreateVariationBufferSource<
                        nngfxToolShaderCompilerShaderSourceFormat_Hlsl >(
                            &pStageSource->beforeVariationBufferView, &pStageSource->afterVariationBufferView,
                            &includeExpanded, pStageVariationDefinition );
                }
            }
        }
    }
}

}
}
