﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/util/util_Compression.h>

#include <nn/gfx/gfx_Enum.h>

#include <gfxTool_Compiler-vk-ir.h>
#include <gfxTool_GlobalWindow.h>
#include <gfxTool_ShaderCompilerContext.h>
#include <gfxTool_CompileOption-glsl.h>
#include <gfxTool_CompileOption-vk.h>
#include <gfxTool_GroupSource.h>
#include <gfxTool_VariationGrouper.h>
#include <gfxTool_CompileOptionManager.h>
#include <gfxTool_ShaderSourceManager.h>
#include <gfxTool_VariationManager.h>
#include <gfxTool_CompileOutput.h>
#include <gfxTool_OptionOutput.h>
#include <gfxTool_BoostPreprocessor.h>
#include <nn/gfxTool/gfxTool_Custom.h>

NN_PRAGMA_PUSH_WARNINGS
#include <vulkan/vulkan.h>
#include <glslang/Include/ShHandle.h>
#include <glslang/Public/ShaderLang.h>
// Temporarily add definitions for glslang::TIntermediate declaration
#define AMD_EXTENSIONS
#define NV_EXTENSIONS
#define ENABLE_HLSL
#define GLSLANG_OSINCLUDE_WIN32
#define ENABLE_OPT
#include "glslang/MachineIndependent/localintermediate.h"
#undef AMD_EXTENSIONS
#undef NV_EXTENSIONS
#undef ENABLE_HLSL
#undef GLSLANG_OSINCLUDE_WIN32
#undef ENABLE_OPT
#include <SPIRV/GlslangToSpv.h>
#include <SPIRV/SPVRemapper.h>
#include <SPIRV/doc.h>
#include <spirv-tools/libspirv.hpp>
#include <spirv-tools/optimizer.hpp>
NN_PRAGMA_POP_WARNINGS

namespace {

struct GlTypeAndGfxType
{
    GLenum glType;
    nngfxToolShaderCompilerReflectionVariableType gfxType;
} s_TypeTable[] =
{
    { GL_BOOL, nngfxToolShaderCompilerReflectionVariableType_Bool32 },
    { GL_INT, nngfxToolShaderCompilerReflectionVariableType_Int32 },
    { GL_UNSIGNED_INT, nngfxToolShaderCompilerReflectionVariableType_Uint32 },
    { GL_FLOAT, nngfxToolShaderCompilerReflectionVariableType_Float32 },
    { GL_DOUBLE, nngfxToolShaderCompilerReflectionVariableType_Float64 },

    { GL_BOOL_VEC2, nngfxToolShaderCompilerReflectionVariableType_Bool32x2 },
    { GL_BOOL_VEC3, nngfxToolShaderCompilerReflectionVariableType_Bool32x3 },
    { GL_BOOL_VEC4, nngfxToolShaderCompilerReflectionVariableType_Bool32x4 },
    { GL_INT_VEC2, nngfxToolShaderCompilerReflectionVariableType_Int32x2 },
    { GL_INT_VEC3, nngfxToolShaderCompilerReflectionVariableType_Int32x3 },
    { GL_INT_VEC4, nngfxToolShaderCompilerReflectionVariableType_Int32x4 },
    { GL_UNSIGNED_INT_VEC2, nngfxToolShaderCompilerReflectionVariableType_Uint32x2 },
    { GL_UNSIGNED_INT_VEC3, nngfxToolShaderCompilerReflectionVariableType_Uint32x3 },
    { GL_UNSIGNED_INT_VEC4, nngfxToolShaderCompilerReflectionVariableType_Uint32x4 },
    { GL_FLOAT_VEC2, nngfxToolShaderCompilerReflectionVariableType_Float32x2 },
    { GL_FLOAT_VEC3, nngfxToolShaderCompilerReflectionVariableType_Float32x3 },
    { GL_FLOAT_VEC4, nngfxToolShaderCompilerReflectionVariableType_Float32x4 },
    { GL_DOUBLE_VEC2, nngfxToolShaderCompilerReflectionVariableType_Float64x2 },
    { GL_DOUBLE_VEC3, nngfxToolShaderCompilerReflectionVariableType_Float64x3 },
    { GL_DOUBLE_VEC4, nngfxToolShaderCompilerReflectionVariableType_Float64x4 },

    { GL_FLOAT_MAT2, nngfxToolShaderCompilerReflectionVariableType_Float32x2x2 },
    { GL_FLOAT_MAT2x3, nngfxToolShaderCompilerReflectionVariableType_Float32x2x3 },
    { GL_FLOAT_MAT2x4, nngfxToolShaderCompilerReflectionVariableType_Float32x2x4 },
    { GL_FLOAT_MAT3x2, nngfxToolShaderCompilerReflectionVariableType_Float32x3x2 },
    { GL_FLOAT_MAT3, nngfxToolShaderCompilerReflectionVariableType_Float32x3x3 },
    { GL_FLOAT_MAT3x4, nngfxToolShaderCompilerReflectionVariableType_Float32x3x4 },
    { GL_FLOAT_MAT4x2, nngfxToolShaderCompilerReflectionVariableType_Float32x4x2 },
    { GL_FLOAT_MAT4x3, nngfxToolShaderCompilerReflectionVariableType_Float32x4x3 },
    { GL_FLOAT_MAT4, nngfxToolShaderCompilerReflectionVariableType_Float32x4x4 },
    { GL_DOUBLE_MAT2, nngfxToolShaderCompilerReflectionVariableType_Float64x2x2 },
    { GL_DOUBLE_MAT2x3, nngfxToolShaderCompilerReflectionVariableType_Float64x2x3 },
    { GL_DOUBLE_MAT2x4, nngfxToolShaderCompilerReflectionVariableType_Float64x2x4 },
    { GL_DOUBLE_MAT3x2, nngfxToolShaderCompilerReflectionVariableType_Float64x3x2 },
    { GL_DOUBLE_MAT3, nngfxToolShaderCompilerReflectionVariableType_Float64x3x3 },
    { GL_DOUBLE_MAT3x4, nngfxToolShaderCompilerReflectionVariableType_Float64x3x4 },
    { GL_DOUBLE_MAT4x2, nngfxToolShaderCompilerReflectionVariableType_Float64x4x2 },
    { GL_DOUBLE_MAT4x3, nngfxToolShaderCompilerReflectionVariableType_Float64x4x3 },
    { GL_DOUBLE_MAT4, nngfxToolShaderCompilerReflectionVariableType_Float64x4x4 },

    { GL_SAMPLER_1D, nngfxToolShaderCompilerReflectionVariableType_Sampler1D },
    { GL_SAMPLER_2D, nngfxToolShaderCompilerReflectionVariableType_Sampler2D },
    { GL_SAMPLER_3D, nngfxToolShaderCompilerReflectionVariableType_Sampler3D },
    { GL_SAMPLER_CUBE, nngfxToolShaderCompilerReflectionVariableType_SamplerCube },
    { GL_SAMPLER_1D_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler1DShadow },
    { GL_SAMPLER_2D_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler2DShadow },
    { GL_SAMPLER_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Sampler1DArray },
    { GL_SAMPLER_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Sampler2DArray },
    { GL_SAMPLER_1D_ARRAY_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler1DArrayShadow },
    { GL_SAMPLER_2D_ARRAY_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler2DArrayShadow },
    { GL_SAMPLER_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_Sampler2DMultisample },
    { GL_SAMPLER_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Sampler2DMultisampleArray },
    { GL_SAMPLER_CUBE_SHADOW, nngfxToolShaderCompilerReflectionVariableType_SamplerCubeShadow },
    { GL_SAMPLER_BUFFER, nngfxToolShaderCompilerReflectionVariableType_SamplerBuffer },
    { GL_SAMPLER_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_Sampler2DRect },
    { GL_SAMPLER_2D_RECT_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler2DRectShadow },
    { GL_INT_SAMPLER_1D, nngfxToolShaderCompilerReflectionVariableType_IntSampler1D },
    { GL_INT_SAMPLER_2D, nngfxToolShaderCompilerReflectionVariableType_IntSampler2D },
    { GL_INT_SAMPLER_3D, nngfxToolShaderCompilerReflectionVariableType_IntSampler3D },
    { GL_INT_SAMPLER_CUBE, nngfxToolShaderCompilerReflectionVariableType_IntSamplerCube },
    { GL_INT_SAMPLER_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSampler1DArray },
    { GL_INT_SAMPLER_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DArray },
    { GL_INT_SAMPLER_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DMultisample },
    { GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DMultisampleArray },
    { GL_INT_SAMPLER_BUFFER, nngfxToolShaderCompilerReflectionVariableType_IntSamplerBuffer },
    { GL_INT_SAMPLER_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DRect },
    { GL_UNSIGNED_INT_SAMPLER_1D, nngfxToolShaderCompilerReflectionVariableType_UintSampler1D },
    { GL_UNSIGNED_INT_SAMPLER_2D, nngfxToolShaderCompilerReflectionVariableType_UintSampler2D },
    { GL_UNSIGNED_INT_SAMPLER_3D, nngfxToolShaderCompilerReflectionVariableType_UintSampler3D },
    { GL_UNSIGNED_INT_SAMPLER_CUBE, nngfxToolShaderCompilerReflectionVariableType_UintSamplerCube },
    { GL_UNSIGNED_INT_SAMPLER_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSampler1DArray },
    { GL_UNSIGNED_INT_SAMPLER_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DArray },
    { GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DMultisample },
    { GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DMultisampleArray },
    { GL_UNSIGNED_INT_SAMPLER_BUFFER, nngfxToolShaderCompilerReflectionVariableType_UintSamplerBuffer },
    { GL_UNSIGNED_INT_SAMPLER_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DRect },
    { GL_SAMPLER_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_SamplerCubeMapArray },
    { GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW, nngfxToolShaderCompilerReflectionVariableType_SamplerCubeMapArrayShadow },
    { GL_INT_SAMPLER_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSamplerCubeMapArray },
    { GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSamplerCubeMapArray },

    { GL_IMAGE_1D, nngfxToolShaderCompilerReflectionVariableType_Image1D },
    { GL_IMAGE_2D, nngfxToolShaderCompilerReflectionVariableType_Image2D },
    { GL_IMAGE_3D, nngfxToolShaderCompilerReflectionVariableType_Image3D },
    { GL_IMAGE_CUBE, nngfxToolShaderCompilerReflectionVariableType_ImageCube },
    { GL_IMAGE_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Image1DArray },
    { GL_IMAGE_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Image2DArray },
    { GL_IMAGE_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_Image2DMultisample },
    { GL_IMAGE_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Image2DMultisampleArray },
    { GL_IMAGE_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_ImageCubeMapArray },
    { GL_IMAGE_BUFFER, nngfxToolShaderCompilerReflectionVariableType_ImageBuffer },
    { GL_IMAGE_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_Image2DRect },
    { GL_INT_IMAGE_1D, nngfxToolShaderCompilerReflectionVariableType_IntImage1D },
    { GL_INT_IMAGE_2D, nngfxToolShaderCompilerReflectionVariableType_IntImage2D },
    { GL_INT_IMAGE_3D, nngfxToolShaderCompilerReflectionVariableType_IntImage3D },
    { GL_INT_IMAGE_CUBE, nngfxToolShaderCompilerReflectionVariableType_IntImageCube },
    { GL_INT_IMAGE_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImage1DArray },
    { GL_INT_IMAGE_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImage2DArray },
    { GL_INT_IMAGE_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_IntImage2DMultisample },
    { GL_INT_IMAGE_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImage2DMultisampleArray },
    { GL_INT_IMAGE_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImageCubeMapArray },
    { GL_INT_IMAGE_BUFFER, nngfxToolShaderCompilerReflectionVariableType_IntImageBuffer },
    { GL_INT_IMAGE_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_IntImage2DRect },
    { GL_UNSIGNED_INT_IMAGE_1D, nngfxToolShaderCompilerReflectionVariableType_UintImage1D },
    { GL_UNSIGNED_INT_IMAGE_2D, nngfxToolShaderCompilerReflectionVariableType_UintImage2D },
    { GL_UNSIGNED_INT_IMAGE_3D, nngfxToolShaderCompilerReflectionVariableType_UintImage3D },
    { GL_UNSIGNED_INT_IMAGE_CUBE, nngfxToolShaderCompilerReflectionVariableType_UintImageCube },
    { GL_UNSIGNED_INT_IMAGE_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImage1DArray },
    { GL_UNSIGNED_INT_IMAGE_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImage2DArray },
    { GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_UintImage2DMultisample },
    { GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImage2DMultisampleArray },
    { GL_UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImageCubeMapArray },
    { GL_UNSIGNED_INT_IMAGE_BUFFER, nngfxToolShaderCompilerReflectionVariableType_UintImageBuffer },
    { GL_UNSIGNED_INT_IMAGE_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_UintImage2DRect }
};

std::string StringifyMessage( spv_message_level_t level, const char* source,
    const spv_position_t& position,
    const char* message )
{
    const char* levelString = nullptr;

    switch ( level )
    {
        case SPV_MSG_FATAL:
        {
            levelString = "fatal";
            break;
        }
        case SPV_MSG_INTERNAL_ERROR:
        {
            levelString = "internal error";
            break;
        }
        case SPV_MSG_ERROR:
        {
            levelString = "error";
            break;
        }
        case SPV_MSG_WARNING:
        {
            levelString = "warning";
            break;
        }
        case SPV_MSG_INFO:
        {
            levelString = "info";
            break;
        }
        case SPV_MSG_DEBUG:
        {
            levelString = "debug";
            break;
        }

        default: NN_UNEXPECTED_DEFAULT;
    }

    std::ostringstream oss;
    oss << levelString << ": ";

    if ( source )
    {
        oss << source << ":";
    }

    oss << position.line << ":" << position.column << ":";
    oss << position.index << ": ";

    if ( message )
    {
        oss << message;
    }
    return oss.str();
}

const char* GetLongStageName( nn::gfxTool::ShaderStage stage )
{
    static const char* s_StageNames[] =
    {
        "----Vertex Shader----",
        "----Hull Shader----",
        "----Domain Shader----",
        "----Geometry Shader----",
        "----Pixel Shader----",
        "----Compute Shader----"
    };

    return s_StageNames[ static_cast<int>( stage ) ];
}

const char* GetStageName(nn::gfxTool::ShaderStage stage)
{
    static const char* s_StageNames[] =
    {
        "Vertex",
        "Hull",
        "Domain",
        "Geometry",
        "Pixel",
        "Compute",
    };

    return s_StageNames[ static_cast<int>( stage ) ];
}

const char* GetStageName(EShLanguage stage)
{
    static const char* s_StageNames[] =
    {
        "Vertex",
        "Hull",
        "Domain",
        "Geometry",
        "Pixel",
        "Compute",
    };

    return s_StageNames[ static_cast<int>( stage ) ];
}

struct TDefaultIoResolver : public glslang::TIoMapResolver
{
    TDefaultIoResolver() :
        intermediate( nullptr ),
        lastValidStage( EShLangCount ),
        nextUniformLocation( 0 )
    { }

    void setLastValidStage( EShLanguage stage )
    {
        lastValidStage = stage;
    }

    void setIntermediate( const glslang::TIntermediate* inter )
    {
        intermediate = inter;
    }

    int getBaseBinding( glslang::TResourceType res, unsigned int set ) const
    {
        return selectBaseBinding( intermediate->getShiftBinding( res ),
            intermediate->getShiftBindingForSet( res, set ) );
    }

    const std::vector<std::string>& getResourceSetBinding() const
    {
        return intermediate->getResourceSetBinding();
    }

    bool doAutoBindingMapping() const
    {
        return intermediate->getAutoMapBindings();
    }

    bool doAutoLocationMapping() const
    {
        return intermediate->getAutoMapLocations();
    }

    typedef std::vector<int> TSlotSet;
    typedef std::unordered_map<int, TSlotSet> TSlotSetMap;
    TSlotSetMap slots;

    // Tracks in/out per stage and make sure they are consistent
    typedef std::unordered_map<std::string, int> TInOutMap;
    std::array<TInOutMap, EShLangCount> stageInputMap;
    std::array<TInOutMap, EShLangCount> stageOutputMap;

    // Track descriptor set slots globally
    typedef int TSlot;
    typedef int TSet;
    typedef std::pair<TSlot, TSet> TSlotSetPair;
    typedef std::unordered_map<std::string, TSlotSetPair> TDescriptorSetMap;
    TDescriptorSetMap descriptorSetResourceMap;

    TSlotSet::iterator findSlot( int set, int slot )
    {
        return std::lower_bound( slots[ set ].begin(), slots[ set ].end(), slot );
    }

    bool checkEmpty( int set, int slot )
    {
        TSlotSet::iterator at = findSlot( set, slot );
        return !( at != slots[ set ].end() && *at == slot );
    }

    std::vector< int >::iterator findLocation( EShLanguage stage, int location, bool isInput )
    {
        auto& v = isInput ? nextInputLocation[ static_cast<int>( stage ) ] : nextOutputLocation[ static_cast<int>( stage ) ];
        return std::lower_bound( v.begin(), v.end(), location );
    }

    bool checkEmptyLocation( EShLanguage stage, int location, bool isInput )
    {
        std::vector< int >::iterator at = findLocation( stage, location, isInput );
        auto& v = isInput ? nextInputLocation[ static_cast<int>( stage ) ] : nextOutputLocation[ static_cast<int>( stage ) ];
        return !( at != v.end() && *at == location );
    }

    int reserveSlot( int set, int slot )
    {
        TSlotSet::iterator at = findSlot( set, slot );

        // tolerate aliasing, by not double-recording aliases
        // (policy about appropriateness of the alias is higher up)
        if ( at == slots[ set ].end() || *at != slot )
        {
            slots[ set ].insert( at, slot );
        }

        return slot;
    }

    int getFreeSlot( int set, int base )
    {
        TSlotSet::iterator at = findSlot( set, base );
        if ( at == slots[ set ].end() )
        {
            return reserveSlot( set, base );
        }

        // look in locksteps, if they not match, then there is a free slot
        for ( ; at != slots[ set ].end(); ++at, ++base )
        {
            if ( *at != base )
            {
                break;
            }
        }
        return reserveSlot( set, base );
    }

    int reserveLocation( EShLanguage stage, int location, bool isInput )
    {
        std::vector< int >::iterator at = findLocation( stage, location, isInput );
        auto& v = isInput ? nextInputLocation[ static_cast<int>( stage ) ] : nextOutputLocation[ static_cast<int>( stage ) ];

        // tolerate aliasing, by not double-recording aliases
        // (policy about appropriateness of the alias is higher up)
        if ( at == v.end() || *at != location )
        {
            v.insert( at, location );
        }

        return location;
    }

    int getFreeLocation( EShLanguage stage, int location, bool isInput )
    {
        std::vector< int >::iterator at = findLocation( stage, location, isInput );
        auto& v = isInput ? nextInputLocation[ static_cast<int>( stage ) ] : nextOutputLocation[ static_cast<int>( stage ) ];
        if ( at == v.end() )
        {
            return reserveLocation( stage, location, isInput );
        }

        // look in locksteps, if they not match, then there is a free slot
        for ( ; at != v.end(); ++at, ++location )
        {
            if ( *at != location )
            {
                break;
            }
        }
        return reserveLocation( stage, location, isInput );
    }

    bool validateBinding( EShLanguage /*stage*/, const char* /*name*/, const glslang::TType& /*type*/, bool /*isLive*/ ) override
    {
        return true;
    }

    void addFixedBinding( const std::string& name, int set, int binding )
    {
        auto resource = descriptorSetResourceMap.find( std::string( name ) );
        if ( resource == descriptorSetResourceMap.end() )
        {
            descriptorSetResourceMap[ name ] = TSlotSetPair( TSlot( binding ), TSet( set ) );
            (void) reserveSlot( set, binding );
        }
    }

    int resolveBinding( EShLanguage stage, const char* name, const glslang::TType& type, bool isLive ) override
    {
        const int set = getLayoutSet( type );
        const char* lookupName = name;

        // Blocks without a final identifier are considered anonymous and name contains "anon@i"
        // Replace this with the type name.
        if ( ( isSsboType( type ) || isUboType( type ) ) && strncmp( name, "anon", 4 ) == 0 )
        {
            lookupName = type.getTypeName().c_str();
        }

        if ( type.getQualifier().hasBinding() )
        {
            auto resource = descriptorSetResourceMap.find( std::string( name ) );
            if ( resource == descriptorSetResourceMap.end() )
            {
                descriptorSetResourceMap[ lookupName ] =
                    TSlotSetPair( TSlot( static_cast< int >( type.getQualifier().layoutBinding ) ),TSet( set ) );

                if ( isImageType( type ) )
                {
                    return reserveSlot( set, getBaseBinding( glslang::EResImage, set ) + type.getQualifier().layoutBinding );
                }

                if ( isTextureType( type ) )
                {
                    return reserveSlot( set, getBaseBinding( glslang::EResTexture, set ) + type.getQualifier().layoutBinding );
                }

                if ( isSsboType( type ) )
                {
                    return reserveSlot( set, getBaseBinding( glslang::EResSsbo, set ) + type.getQualifier().layoutBinding );
                }

                if ( isSamplerType( type ) )
                {
                    return reserveSlot( set, getBaseBinding( glslang::EResSampler, set ) + type.getQualifier().layoutBinding );
                }

                if ( isUboType( type ) )
                {
                    return reserveSlot( set, getBaseBinding( glslang::EResUbo, set ) + type.getQualifier().layoutBinding );
                }
            }
            else
            {
                if ( resource->second.first != -1 && resource->second.first != static_cast< int >( type.getQualifier().layoutBinding ) )
                {
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError,
                        "Explicity resource binding %d for \"%s\" in stage \"%s\" does not match previous binding %d in other stages.",
                        type.getQualifier().layoutBinding, lookupName, GetStageName( stage ), resource->second.first );
                }
                resource->second.first = TSlot( type.getQualifier().layoutBinding );
            }
        }
        else if ( isLive && doAutoBindingMapping() )
        {
            auto resource = descriptorSetResourceMap.find( lookupName );
            if ( resource == descriptorSetResourceMap.end() )
            {
                int slot = -1;
                // find free slot, the caller did make sure it passes all vars with binding
                // first and now all are passed that do not have a binding and needs one
                if ( isImageType( type ) )
                {
                    slot = getFreeSlot( set, getBaseBinding( glslang::EResImage, set ) );
                }
                else if ( isTextureType( type ) )
                {
                    slot = getFreeSlot( set, getBaseBinding( glslang::EResTexture, set ) );
                }
                else if ( isSsboType( type ) )
                {
                    slot = getFreeSlot( set, getBaseBinding( glslang::EResSsbo, set ) );
                }
                else if ( isSamplerType( type ) )
                {
                    slot = getFreeSlot( set, getBaseBinding( glslang::EResSampler, set ) );
                }
                else if ( isUboType( type ) )
                {
                    slot = getFreeSlot( set, getBaseBinding( glslang::EResUbo, set ) );
                }

                descriptorSetResourceMap[ lookupName ] = TSlotSetPair( TSlot( slot ), TSet( set ) );

                return slot;
            }
            else
            {
                if ( resource->second.first == -1 )
                {
                    int slot = -1;
                    // find free slot, the caller did make sure it passes all vars with binding
                    // first and now all are passed that do not have a binding and needs one
                    if ( isImageType( type ) )
                    {
                        slot = getFreeSlot( set, getBaseBinding( glslang::EResImage, set ) );
                    }
                    else if ( isTextureType( type ) )
                    {
                        slot = getFreeSlot( set, getBaseBinding( glslang::EResTexture, set ) );
                    }
                    else if ( isSsboType( type ) )
                    {
                        slot = getFreeSlot( set, getBaseBinding( glslang::EResSsbo, set ) );
                    }
                    else if ( isSamplerType( type ) )
                    {
                        slot = getFreeSlot( set, getBaseBinding( glslang::EResSampler, set ) );
                    }
                    else if ( isUboType( type ) )
                    {
                        slot = getFreeSlot( set, getBaseBinding( glslang::EResUbo, set ) );
                    }

                    resource->second.first = slot;
                }

                return resource->second.first;
            }
        }

        return -1;
    }

    int resolveSet( EShLanguage /*stage*/, const char* name, const glslang::TType& type, bool /*isLive*/ ) override
    {
        int setNumber = getLayoutSet(type);

        auto resource = descriptorSetResourceMap.find( std::string( name ) );
        if ( resource == descriptorSetResourceMap.end() )
        {
            descriptorSetResourceMap[ name ] = TSlotSetPair(TSlot(-1), TSet(setNumber));
        }
        else
        {
            NN_SDK_ASSERT( resource->second.second == -1 || resource->second.second == setNumber );
            resource->second.second = TSet( setNumber );
        }

        return setNumber;
    }

    int resolveUniformLocation( EShLanguage /*stage*/, const char* /*name*/, const glslang::TType& type, bool /*isLive*/ ) override
    {
        // kick out of not doing this
        if ( !doAutoLocationMapping() )
        {
            return -1;
        }

        // no locations added if already present, a built-in variable, a block, or an opaque
        if ( type.getQualifier().hasLocation() || type.isBuiltIn() ||
            type.getBasicType() == glslang::EbtBlock || type.containsOpaque() )
        {
            return -1;
        }

        // no locations on blocks of built-in variables
        if ( type.isStruct() )
        {
            if ( type.getStruct()->size() < 1 )
            {
                return -1;
            }
            if ( ( *type.getStruct() )[ 0 ].type->isBuiltIn() )
            {
                return -1;
            }
        }

        return nextUniformLocation++;
    }

    bool validateInOut( EShLanguage /*stage*/, const char* /*name*/, const glslang::TType& /*type*/, bool /*isLive*/ ) override
    {
        return true;
    }

    int resolveInOutLocation( EShLanguage stage, const char* name, const glslang::TType& type, bool /* isLive */ ) override
    {
        if ( !doAutoLocationMapping() )
        {
            return -1;
        }

        // Don't add a location if it's a built-in variable
        if ( type.isBuiltIn() )
        {
            return -1;
        }

        // no locations on blocks of built-in variables
        if ( type.isStruct() )
        {
            if ( type.getStruct()->size() < 1 )
            {
                return -1;
            }

            if ( ( *type.getStruct() )[ 0 ].type->isBuiltIn() )
            {
                return -1;
            }
        }

        int location = -1;
        if ( stage != EShLangVertex && type.getQualifier().isPipeInput())
        {
            // Make sure output from previous stage matches input from current stage.
            auto varying = stageOutputMap[ static_cast<int>( lastValidStage ) ].find( name );
            if ( varying == stageOutputMap[ static_cast<int>( lastValidStage ) ].end() )
            {
                // Normally it is bad behavior for this to occur but there are times where
                // the previous stage defines the output with a different name than the
                // input in the next stage. In this case the base we can do is give it
                // the next location.

                if ( type.getQualifier().hasLocation() )
                {
                    location = type.getQualifier().layoutLocation;
                    if ( !checkEmptyLocation(stage, location, type.getQualifier().isPipeInput() ) )
                    {
                        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError,
                            "Explicit layout location %d for %s in the %s stage conflicts with previous automatic location generation!\n",
                            type.getQualifier().layoutLocation, name, GetStageName( stage ) );
                    }
                }
                else
                {
                    location = getFreeLocation( stage, 0, type.getQualifier().isPipeInput() );
                }
            }
            else
            {
                location = varying->second;
            }
        }
        else
        {
            if ( type.getQualifier().hasLocation() )
            {
                location = reserveLocation( stage, type.getQualifier().layoutLocation, type.getQualifier().isPipeInput() );
            }
            else
            {
                location = getFreeLocation( stage, 0, type.getQualifier().isPipeInput() );
            }
        }

        if ( type.getQualifier().isPipeInput() )
        {
            stageInputMap[ static_cast< int >( stage ) ][ name ] = location;
        }
        else
        {
            stageOutputMap[ static_cast< int >( stage ) ][ name ] = location;
        }

        return location;
    }

    int resolveInOutComponent( EShLanguage /*stage*/, const char* /*name*/, const glslang::TType& /*type*/, bool /*isLive*/ ) override
    {
        return -1;
    }

    int resolveInOutIndex( EShLanguage /*stage*/, const char* /*name*/, const glslang::TType& /*type*/, bool /*isLive*/ ) override
    {
        return -1;
    }

    void notifyBinding( EShLanguage, const char* /*name*/, const glslang::TType&, bool /*isLive*/ ) override {}
    void notifyInOut( EShLanguage, const char* /*name*/, const glslang::TType&, bool /*isLive*/ ) override {}
    void endNotifications( EShLanguage ) override {}
    void beginNotifications( EShLanguage ) override {}
    void beginResolve( EShLanguage ) override {}
    void endResolve( EShLanguage ) override {}

protected:
    const glslang::TIntermediate* intermediate;
    EShLanguage lastValidStage;
    int nextUniformLocation;
    std::array<std::vector<int>, EShLangCount> nextInputLocation;
    std::array<std::vector<int>, EShLangCount> nextOutputLocation;

    // Return descriptor set specific base if there is one, and the generic base otherwise.
    int selectBaseBinding( int base, int descriptorSetBase ) const
    {
        return descriptorSetBase != -1 ? descriptorSetBase : base;
    }

    int getLayoutSet( const glslang::TType& type )
    {
        if ( type.getQualifier().hasSet() )
        {
            return type.getQualifier().layoutSet;
        }

        // If a command line or API option requested a single descriptor set,
        // use that (if not overrided by spaceN)
        if ( getResourceSetBinding().size() == 1 )
        {
            return atoi( getResourceSetBinding()[ 0 ].c_str() );
        }

        return 0;
    }

    static bool isSamplerType( const glslang::TType& type )
    {
        return type.getBasicType() == glslang::EbtSampler && type.getSampler().isPureSampler();
    }

    static bool isTextureType( const glslang::TType& type )
    {
        return ( type.getBasicType() == glslang::EbtSampler &&
            ( type.getSampler().isTexture() || type.getSampler().isSubpass() ) );
    }

    static bool isUboType( const glslang::TType& type )
    {
        return type.getQualifier().storage == glslang::EvqUniform;
    }

    static bool isImageType( const glslang::TType& type )
    {
        return type.getBasicType() == glslang::EbtSampler && type.getSampler().isImage();
    }

    static bool isSsboType( const glslang::TType& type )
    {
        return type.getQualifier().storage == glslang::EvqBuffer;
    }
};

nngfxToolShaderCompilerReflectionVariableType GlTypeToGfxType( GLenum glType )
{
    auto pEnd = s_TypeTable + sizeof( s_TypeTable ) / sizeof( *s_TypeTable );
    auto found = std::find_if( s_TypeTable, pEnd, [ glType ]( const decltype( *s_TypeTable ) element )
    {
        return element.glType == glType;
    } );
    return found == pEnd ? nngfxToolShaderCompilerReflectionVariableType_Unknown : found->gfxType;
}

const EShLanguage s_StageTable[] =
{
    EShLangVertex,
    EShLangTessControl,
    EShLangTessEvaluation,
    EShLangGeometry,
    EShLangFragment,
    EShLangCompute
};

EShLanguage GfxStageToShLang( nn::gfxTool::ShaderStage stage )
{
    NN_SDK_ASSERT_MINMAX( static_cast< int >( stage ), 0, EShLangCount );
    return s_StageTable[ static_cast< int >( stage ) ];
}


void InitResources( TBuiltInResource* pResources )
{
    pResources->maxLights = 32;
    pResources->maxClipPlanes = 6;
    pResources->maxTextureUnits = 32;
    pResources->maxTextureCoords = 32;
    pResources->maxVertexAttribs = 64;
    pResources->maxVertexUniformComponents = 4096;
    pResources->maxVaryingFloats = 64;
    pResources->maxVertexTextureImageUnits = 32;
    pResources->maxCombinedTextureImageUnits = 80;
    pResources->maxTextureImageUnits = 32;
    pResources->maxFragmentUniformComponents = 4096;
    pResources->maxDrawBuffers = 32;
    pResources->maxVertexUniformVectors = 128;
    pResources->maxVaryingVectors = 8;
    pResources->maxFragmentUniformVectors = 16;
    pResources->maxVertexOutputVectors = 16;
    pResources->maxFragmentInputVectors = 15;
    pResources->minProgramTexelOffset = -8;
    pResources->maxProgramTexelOffset = 7;
    pResources->maxClipDistances = 8;
    pResources->maxComputeWorkGroupCountX = 65535;
    pResources->maxComputeWorkGroupCountY = 65535;
    pResources->maxComputeWorkGroupCountZ = 65535;
    pResources->maxComputeWorkGroupSizeX = 1024;
    pResources->maxComputeWorkGroupSizeY = 1024;
    pResources->maxComputeWorkGroupSizeZ = 64;
    pResources->maxComputeUniformComponents = 1024;
    pResources->maxComputeTextureImageUnits = 16;
    pResources->maxComputeImageUniforms = 8;
    pResources->maxComputeAtomicCounters = 8;
    pResources->maxComputeAtomicCounterBuffers = 1;
    pResources->maxVaryingComponents = 60;
    pResources->maxVertexOutputComponents = 64;
    pResources->maxGeometryInputComponents = 64;
    pResources->maxGeometryOutputComponents = 128;
    pResources->maxFragmentInputComponents = 128;
    pResources->maxImageUnits = 8;
    pResources->maxCombinedImageUnitsAndFragmentOutputs = 8;
    pResources->maxCombinedShaderOutputResources = 8;
    pResources->maxImageSamples = 0;
    pResources->maxVertexImageUniforms = 0;
    pResources->maxTessControlImageUniforms = 0;
    pResources->maxTessEvaluationImageUniforms = 0;
    pResources->maxGeometryImageUniforms = 0;
    pResources->maxFragmentImageUniforms = 8;
    pResources->maxCombinedImageUniforms = 8;
    pResources->maxGeometryTextureImageUnits = 16;
    pResources->maxGeometryOutputVertices = 256;
    pResources->maxGeometryTotalOutputComponents = 1024;
    pResources->maxGeometryUniformComponents = 1024;
    pResources->maxGeometryVaryingComponents = 64;
    pResources->maxTessControlInputComponents = 128;
    pResources->maxTessControlOutputComponents = 128;
    pResources->maxTessControlTextureImageUnits = 16;
    pResources->maxTessControlUniformComponents = 1024;
    pResources->maxTessControlTotalOutputComponents = 4096;
    pResources->maxTessEvaluationInputComponents = 128;
    pResources->maxTessEvaluationOutputComponents = 128;
    pResources->maxTessEvaluationTextureImageUnits = 16;
    pResources->maxTessEvaluationUniformComponents = 1024;
    pResources->maxTessPatchComponents = 120;
    pResources->maxPatchVertices = 32;
    pResources->maxTessGenLevel = 64;
    pResources->maxViewports = 16;
    pResources->maxVertexAtomicCounters = 0;
    pResources->maxTessControlAtomicCounters = 0;
    pResources->maxTessEvaluationAtomicCounters = 0;
    pResources->maxGeometryAtomicCounters = 0;
    pResources->maxFragmentAtomicCounters = 8;
    pResources->maxCombinedAtomicCounters = 8;
    pResources->maxAtomicCounterBindings = 1;
    pResources->maxVertexAtomicCounterBuffers = 0;
    pResources->maxTessControlAtomicCounterBuffers = 0;
    pResources->maxTessEvaluationAtomicCounterBuffers = 0;
    pResources->maxGeometryAtomicCounterBuffers = 0;
    pResources->maxFragmentAtomicCounterBuffers = 1;
    pResources->maxCombinedAtomicCounterBuffers = 1;
    pResources->maxAtomicCounterBufferSize = 16384;
    pResources->maxTransformFeedbackBuffers = 4;
    pResources->maxTransformFeedbackInterleavedComponents = 64;
    pResources->maxCullDistances = 8;
    pResources->maxCombinedClipAndCullDistances = 8;
    pResources->maxSamples = 4;
    pResources->limits.nonInductiveForLoops = 1;
    pResources->limits.whileLoops = 1;
    pResources->limits.doWhileLoops = 1;
    pResources->limits.generalUniformIndexing = 1;
    pResources->limits.generalAttributeMatrixVectorIndexing = 1;
    pResources->limits.generalVaryingIndexing = 1;
    pResources->limits.generalSamplerIndexing = 1;
    pResources->limits.generalVariableIndexing = 1;
    pResources->limits.generalConstantMatrixVectorIndexing = 1;
}

struct StageIntermediateResult
{
    glslang::TShader* pShader;
    std::shared_ptr< nn::gfxTool::Custom< std::string >::Type > pInfoLog;
    std::vector< const GLchar* > sources;
    std::vector< GLint > lengths;
    nn::gfxTool::Custom< std::string >::Type preprocessedSource;
    nn::gfxTool::BoostPreprocessor* pPreprocessor;
};

const char* OffsetBom( const char* pSource )
{
    return nn::gfxTool::IsBom( pSource ) ? ( pSource + 3 ) : pSource;
}

size_t LengthBom( const char* pSource, size_t len )
{
    return nn::gfxTool::IsBom( pSource ) ? len - 3 : len;
}

void PrepareStage( StageIntermediateResult* pResult,
    const nn::gfxTool::ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg* pArg,
    nn::gfxTool::ShaderStage stage, int idxVariation )
{
    auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    auto pGlslOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Glsl >();

    auto variationConstantGroup = pContext->GetVariationManager(
        )->GetVariationConstantGroup( stage )->GetVariationToGroupTable().at( idxVariation );
    auto& variationConstantSource = pContext->GetVariationManager(
        )->GetVariationConstantSource( stage )->GetSources().at( variationConstantGroup );
    auto preprocessorDefinitionGroup = pContext->GetVariationManager(
        )->GetPreprocessorDefinitionGroup( stage )->GetVariationToGroupTable().at( idxVariation );
    auto& preprocessorDefinitionSource = pContext->GetVariationManager(
        )->GetPreprocessorDefinitionSource( stage )->GetSources().at( preprocessorDefinitionGroup );

    auto pShaderSource = pContext->GetShaderSourceManager(
        )->GetShaderSource( StaticCastAuto( stage ) );

    if ( pCommonOption->IsPreprocessEnabled() )
    {
        auto& stageSource = GetStageSource( pArg, stage );
        nn::gfxTool::Custom< std::string >::Type source;

        // Remove UTF-8 markers since glslang cannot handle this (issue #425)
        source.append( OffsetBom( pGlslOption->GetGlslHeader()->c_str() ) );
        source.append( OffsetBom( pCommonOption->GetPreprocessorDefinitionSource()->c_str() ) );
        source.append( OffsetBom( preprocessorDefinitionSource.c_str() ) );
        source.append( OffsetBom( variationConstantSource.c_str() ) );

        // Reconfirm all lengths since the file length may be inaccurate.
        uint32_t len = stageSource.length;
        len = static_cast< uint32_t >( strnlen( stageSource.pValue, len ) );
        if ( nn::gfxTool::IsBom( stageSource.pValue ) )
        {
            source.append( stageSource.pValue + 3, len - 3 );
        }
        else
        {
            source.append( stageSource.pValue, len );
        }

        nn::gfxTool::BoostPreprocessor& preprocessor = *pResult->pPreprocessor;
        preprocessor.SetReadFileCallback( pArg->pReadIncludeFileCallback,
            pArg->pReadIncludeFileCallbackParam );
        if ( pArg->pVariationDefinitionArg )
        {
            preprocessor.SetVariationDefinition( GetStageVariationDefinition(
                pArg->pVariationDefinitionArg, stage ) );
        }
        preprocessor.SetVariationConstantEmulationEnabled( true );
        auto& uniformRegisterBlockName = pCommonOption->GetUniformRegisterBlockName();
        if ( !uniformRegisterBlockName.empty() )
        {
            preprocessor.SetUniformRegisterToBlockEnabled( true );
        }
        preprocessor.SetInvertYEnabled( stage == nn::gfxTool::ShaderStage::Vertex && pCommonOption->IsInvertYEnabled() );
        preprocessor.SetRemapZEnabled(  stage == nn::gfxTool::ShaderStage::Vertex && pCommonOption->IsRemapZEnabled() );
        preprocessor.Preprocess( source.c_str(), source.length() );
    }
    else
    {
        // Create one large string since this simplifies decoding glslang errors
        // Reconfirm all lengths since the file length may be inaccurate from some tools.
        pResult->preprocessedSource.append( OffsetBom( pGlslOption->GetGlslHeader()->c_str() ) );
        pResult->preprocessedSource.append( OffsetBom( pCommonOption->GetPreprocessorDefinitionSource()->c_str() ) );
        pResult->preprocessedSource.append( OffsetBom( preprocessorDefinitionSource.c_str() ) );
        pResult->preprocessedSource.append( OffsetBom( variationConstantSource.c_str() ) );
        pResult->preprocessedSource.append( OffsetBom( pShaderSource->beforeVariationBufferView.data() ),
            LengthBom( pShaderSource->beforeVariationBufferView.data(), pShaderSource->beforeVariationBufferView.size() ) );
        pResult->preprocessedSource.append( OffsetBom( pShaderSource->variationBufferSource.c_str() ) );
        pResult->preprocessedSource.append( OffsetBom( pShaderSource->afterVariationBufferView.data() ),
            LengthBom( pShaderSource->afterVariationBufferView.data(), pShaderSource->afterVariationBufferView.size() ) );

        pResult->sources.push_back( pResult->preprocessedSource.c_str() );
        pResult->lengths.push_back( nn::gfxTool::NumericCastAuto( pResult->preprocessedSource.size() ) );
    }
}

void CompileStage( StageIntermediateResult* pResult,
    const nn::gfxTool::ShaderCompilerContext* pContext,
    nn::gfxTool::ShaderStage stage, int idxVariation )
{
    auto pVkOption = pContext->GetCompileOptionManager()->GetCompileOption< static_cast<
        nngfxToolShaderCompilerOptionType >( nngfxToolShaderCompilerOptionType_Vk )>();

    pResult->pShader = new glslang::TShader( GfxStageToShLang( stage ) );
    pResult->pShader->setStringsWithLengths( &pResult->sources[ 0 ], &pResult->lengths[ 0 ], nn::gfxTool::StaticCastAuto( pResult->sources.size() ) );
    pResult->pShader->setAutoMapBindings( pVkOption->GetOption()->enableAutoMapUniformBinding != 0 );
    pResult->pShader->setAutoMapLocations( pVkOption->GetOption()->enableAutoMapInOutLocation != 0 );

    ::TBuiltInResource Resources;
    InitResources( &Resources );

    // Enable SPIR-V and Vulkan rules when parsing GLSL
    EShMessages messages = ( EShMessages ) ( EShMsgSpvRules | EShMsgVulkanRules );
    bool compileStatus = pResult->pShader->parse( &Resources, 100, false, messages );

    auto infoLog = pResult->pShader->getInfoLog();
    auto infoLogSize = infoLog != nullptr ? strlen( infoLog ) : 0;
    if ( infoLogSize > 0 )
    {
        pResult->pInfoLog->append( &infoLog[ 0 ], infoLogSize );
    }

    auto infoDebugLog = pResult->pShader->getInfoDebugLog();
    auto infoDebugLogSize = infoLog != nullptr ? strlen( infoDebugLog ) : 0;
    if ( infoDebugLogSize > 0 )
    {
        pResult->pInfoLog->append( &infoDebugLog[ 0 ], infoDebugLogSize );
    }

    if ( !compileStatus )
    {
        auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
            nngfxToolShaderCompilerOptionType_Common >();
        std::string source;
        for ( size_t idx = 0; idx < pResult->sources.size(); idx++ )
        {
            source.append( pResult->sources[ idx ] );
        }
        auto result = nn::gfxTool::ConvertEncoding( nn::gfxTool::AddLineNumber( nn::util::string_view(
            source.c_str(), source.length() ) ), pCommonOption->GetCodePage(), 0 );
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError,
            "\n[Variation: %d]\n%s\n----Error Log----\n%s",
            idxVariation, result.c_str(), pResult->pInfoLog->c_str() );
    }
}

nn::gfxTool::Custom< std::string >::Type CreateErrorSource(
    nn::gfxTool::ShaderStage stage, const StageIntermediateResult* pResult, int codePage )
{
    if ( pResult == nullptr )
    {
        return nn::gfxTool::Custom< std::string >::Type();
    }

    nn::gfxTool::Custom< std::string >::Type ret;
    ret.append( GetLongStageName( stage ) ).append( "\n" );
    nn::gfxTool::Custom< std::string >::Type source;
    for ( int idxSource = 0; idxSource < static_cast< int >( pResult->sources.size() ); ++idxSource )
    {
        source.append( pResult->sources[ idxSource ], pResult->lengths[ idxSource ] );
    }
    ret.append( nn::gfxTool::ConvertEncoding( nn::gfxTool::AddLineNumber( nn::util::string_view(
        source.c_str(), source.length() ) ), codePage, 0 ) );
    ret.append( "\n" ).append( *pResult->pInfoLog );
    return std::move( ret );
}

void RetrieveOutput( nn::gfxTool::ProgramOutput* pOutput,
    glslang::TProgram* pProgram, TDefaultIoResolver* pResolver, nn::gfxTool::ShaderStage stage, bool compress, nngfxToolShaderCompilerDebugInfoLevel debugLevel )
{
    pResolver->setIntermediate( pProgram->getIntermediate( GfxStageToShLang( stage ) ) );
    if ( !pProgram->mapIO( pResolver ) )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Failed to auto-map location/binding information!\n%s\n",
            pProgram->getInfoLog() );
    }
    pResolver->setLastValidStage( GfxStageToShLang( stage ) );

    std::vector<unsigned int > spirv;
    glslang::GlslangToSpv( *pProgram->getIntermediate( GfxStageToShLang( stage ) ), spirv );

    if ( debugLevel <= nngfxToolShaderCompilerDebugInfoLevel_Debug )
    {
        spv_target_env targetEnv = SPV_ENV_VULKAN_1_0;
        spvtools::Optimizer optimizer( targetEnv );
        optimizer.SetMessageConsumer( [ ] ( spv_message_level_t level, const char* source,
            const spv_position_t& position,
            const char* message ) {
            std::cerr << StringifyMessage( level, source, position, message )
                << std::endl;
        } );

        // Equivalent to spirv-opt -O flag
        optimizer.RegisterPerformancePasses();

        bool ok = optimizer.Run( spirv.data(), spirv.size(), &spirv );
        if ( !ok )
        {
            NN_GFXTOOL_PRINT_WARNING( "Unable to perform SPIRV optimizations, resulting SPIRV is unchanged." );
        }
        else
        {
            spv::spirvbin_t( 0 ).remap( spirv, spv::spirvbin_t::DCE_ALL );
        }
    }

    size_t binaryLength = spirv.size() * sizeof( spirv[ 0 ] );
    std::shared_ptr< char > pBinary( new char[ binaryLength ], std::default_delete< char[] >() );
    std::memcpy( pBinary.get(), spirv.data(), binaryLength );
    size_t decompressedCodeSize = 0;
    if ( compress )
    {
        decompressedCodeSize = nn::gfxTool::NumericCastAuto( binaryLength );
        nn::gfxTool::Custom< std::unique_ptr< char[] > >::Type pTmpCompressdBinary( new char[ binaryLength ] );
        nn::gfxTool::Custom< std::unique_ptr< char[] > >::Type pWorkBuffer(
            new char[ nn::util::CompressZlibWorkBufferSizeDefault ] );
        size_t compressedSize;
        if ( !nn::util::CompressZlib( &compressedSize, pTmpCompressdBinary.get(), binaryLength,
            pBinary.get(), binaryLength, pWorkBuffer.get(), nn::util::CompressZlibWorkBufferSizeDefault ) )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_FailedToCompress );
        }
        binaryLength = nn::gfxTool::NumericCastAuto( compressedSize );
        pBinary.reset( new char[ compressedSize ], std::default_delete< char[] >() );
        memcpy( pBinary.get(), pTmpCompressdBinary.get(), compressedSize );
    }
    pOutput->GetInfo()->binaryFormat = 0;
    pOutput->SetShaderCode( stage, pBinary,
        nn::gfxTool::NumericCastAuto( binaryLength ), nn::gfxTool::NumericCastAuto( decompressedCodeSize ) );
}

void DumpShaderSource( nn::gfxTool::ProgramOutput* pOutput,
    const StageIntermediateResult* pResult, nn::gfxTool::ShaderStage stage )
{
    if ( pResult == nullptr || pResult->sources.size() == 0 )
    {
        return;
    }

    std::shared_ptr< nn::gfxTool::Custom< std::string >::Type > pDump( new nn::gfxTool::Custom< std::string >::Type() );
    for ( int idxSource = 0; idxSource < static_cast< int >( pResult->sources.size() ); ++idxSource )
    {
        pDump.get()->append( pResult->sources[ idxSource ], pResult->lengths[ idxSource ] );
    }
    auto pOptionOutputCommonProgram = static_cast< nn::gfxTool::OptionOutputProgramCommon* >(
        pOutput->GetOptionOutput( nngfxToolShaderCompilerOptionOutputType_ProgramCommon ) );
    pOptionOutputCommonProgram->GetOptionOutputStageCommon( stage )->SetDump( pDump );
}

bool RetrieveReflection( nn::gfxTool::OptionOutputReflection* pOutput,
    glslang::TProgram** pProgram, bool )
{
    bool status = true;

    // TODO: Right now we don't utilize the separable flags
    static const int s_StageReferences[] =
    {
        nngfxToolShaderCompilerReflectionStageReference_Vertex,
        nngfxToolShaderCompilerReflectionStageReference_Hull,
        nngfxToolShaderCompilerReflectionStageReference_Domain,
        nngfxToolShaderCompilerReflectionStageReference_Geometry,
        nngfxToolShaderCompilerReflectionStageReference_Pixel,
        nngfxToolShaderCompilerReflectionStageReference_Compute
    };

    static int32_t nngfxToolShaderCompilerShaderSlot::* const s_pShaderSlotStages[] =
    {
        &nngfxToolShaderCompilerShaderSlot::vertexShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::hullShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::domainShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::geometryShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::pixelShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::computeShaderSlot
    };

    for ( int idxStage = 0; idxStage < static_cast< int >( nn::gfxTool::ShaderStage::End ); idxStage++ )
    {
        if ( pProgram[ idxStage ] != nullptr )
        {
            bool reflectionStatus = pProgram[ idxStage ]->buildReflection();

            if ( !reflectionStatus )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Failed to build glslang reflection data!\n" );
            }
        }
    }

    std::unordered_map< std::string, nngfxToolShaderCompilerConstantBuffer > constantBufferSet;
    std::unordered_map< std::string, nngfxToolShaderCompilerUnorderedAccessBuffer > unorderedAccessBufferSet;
    std::unordered_map< std::string, nngfxToolShaderCompilerShaderInput > shaderInputSet;
    std::unordered_map< std::string, nngfxToolShaderCompilerSampler > samplerSet;
    std::unordered_map< std::string, nngfxToolShaderCompilerImage > imageSet;
    std::unordered_map< std::string, nngfxToolShaderCompilerConstantBufferVariable > constantBufferVariableSet;
    // TODO: std::unordered_map< std::string, nngfxToolShaderCompilerUnorderedAccessBufferVariable > unorderedAccessBufferVariableSet;

    auto RetrieveName = [ & ] ( const char** pName, GLenum interfaceType, int indexResource, int idxStage )
    {
        switch ( interfaceType )
        {
            case GL_UNIFORM:
            {
                *pName = pProgram[ idxStage ]->getUniformName( indexResource );
                break;
            }

            case GL_BUFFER_VARIABLE:
            case GL_UNIFORM_BLOCK:
            case GL_SHADER_STORAGE_BLOCK:
            {
                *pName = pProgram[ idxStage ]->getUniformBlockName( indexResource );
                break;
            }

            case GL_PROGRAM_INPUT:
            {
                *pName = pProgram[ idxStage ]->getAttributeName( indexResource );
                break;
            }

            default: NN_UNEXPECTED_DEFAULT;
        }

        if ( nullptr == *pName )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "RerieveName failed to find resource name!" );
        }
    };

    auto RetrieveStageReferences = [ & ] ( int32_t* pDst, GLenum interfaceType, int indexResource, int idxStage )
    {
        int location = -1;
        switch ( interfaceType )
        {
        case GL_UNIFORM_BLOCK:
        {
            auto pType = pProgram[ idxStage ]->getUniformBlockTType( indexResource );
            if ( pType->getQualifier().storage == glslang::TStorageQualifier::EvqUniform )
            {
                location = pProgram[ idxStage ]->getUniformBlockBinding( indexResource );
            }
            break;
        }
        case GL_SHADER_STORAGE_BLOCK:
        {
            auto pType = pProgram[ idxStage ]->getUniformBlockTType( indexResource );
            if ( pType->getQualifier().storage == glslang::TStorageQualifier::EvqBuffer )
            {
                location = pProgram[ idxStage ]->getUniformBlockBinding( indexResource );
            }
            break;
        }
        case GL_UNIFORM:
        {
            location = pProgram[ idxStage ]->getUniformBinding( indexResource );
            break;
        }

        case GL_PROGRAM_INPUT:
        {
            location = pProgram[ idxStage ]->getAttributeLocation( indexResource );
            break;
        }
        // TODO: glslang doesn't support this yet.
        // case GL_PROGRAM_OUTPUT:
        // case GL_BUFFER_VARIABLE:
        default: break;
        }

        *pDst |= ( location >= 0 ) ? s_StageReferences[ idxStage ] : 0;
    };

    auto ForEachResource = [ & ] ( GLenum interfaceType, std::function< void( int idx, int idxStage ) > func )
    {
        for ( int idxStage = 0; idxStage < static_cast< int >( nn::gfxTool::ShaderStage::End ); idxStage++ )
        {
            GLint count = 0;

            if ( !pProgram[ idxStage ] )
            {
                continue;
            }

            switch ( interfaceType )
            {
                case GL_UNIFORM:
                {
                    count = pProgram[ idxStage ]->getNumLiveUniformVariables();
                    break;
                }

                case GL_BUFFER_VARIABLE:
                case GL_UNIFORM_BLOCK:
                case GL_SHADER_STORAGE_BLOCK:
                {
                    count = pProgram[ idxStage ]->getNumLiveUniformBlocks();
                    break;
                }

                case GL_PROGRAM_INPUT:
                {
                    count = pProgram[ idxStage ]->getNumLiveAttributes();
                    break;
                }

                default: NN_UNEXPECTED_DEFAULT;
            }

            for ( int idx = 0; idx < count; ++idx )
            {
                func( idx, idxStage );
            }
        }
    };

    auto RetrieveActiveVariableCount = [ & ] ( uint32_t* pCount, int bufferIdx, int idxStage )
    {
        int count = 0;
        int uniformCount = pProgram[ idxStage ]->getNumLiveUniformVariables();
        for ( int uniformIdx = 0; uniformIdx < uniformCount; uniformIdx++ )
        {
            if ( bufferIdx == pProgram[ idxStage ]->getUniformBlockIndex( uniformIdx ) )
            {
                count++;
            }
        }
        *pCount = count;
    };

    auto SetShaderSlot = [ & ] ( nngfxToolShaderCompilerShaderSlot* pDst, int32_t& stages, GLint value )
    {
        for ( int idxStage = 0; idxStage < static_cast< int >( nn::gfxTool::ShaderStage::End ); ++idxStage )
        {
            pDst->*s_pShaderSlotStages[ idxStage ] = ( stages & s_StageReferences[ idxStage ] ) ? value : -1;
        }
    };

    auto RetrieveConstantBuffer = [ & ] ( int idx, int idxStage )
    {
        const char* pName = nullptr;
        RetrieveName( &pName, GL_UNIFORM_BLOCK, idx, idxStage );

        std::string name( pName );
        auto iter = constantBufferSet.find( std::string( pName ) );
        if ( iter == constantBufferSet.end() )
        {
            nngfxToolShaderCompilerConstantBuffer& constantBuffer = constantBufferSet[ name ];
            constantBuffer.stages = 0;
            constantBuffer.name.length = static_cast< uint32_t >( name.size() );
            constantBuffer.name.pValue = pName;
            constantBuffer.size = pProgram[ idxStage ]->getUniformBlockSize( idx );
            RetrieveActiveVariableCount( &constantBuffer.activeVariableCount, idx, idxStage );
            RetrieveStageReferences( &constantBuffer.stages, GL_UNIFORM_BLOCK, idx, idxStage );
            SetShaderSlot( &constantBuffer.shaderSlot, constantBuffer.stages,
                pProgram[ idxStage ]->getUniformBlockBinding( idx ) );
        }
        else
        {
            nngfxToolShaderCompilerConstantBuffer& constantBuffer = iter->second;
            if ( static_cast< int >( constantBuffer.size ) != pProgram[ idxStage ]->getUniformBlockSize( idx ) )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Uniform block '%s' has different size between stages!\n", name.c_str() );
            }
            RetrieveStageReferences( &constantBuffer.stages, GL_UNIFORM_BLOCK, idx, idxStage );
            SetShaderSlot( &constantBuffer.shaderSlot, constantBuffer.stages,
                pProgram[ idxStage ]->getUniformBlockBinding( idx ) );
        }
    };

    auto RetrieveUnorderedAccessBuffer = [ & ] ( int idx, int idxStage )
    {
        // TReflection doesn't differentiate between uniform and buffer.
        if ( pProgram[ idxStage ]->getUniformBlockTType( idx )->getQualifier().storage != glslang::EvqBuffer )
        {
            return;
        }

        const char* pName = nullptr;
        RetrieveName( &pName, GL_SHADER_STORAGE_BLOCK, idx, idxStage );

        std::string name( pName );
        auto iter = unorderedAccessBufferSet.find( std::string( pName ) );
        if ( iter == unorderedAccessBufferSet.end() )
        {
            nngfxToolShaderCompilerUnorderedAccessBuffer& unorderedAccessBuffer = unorderedAccessBufferSet[ name ];
            unorderedAccessBuffer.name.length = static_cast< uint32_t >( name.size() );
            unorderedAccessBuffer.name.pValue = pName;
            unorderedAccessBuffer.size = pProgram[ idxStage ]->getUniformBlockSize( idx );
            RetrieveActiveVariableCount( &unorderedAccessBuffer.activeVariableCount, idx, idxStage );
            unorderedAccessBuffer.activeVariableCount = 1;
            RetrieveStageReferences( &unorderedAccessBuffer.stages, GL_SHADER_STORAGE_BLOCK, idx, idxStage );
            SetShaderSlot( &unorderedAccessBuffer.shaderSlot, unorderedAccessBuffer.stages,
                pProgram[ idxStage ]->getUniformBlockBinding( idx ) );
        }
        else
        {
            nngfxToolShaderCompilerUnorderedAccessBuffer& unorderedAccessBuffer = iter->second;
            if ( static_cast< int >( unorderedAccessBuffer.size ) != pProgram[ idxStage ]->getUniformBlockSize( idx ) )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "SSBO block '%s' has different size between stages!\n", name.c_str() );
            }
            RetrieveStageReferences( &unorderedAccessBuffer.stages, GL_SHADER_STORAGE_BLOCK, idx, idxStage );
            SetShaderSlot( &unorderedAccessBuffer.shaderSlot, unorderedAccessBuffer.stages,
                pProgram[ idxStage ]->getUniformBlockBinding( idx ) );
        }
    };

    auto RetrieveShaderInput = [ & ] ( int idx, int idxStage )
    {
        const char* pName = nullptr;
        RetrieveName( &pName, GL_PROGRAM_INPUT, idx, idxStage );

        std::string name( pName );
        auto iter = shaderInputSet.find( std::string( pName ) );
        if ( iter == shaderInputSet.end() )
        {
            nngfxToolShaderCompilerShaderInput& shaderInput = shaderInputSet[ name ];
            shaderInput.name.length = static_cast< uint32_t >( name.size() );
            shaderInput.name.pValue = pName;
            shaderInput.type = GlTypeToGfxType( pProgram[ idxStage ]->getAttributeType( idx ) );
            shaderInput.shaderSlot = pProgram[ idxStage ]->getAttributeLocation( idx );
            auto pTType = pProgram[ idxStage ]->getAttributeTType( idx );
            shaderInput.arrayCount = pTType->isArray() ? pTType->getArraySizes()->getImplicitSize() : 1;
            RetrieveStageReferences( &shaderInput.stages, GL_PROGRAM_INPUT, idx, idxStage );

            if ( shaderInput.shaderSlot < 0 &&
                name != "gl_InstanceIndex" && name != "gl_VertexIndex" )
            {
                NN_GFXTOOL_PRINT_ERROR(
                    "\n----%s missing location information----\nVertex input attribute %s location = -1",
                    GetStageName( static_cast< nn::gfxTool::ShaderStage >( idxStage ) ), name.c_str() );
                status = false;
            }
        }
        else
        {
            // Currently glslang only supports GL_PROGRAM_INPUT for vertex shaders
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Unexpected shader input in %s stage!\n",
                GetStageName( static_cast< nn::gfxTool::ShaderStage >( idxStage ) ) );
        }
    };

#if 0 // TODO: glslang doesn't directly track this.
    auto RetrieveShaderOutput = [ & ]( int idx, int idxStage )
    {
        const GLenum props[] = { GL_TYPE, GL_LOCATION, GL_ARRAY_SIZE };
        GLint values[ 3 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_PROGRAM_OUTPUT, idx, 3, props, 3, nullptr, values );
        nngfxToolShaderCompilerShaderOutput shaderOutput;
        shaderOutput.stages = 0;
        shaderOutput.type = GlTypeToGfxType( values[ 0 ] );
        shaderOutput.shaderSlot = values[ 1 ];
        shaderOutput.arrayCount = values[ 2 ];
        //RetrieveName( &shaderOutput.name, GL_PROGRAM_OUTPUT, idx, idxStage );
        RetrieveStageReferences( &shaderOutput.stages, GL_PROGRAM_OUTPUT, idx, idxStage );
        pOutput->AddReflection( shaderOutput );
    };
#endif

    auto RetrieveSamplerAndImageAndConstantBufferVariable = [ & ]( int idx, int idxStage )
    {
        const char* pName = nullptr;
        RetrieveName( &pName, GL_UNIFORM, idx, idxStage );

        std::string name( pName );
        auto type = GlTypeToGfxType( pProgram[ idxStage ]->getUniformType( idx ) );
        int binding = pProgram[ idxStage ]->getUniformBinding( idx );

        if ( ( nn::gfxTool::IsSamplerType( type ) || nn::gfxTool::IsImageType( type ) ) )
        {
            if ( binding < 0 )
            {
                NN_GFXTOOL_PRINT_ERROR(
                    "\n----%s missing binding information----\n%s binding = -1",
                    GetStageName( static_cast< nn::gfxTool::ShaderStage >( idxStage ) ), name.c_str() );
                status = false;
            }
            else
            {
                if ( nn::gfxTool::IsSamplerType( type ) )
                {
                    auto iter = samplerSet.find( std::string( pName ) );
                    if ( iter == samplerSet.end() )
                    {
                        nngfxToolShaderCompilerSampler& sampler = samplerSet[ name ];
                        sampler.name.length = static_cast< uint32_t >( name.size() );
                        sampler.name.pValue = pName;
                        sampler.type = type;
                        sampler.stages = 0;
                        RetrieveStageReferences( &sampler.stages, GL_UNIFORM, idx, idxStage );
                        SetShaderSlot( &sampler.shaderSlot, sampler.stages, binding );
                    }
                    else
                    {
                        nngfxToolShaderCompilerSampler& sampler = iter->second;
                        RetrieveStageReferences( &sampler.stages, GL_UNIFORM, idx, idxStage );
                        SetShaderSlot( &sampler.shaderSlot, sampler.stages, binding );
                    }
                }
                else if ( nn::gfxTool::IsImageType( type ) )
                {
                    auto iter = imageSet.find( std::string( pName ) );
                    if ( iter == imageSet.end() )
                    {
                        nngfxToolShaderCompilerImage& image = imageSet[ name ];
                        image.name.length = static_cast< uint32_t >( name.size() );
                        image.name.pValue = pName;
                        image.type = type;
                        image.stages = 0;
                        RetrieveStageReferences( &image.stages, GL_UNIFORM, idx, idxStage );
                        SetShaderSlot( &image.shaderSlot, image.stages, binding );
                    }
                    else
                    {
                        nngfxToolShaderCompilerImage& image = iter->second;
                        RetrieveStageReferences( &image.stages, GL_UNIFORM, idx, idxStage );
                        SetShaderSlot( &image.shaderSlot, image.stages, binding );
                    }
                }
            }
        }
        else
        {
            auto iter = constantBufferVariableSet.find( std::string( pName ) );
            if ( iter == constantBufferVariableSet.end() )
            {
                nngfxToolShaderCompilerConstantBufferVariable& constantBufferVariable = constantBufferVariableSet[ name ];
                constantBufferVariable.name.length = static_cast< uint32_t >( name.size() );
                constantBufferVariable.name.pValue = pName;
                constantBufferVariable.stages = 0;
                constantBufferVariable.type = type;
                constantBufferVariable.blockIndex = pProgram[ idxStage ]->getUniformBlockIndex( idx );
                constantBufferVariable.offset = pProgram[ idxStage ]->getUniformBufferOffset( idx );
                constantBufferVariable.arrayCount = pProgram[ idxStage ]->getUniformArraySize( idx );
                RetrieveStageReferences( &constantBufferVariable.stages, GL_UNIFORM, idx, idxStage );
            }
            else
            {
                nngfxToolShaderCompilerConstantBufferVariable& constantBufferVariable = iter->second;
                if ( constantBufferVariable.offset != pProgram[ idxStage ]->getUniformBufferOffset( idx )
                    || static_cast< int >( constantBufferVariable.arrayCount ) != pProgram[ idxStage ]->getUniformArraySize( idx ) )
                {
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Unexpected shader input in %s stage!\n",
                        GetStageName( static_cast< nn::gfxTool::ShaderStage >( idxStage ) ) );
                }
                RetrieveStageReferences( &constantBufferVariable.stages, GL_UNIFORM, idx, idxStage );
            }
        }
    };

#if 0 // TODO: glslang doesn't directly track this.
    auto RetrieveUnorderedAccessBufferVariable = [ & ]( int idx, int idxStage )
    {
        const GLenum props[] = { GL_TYPE, GL_BLOCK_INDEX, GL_OFFSET, GL_ARRAY_SIZE };
        GLint values[ 4 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_BUFFER_VARIABLE, idx, 4, props, 4, nullptr, values );
        nngfxToolShaderCompilerUnorderedAccessBufferVariable unorderedAccessBufferVariable;
        unorderedAccessBufferVariable.stages = 0;
        unorderedAccessBufferVariable.type = GlTypeToGfxType( values[ 0 ] );
        unorderedAccessBufferVariable.blockIndex = values[ 1 ];
        unorderedAccessBufferVariable.offset = values[ 2 ];
        unorderedAccessBufferVariable.arrayCount = values[ 3 ];
        //RetrieveName( &unorderedAccessBufferVariable.name, GL_BUFFER_VARIABLE, idx, idxStage );
        RetrieveStageReferences( &unorderedAccessBufferVariable.stages, GL_BUFFER_VARIABLE, idx, idxStage );
        pOutput->AddReflection( unorderedAccessBufferVariable );
    };
#endif

    ForEachResource( GL_UNIFORM_BLOCK, RetrieveConstantBuffer );
    ForEachResource( GL_SHADER_STORAGE_BLOCK, RetrieveUnorderedAccessBuffer );
    ForEachResource( GL_PROGRAM_INPUT, RetrieveShaderInput );
    // TODO: glslang doesn't directly track this
    // ForEachResource( GL_PROGRAM_OUTPUT, RetrieveShaderOutput );
    ForEachResource( GL_UNIFORM, RetrieveSamplerAndImageAndConstantBufferVariable );
    // TODO: glslang doesn't directly track this
    // ForEachResource( GL_BUFFER_VARIABLE, RetrieveUnorderedAccessBufferVariable );

    for ( auto& it : constantBufferSet )
    {
        auto& buffer = it.second;
        pOutput->AddReflection( buffer );
    }

    for ( auto& it : unorderedAccessBufferSet )
    {
        auto& buffer = it.second;
        pOutput->AddReflection( buffer );
    }

    for ( auto& it : shaderInputSet )
    {
        auto& buffer = it.second;
        pOutput->AddReflection( buffer );
    }

    for ( auto& it : samplerSet )
    {
        auto& buffer = it.second;
        pOutput->AddReflection( buffer );
    }

    for ( auto& it : imageSet )
    {
        auto& buffer = it.second;
        pOutput->AddReflection( buffer );
    }

    for ( auto& it : constantBufferVariableSet )
    {
        auto& buffer = it.second;
        pOutput->AddReflection( buffer );
    }

    if ( pProgram[ static_cast< int >( nn::gfxTool::ShaderStage::Compute ) ] != nullptr )
    {
        glslang::TIntermediate* pIntermediate =
            pProgram[ static_cast< int >( nn::gfxTool::ShaderStage::Compute ) ]->getIntermediate( EShLangCompute );
        if ( pIntermediate == nullptr )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Work group dimensions missing!\n" );
        }
        pOutput->GetOutput()->computeWorkGroupSizeX = pIntermediate->getLocalSize(0);
        pOutput->GetOutput()->computeWorkGroupSizeY = pIntermediate->getLocalSize(1);
        pOutput->GetOutput()->computeWorkGroupSizeZ = pIntermediate->getLocalSize(2);
    }

    return status;
} // NOLINT

}

namespace nn {
namespace gfxTool {

typedef CompilerVariation< static_cast<
    nngfxToolShaderCompilerLowLevelApiType >( nngfxToolShaderCompilerLowLevelApiType_Vk ),
    nngfxToolShaderCompilerCodeType_Ir > Target;

void Compiler< Target >::PreCompile( CompileOutput*,
    const ShaderCompilerContext*, const nngfxToolShaderCompilerCompileArg* )
{
    if ( !GlobalWindow::GetInstance().IsInitialized() )
    {
        GlobalWindow::GetInstance().Initialize();
    }

    // Initializes thread safety features
    ShInitialize();

    // Work around for glslang issue #342 (spirvbin_t::remap is not thread safe)
    spv::Parameterize();
}

void Compiler< Target >::CompileGroup( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
    const nngfxToolShaderCompilerCompileArg* pArg, int idxGroup )
{
    auto idxVariation = idxGroup;

    auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    auto pProgramOutput = pOutput->GetVariationOutput( idxVariation )->GetIntermediateLanguageOutput();

    // TProgram doesn't support cross stage error checking and reflection information
    // is collected per TProgram. So for now, force separate shaders
    pProgramOutput->GetInfo()->flags.SetBit( nn::gfx::ShaderInfoData::Flag_SeparationEnable,
        true );
    pProgramOutput->GetInfo()->codeType = StaticCastAuto( nn::gfx::ShaderCodeType_Ir );
    pProgramOutput->GetInfo()->sourceFormat = 0;

    std::shared_ptr< OptionOutputProgramCommon > pOptionOutputProgramCommon(
        new OptionOutputProgramCommon() );
    pOptionOutputProgramCommon->Initialize( pArg );
    pProgramOutput->AddOptionOutput(
    nngfxToolShaderCompilerOptionOutputType_ProgramCommon, pOptionOutputProgramCommon );

    StageIntermediateResult stageResults[ static_cast< size_t >( ShaderStage::End ) ];
    std::shared_ptr< OptionOutputReflection > pReflection;
    if ( pCommonOption->IsReflectionEnabled() )
    {
        pReflection.reset( new OptionOutputReflection() );
    }

    BoostPreprocessor preprocessors[ static_cast< int >( ShaderStage::End ) ];
    for ( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
    {
        auto& stageResult = stageResults[ idxStage ];
        auto stage = static_cast< ShaderStage >( idxStage );
        if ( GetStageSource( pArg, stage ).pValue == nullptr )
        {
            stageResult.pShader = nullptr;
            continue;
        }

        stageResult.pInfoLog.reset( new nn::gfxTool::Custom< std::string >::Type() );
        stageResult.pPreprocessor = &preprocessors[ idxStage ];

        PrepareStage( &stageResult, pContext, pArg, stage, idxVariation );
    }

    if ( pCommonOption->IsPreprocessEnabled() )
    {
        nn::gfxTool::BoostPreprocessor::ResolveUniformRegisterBlock(
            static_cast< int >( ShaderStage::End ),
            &preprocessors[0], pCommonOption->GetUniformRegisterBlockName().c_str());

        for ( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
        {
            auto& stageResult = stageResults[ idxStage ];
            auto stage = static_cast< ShaderStage >( idxStage );
            if ( GetStageSource( pArg, stage ).pValue == nullptr )
            {
                stageResult.pShader = nullptr;
                continue;
            }
            stageResult.preprocessedSource = preprocessors[ idxStage ].GetResult().str();

            // Set the preprocessed sources for compilation
            stageResult.sources.push_back( stageResult.preprocessedSource.c_str() );
            stageResult.lengths.push_back( nn::gfxTool::NumericCastAuto( stageResult.preprocessedSource.length() ) );
        }
    }

    TDefaultIoResolver resolver;
    auto& uniformRegisterBlockName = pCommonOption->GetUniformRegisterBlockName();
    if ( !uniformRegisterBlockName.empty() )
    {
        resolver.addFixedBinding( uniformRegisterBlockName, 0, 0 );
    }
    glslang::TProgram* pPrograms[ static_cast< int >( ShaderStage::End ) ]{};
    for ( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
    {
        auto& stageResult = stageResults[ idxStage ];
        auto stage = static_cast< ShaderStage >( idxStage );

        if ( GetStageSource( pArg, stage ).pValue == nullptr )
        {
            stageResult.pShader = nullptr;
            continue;
        }

        pPrograms[ idxStage ] = new glslang::TProgram;
        auto& program = *pPrograms[ idxStage ];

        CompileStage( &stageResult, pContext, stage, idxVariation );

        program.addShader( stageResult.pShader );

        EShMessages messages = (EShMessages)( EShMsgSpvRules | EShMsgVulkanRules );
        bool linkStatus = program.link( messages );

        auto infoLog = program.getInfoLog();
        auto infoLogSize = infoLog != nullptr ? strlen( infoLog ) : 0;
        if ( infoLogSize > 0 )
        {
            stageResult.pInfoLog->append( &infoLog[ 0 ], infoLogSize );
        }

        auto infoDebugLog = program.getInfoDebugLog();
        auto infoDebugLogSize = infoLog != nullptr ? strlen( infoDebugLog ) : 0;
        if ( infoDebugLogSize > 0 )
        {
            stageResult.pInfoLog->append( &infoDebugLog[ 0 ], infoDebugLogSize );
        }

        if ( !linkStatus )
        {
            // TODO
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "%s",
                CreateErrorSource( stage, &stageResult, pCommonOption->GetCodePage() ).c_str() );
        }

        RetrieveOutput( pProgramOutput, &program, &resolver, stage, pCommonOption->IsCompressionEnabled(), pCommonOption->GetDebugInfoLevel() );

        if ( pCommonOption->IsDumpEnabled() )
        {
            DumpShaderSource( pProgramOutput, &stageResult, stage );
        }
    }

    if( pCommonOption->IsReflectionEnabled() )
    {
        if ( !RetrieveReflection( pReflection.get(), pPrograms, pCommonOption->IsSeparationEnabled() ) )
        {
            for ( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
            {
                nn::gfxTool::Custom< std::string >::Type source;
                for ( int idxSource = 0, sourceLength = static_cast< int >( stageResults[ idxStage ].sources.size() );
                    idxSource < sourceLength; ++idxSource )
                {
                    source.append( stageResults[ idxStage ].sources[ idxSource ], stageResults[ idxStage ].lengths[ idxSource ] );
                }
                auto result = nn::gfxTool::ConvertEncoding( nn::gfxTool::AddLineNumber( nn::util::string_view(
                    source.c_str(), source.length() ) ), pCommonOption->GetCodePage(), 0 );
                NN_GFXTOOL_PRINT_ERROR( "\n[Variation: %d, Stage: %s]\n%s\n",
                    idxVariation, GetStageName( static_cast< ShaderStage >( idxStage ) ), result.c_str() );
            }

            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "RetrieveReflection failed!" );
        }
    }

    for ( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        pOptionOutputProgramCommon->GetOptionOutputStageCommon(
            static_cast< ShaderStage >( idxStage ) )->SetInfoLog( stageResults[ idxStage ].pInfoLog );
    }

    if ( pCommonOption->IsReflectionEnabled() )
    {
        pOptionOutputProgramCommon->SetReflection( pReflection );
    }
}

void Compiler< Target >::PostCompile( CompileOutput* pOutput,
    const ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg* pArg )
{
    // no-op
    NN_UNUSED( pOutput );
    NN_UNUSED( pContext );
    NN_UNUSED( pArg );
}

}
}
