﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <memory>

#include <gfxTool_Common.h>

namespace nn {
namespace gfxTool {

class ShaderSourceManager;
class CompileOptionManager;
class VariationManager;
class CompilerManager;

class ShaderCompilerContext
{
public:
    typedef ShaderCompilerContext SelfType;
    typedef nngfxToolHandle HandleType;

    enum class State
    {
        Created = 0x01,
        Compiled = 0x02
    };

    ShaderCompilerContext();

    ~ShaderCompilerContext();

    static const int MajorVersion = nngfxToolShaderCompilerVersion_Major;
    static const int MinorVersion = nngfxToolShaderCompilerVersion_Minor;
    static const int MicroVersion = nngfxToolShaderCompilerVersion_Micro;

    nngfxToolResultCode Reset();

    nngfxToolResultCode Compile( nngfxToolShaderCompilerCompileOutput** ppOutput,
        const nngfxToolShaderCompilerCompileArg* pArg );

    const ShaderSourceManager* GetShaderSourceManager() const
    {
        return m_pShaderSourceManager.get();
    }

    const CompileOptionManager* GetCompileOptionManager() const
    {
        return m_pCompileOptionManager.get();
    }

    const VariationManager* GetVariationManager() const
    {
        return m_pVariationManager.get();
    }

    static SelfType* Get( HandleType handle )
    {
        return reinterpret_cast< SelfType* >( handle );
    }

    static HandleType Create();

    static nngfxToolResultCode Delete( HandleType handle );

private:
    bool CheckState( State state )
    {
        return ( m_State & static_cast< int >( state ) ) != 0;
    }

    void AddState( State state )
    {
        m_State |= static_cast< int >( state );
    }

    void SetupSources( const nngfxToolShaderCompilerCompileArg* pArg );
    void SetupOptions( const nngfxToolShaderCompilerCompileArg* pArg );
    void SetupVariations( const nngfxToolShaderCompilerCompileArg* pArg );
    void CompileImpl( nngfxToolShaderCompilerCompileOutput** ppOutput,
        const nngfxToolShaderCompilerCompileArg* pArg );

    void ValidateCompileArg( const nngfxToolShaderCompilerCompileArg* pArg );

private:
    int m_State;

    Custom< std::unique_ptr< ShaderSourceManager > >::Type m_pShaderSourceManager;

    Custom< std::unique_ptr< CompileOptionManager > >::Type m_pCompileOptionManager;

    Custom< std::unique_ptr< VariationManager > >::Type m_pVariationManager;

    Custom< std::unique_ptr< CompilerManager > >::Type m_pCompilerManager;

    Custom< std::unique_ptr< nngfxToolShaderCompilerCompileOutput > >::Type m_pCompileOutput;
};

}
}
