﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <vector>
#include <string>
#include <utility>
#include <sstream>
#include <hlslcc.h>
//#include <internal_includes/HLSLccToolkit.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi.h>

namespace nn { namespace gfxTool {
class HlslShaderCompilerReflection : public HLSLccReflection
{
public:
    HlslShaderCompilerReflection();
    virtual ~HlslShaderCompilerReflection() override;

    // Called on errors or diagnostic messages
    void OnDiagnostics(const std::string &error, int line, bool isError) override;

    virtual void OnInputBinding(const std::string &name, int bindIndex) override;

    virtual bool OnConstantBuffer(const std::string &name, size_t bufferSize, size_t memberCount) override;

    virtual bool OnConstant(const std::string &name, int bindIndex, SHADER_VARIABLE_TYPE cType, int rows, int cols, bool isMatrix, int arraySize) override;

    virtual void OnConstantBufferBinding(const std::string &name, int bindIndex) override;

    virtual void OnTextureBinding(const std::string &name, int bindIndex, int samplerIndex, HLSLCC_TEX_DIMENSION dim, bool isUAV) override;

    virtual void OnBufferBinding(const std::string &name, int bindIndex, bool isUAV) override;

    virtual void OnThreadGroupSize(unsigned int xSize, unsigned int ySize, unsigned int zSize) override;

    typedef struct InputBinding
    {
        std::string name;
        int bindIndex;
    } InputBinding;
    typedef struct ConstantBuffer
    {
        std::string name;
        size_t bufferSize;
        size_t memberCount;
    } ConstantBuffer;
    typedef struct Constant
    {
        std::string name;
        int bindIndex;
        SHADER_VARIABLE_TYPE variableType;
        int rows;
        int cols;
        bool isMatrix;
        int arraySize;
    } Constant;
    typedef struct ConstantBufferBinding
    {
        std::string name;
        int bindIndex;
    } ConstantBufferBinding;
    typedef struct TextureBinding
    {
        std::string name;
        int bindIndex;
        int samplerIndex;
        HLSLCC_TEX_DIMENSION dim;
        bool isUAV;
    } TextureBinding;
    typedef struct BufferBinding
    {
        std::string name;
        int bindIndex;
        bool isUav;
    } BufferBinding;

    typedef std::vector<InputBinding>			InputBindings;
    typedef std::vector<ConstantBuffer>			ConstantBuffers;
    typedef std::vector<Constant>				Constants;
    typedef std::vector<ConstantBufferBinding>	ConstantBufferBindings;
    typedef std::vector<TextureBinding>				TextureBindings;
    typedef std::vector<BufferBinding>			BufferBindings;

    const InputBindings& GetInputBindings() const
    {
        return m_InputBindings;
    }

    const ConstantBuffers& GetConstantBuffers() const
    {
        return m_ConstantBuffers;
    }

    const Constants& GetConstants() const
    {
        return m_Constants;
    }

    const ConstantBufferBindings GetConstantBufferBindings() const
    {
        return m_ConstantBufferBindings;
    }

    const TextureBindings GetTextureBindings() const
    {
        return m_TextureBindings;
    }

    const BufferBindings GetBufferBindings() const
    {
        return m_BufferBindings;
    }

    void GetThreadGroupSize(unsigned int& xSize, unsigned int& ySize, unsigned int& zSize) const
    {
        xSize = m_ThreadGroupSizeX;
        ySize = m_ThreadGroupSizeY;
        zSize = m_ThreadGroupSizeZ;
    }

    void DumpReflection( std::string& buff ) const
    {
        std::stringstream reflection;
        reflection << "InputBindings\n";
        for( auto inputBinding : m_InputBindings )
        {
            reflection << inputBinding.name << "\n";
            reflection << "bindIdx:" << inputBinding.bindIndex << "\n";
        }
        reflection << "==============================\n";
        reflection << "ConstantBuffer\n";
        for( auto constantBuffer : m_ConstantBuffers )
        {
            reflection << constantBuffer.name << "\n";
            reflection << "bufferSize:" << constantBuffer.bufferSize;
            reflection << "memberCount:" << constantBuffer.memberCount;
        }
        reflection << "==============================\n";
        reflection << "Constant\n";
        for( auto constant : m_Constants )
        {
        }
        reflection << "==============================\n";
        reflection << "ConstantBufferBinding\n";
        for( auto constantBufferBinding : m_ConstantBufferBindings )
        {
        }
        reflection << "==============================\n";
        reflection << "TextureBinding\n";
        for( auto textureBinding : m_TextureBindings )
        {
        }
        reflection << "==============================\n";
        reflection << "BufferBinding\n";
        for( auto bufferBinding : m_BufferBindings )
        {
        }
        reflection << "==============================\n";

        buff = reflection.str();
    }

private:
    InputBindings	m_InputBindings;
    ConstantBuffers m_ConstantBuffers;
    Constants		m_Constants;
    ConstantBufferBindings m_ConstantBufferBindings;
    TextureBindings m_TextureBindings;
    BufferBindings	m_BufferBindings;
    int m_ThreadGroupSizeX;
    int m_ThreadGroupSizeY;
    int m_ThreadGroupSizeZ;
    std::string m_DumpReflection;

};
}}
