﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Allocator.h>

#include <nn/util/util_IntrusiveList.h>
#include "DebugFontGlyphTreeMap.h"

namespace nn
{
namespace fontll
{
    class ScalableFontEngine;
    typedef int32_t Fixed32;
} // namespace nn::fontll
} // namespace nn


//---------------------------------------------------------------------------
//! @brief スケーラブルフォントのテクスチャキャッシュを表すクラスです。
//---------------------------------------------------------------------------
class DebugFontTextureCache
{
public:
    //! @brief ラインの種類を表す列挙型です。
    //!
    enum LineHeight
    {
        LineHeight_16 = 16,     //!< 高さが16のラインです。
        LineHeight_18 = 18,     //!< 高さが18のラインです。
        LineHeight_20 = 20,     //!< 高さが20のラインです。
        LineHeight_22 = 22,     //!< 高さが22のラインです。
        LineHeight_32 = 32,     //!< 高さが32のラインです。
        LineHeight_64 = 64      //!< 高さが64のラインです。
    };

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 実質的な初期化はInitializeで行われます。ここでは、
    //! メンバ変数の初期化のみ行っています。
    //!
    DebugFontTextureCache();

    //! @brief デストラクタです。
    //!
    //! @details
    //! このクラスでは何も行いません。メモリやWindowsでのGLのオブジェクトの
    //! 破棄を行うにはFinalizeを呼び出す必要がありますので、ご注意ください。
    //!
    virtual ~DebugFontTextureCache();

    //! @brief 初期化のパラメータを指定するための構造体です。
    //!
    struct InitializeArg {

        //! テクスチャキャッシュの横幅です。必ず2の累乗としてください。
        //! デフォルトでは1024です。
        uint32_t textureCacheWidth;
        //! テクスチャキャッシュの縦幅です。
        //! デフォルトでは1024です。
        uint32_t textureCacheHeight;
        //! メモリを確保するためのアロケータです。メモリの確保はInitializeの中のみで行われます。
        //! 確保したメモリはFinalizeで解放されます。
        //! デフォルト値はありません。必ず指定してください。
        nn::AlignedAllocateFunctionWithUserData allocateFunction;
        //! メモリを確保するためのアロケータに引数として渡されるユーザーデータです。
        void* pUserDataForAllocateFunction;
        //! フォントデータへのポインタを渡します。(ポインタが示す先はttf(otf)ファイルもしくはbfttfファイルのバイナリデータです。)
        //! デフォルト値はありません。必ず指定してください。
        const void* pFontData;

        //! スケーラブルフォントエンジンに渡すワークメモリのサイズです。大きいほどエンジン内で文字を
        //! キャッシュしてくれます。
        //! デフォルトでは512KBです。
        size_t workMemorySize;
        //! プロットと並行して処理ができるように、プロット用とは別に用意しているスケーラブルフォントエンジン
        //! のインスタンスのためのワークメモリのサイズです。通常はデフォルトのサイズで問題ありませんが、
        //! フォントによってはサイズが足りなくなる場合がありますので、その場合は増やしてください。
        //! サイズが足りなくなった場合、アサートに失敗し、"error: 201"というメッセージが表示されます。
        //! デフォルトでは50KBです。
        size_t noPlotWorkMemorySize;
        //! テクスチャキャッシュ内でグリフを管理するためのノードの最大数です。
        //! 管理するグリフの数が多いほど(テクスチャキャッシュが大きいほど)大きい数を指定する必要があります。
        //! デフォルトでは1024個です。
        uint32_t glyphNodeCountMax;
        //! ttc ファイルを使用する際に、レンダリングに使用するフォントのインデックスを指定します。
        //! その他の形式のフォントファイルを使用する場合は 0 を指定します。
        //! デフォルトは 0 です。
        uint32_t fontIndex;

        //! このインスタンスで扱うフォントのサイズです。
        //! デフォルトでは40です。
        int fontSize;

        //! @brief デフォルト値を設定します。
        void SetDefault();
    };

    void Initialize(const InitializeArg& arg);
    void Finalize(nn::FreeFunctionWithUserData freeFunction, void* pUserDataForFreeFunction);

    void RegisterGlyphs(const uint16_t *str);
    bool RegisterGlyph(uint16_t code);
    void UpdateTextureCache();
    void CompleteTextureCache();
    bool CreateNewLine(LineHeight lineHeight)
    {
        return (CreateNewLineImpl(lineHeight) != NULL);
    }

    uint32_t GetTextureCacheWidth() const
    {
        return mTextureCacheWidth;
    }

    uint32_t GetTextureCacheHeight() const
    {
        return mTextureCacheHeight;
    }

    //! @brief  フォントの中に指定した文字コードの文字が含まれているかを返します。
    //!
    //! @details
    //! このメソッドが返す値は、文字がプロットされているかどうかには関係しません。
    //! 初期化時に与えたフォントデータ内に文字が含まれているかのみに依存します。
    //!
    //! @param[in] code         幅を求めたい文字
    //! @return 含まれるか否か
    //!
    bool IsGlyphExistInFont(uint16_t code);

    struct FontMetrics {
        float ascent_ratio;     // アセントの割合。ベースラインとも呼びます。
        float height_ratio;     // 高さの割合。これからアセントを引いた値がディセントになります。
        float boundingBoxAscentRatio; // 実際の文字が収まる矩形から算出したアセントの割合です。
        float boundingBoxHeightRatio; // 実際の文字が収まる矩形から算出した高さの割合です。
    };

    const FontMetrics& GetFontMetrics();
    GlyphTreeMap&			 GetGlyphTreeMap()			{ return mGlyphTreeMap; }

    uint32_t GetFontSize() const { return m_FontSize; }
    int             GetWidth() const { return m_FontSize; }
    int             GetAscent() const { return m_FontAscent; }
    int             GetLineFeed() const { return m_FontLineFeed; }
    int             GetBaselinePos() const { return m_FontBaselinePos; }

private:

    static const int FontNameLengthMax = 50;
    static const int WorkMemorySizeDefault = 512 * 1024;
    static const int WorkMemorySizeNoPlotDefault = 50 * 1024;
    static const int GlyphNodeCountMaxDefault = 1024;
    static const int TextureCacheLineCountMax = 256;
    static const int GlyphPadding = 2;

    struct LineInfo {
        GlyphLineList list;
        uint16_t y;
        uint16_t currentX;
        uint8_t kind;
        uint8_t no;
        uint8_t dummy[2];
    };

    LineInfo* CreateNewLineImpl(LineHeight lineHeight);
    GlyphNode* FindAndReserveEraseGlyph(uint8_t lineKind, uint16_t glyphWidth);
    void AssertFsError(const char* api_name);
    void AssertFsErrorNoPlot(const char* api_name);
    uint32_t EraseNotInFontGlyphs();

    //----------------------------------------------------------

    nn::fontll::ScalableFontEngine* mFontEngine;
    nn::fontll::ScalableFontEngine* mFontEngineNoPlot;
    char mFontNameBuf[FontNameLengthMax];
public:
    uint8_t* mTextureBitMap;
private:
    uint32_t mTextureCacheWidth;
    uint32_t mTextureCacheHeight;
    GlyphTreeMap mGlyphTreeMap;
    GlyphList mNeedPlotGlyphList;
    GlyphList mNeedEraseGlyphList;
    GlyphList mNotInFontGlyphList;
    LineInfo mLineInfos[TextureCacheLineCountMax];
    uint32_t mLineCurrentPos;
    FontMetrics mFontMetrics;

private:

    uint32_t m_FontSize;
    int m_FontHeight;
    int m_FontAscent;
    int m_FontBaselinePos;
    int m_FontLineFeed;
};
