﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_SdkAssert.h>

#include <nn/nn_Allocator.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_IntrusiveList.h>


namespace detail
{
    /**
    *  固定サイズのメモリエリアを管理する仕組み（フリーリスト）です。
    */
    class FreeList
    {
    public:
        FreeList() : mFree(NULL), mWork(NULL) {}

        /**
        *  フリーリストを初期化します。
        *
        *  @param[in] work         ワーク領域。elem_size*countの大きさとしてください。４バイトアライメントである必要があります。
        *  @param[in] elem_size    一つの要素のサイズ。４の倍数としてください。
        *  @param[in] count          要素の最大数。
        */
        void Initialize(void* work, int elem_size, int count)
        {
            NN_SDK_ASSERT(work);
            NN_SDK_ASSERT(elem_size > 0 && elem_size % 4 == 0);
            NN_SDK_ASSERT(count > 0);

            void** work_as_ptr = static_cast<void**>(work);
            int size_div4 = elem_size / 4;

            mFree = work;
            /*
            *  フリーリストに繋がれている間は、領域をリンクリストとして使う。
            */
            for (int i = 0; i < count - 1; i++) {
                work_as_ptr[size_div4*i] = &work_as_ptr[size_div4*(i + 1)];
            }
            work_as_ptr[size_div4*(count - 1)] = NULL;

            mWork = work;
        }

        /**
        * フリーリストから要素を得ます。
        *
        * @return  新しい要素へのポインタ。領域のデータは不定です。
        *          フリーリストが空のときにはNULLが返ります。
        */
        void* Get()
        {
            void* ret = mFree;

            if (ret == NULL) {
                return NULL;
            }
            else {
                mFree = *static_cast<void**>(ret);  // 次の要素のポインタを辿ってmFreeに入れる
                return ret;
            }
        }

        /**
        *  不要な要素をフリーリストへ戻します。
        *
        *  @param[in] elem     要素へのポインタ。
        */
        void Put(void* elem)
        {
            // リストに戻されたのだから、領域内にポインタを書き込んでもOK
            //
            // 戻されたものの先頭に、今のmFreeのアドレスを記憶しておく
            // そして、mFreeを、今返されたelemにする。
            *static_cast<void**>(elem) = mFree;
            mFree = elem;
        }

        /**
        *  フリーリストのワークエリアへのポインタを得ます。
        *
        *  @return ワークエリアへのポインタ
        */
        void* GetWork() const
        {
            return mWork;
        }

        /**
        *  内部状態を生成時の状態に戻します。
        *
        *  initで与えたバッファを解放したときなど、内部に保持しているポインタが
        *  使用できなくなったとき等にご使用ください。
        *
        *  このメソッドを呼んだ後再び使うには、initを呼び出す必要があります。
        */
        void Cleanup()
        {
            mFree = NULL;
            mWork = NULL;
        }

        /**
        *  フリーリストが空になっているか否かを返します。
        *
        *  @return 空になっていて要素を確保できないなら true、まだ要素があるなら false を返します。
        */
        bool IsEmpty() const
        {
            return (mFree == NULL);
        }

    private:
        void* mFree;        // 未使用領域のリンクリストの先頭
        void* mWork;        // ワークエリアへのポインタ
    };
} // namespace detail


//---------------------------------------------------------------------------
//! @brief テクスチャキャッシュ内のグリフを表すクラスです。
//!
//! @details
//! ライブラリユーザが直接使用するクラスではないため、詳しいDoxyコメントは
//! 付けていません。
//---------------------------------------------------------------------------
class GlyphNode
{
friend class GlyphTreeMap;
public:
    GlyphNode(uint16_t code, uint16_t size)
     : mLeft(NULL)
     , mRight(NULL)
     , mColor(Red)
     , mLineKind(0)
     , mLineNo(0)
     , mCachePosX(0)
     , mCachePosY(0)
     , mGlyphWidth(0)
     , mGlyphHeight(0)
     , mAdvanceX(0)
     , mLeftOffset(0)
    {
        mKey.detail.code = code;
        mKey.detail.size = size;

        mFlag.Clear();
    }

    uint16_t GetCode() const { return mKey.detail.code; }
    uint16_t GetSize() const { return mKey.detail.size; }

    void SetRequested(bool b) { mFlag.SetMaskedBits(FlagMask_Requested, b ? FlagMask_Requested : 0); }
    bool IsRequested() const { return mFlag.IsAnyBitOn(FlagMask_Requested); }

    void SetPlotting(bool b) { mFlag.SetMaskedBits(FlagMask_Plotting, b ? FlagMask_Plotting : 0); }
    bool IsPlotting() const { return mFlag.IsAnyBitOn(FlagMask_Plotting); }

    void SetErased(bool b) { mFlag.SetMaskedBits(FlagMask_Erased, b ? FlagMask_Erased : 0); }
    bool IsErased() const { return mFlag.IsAnyBitOn(FlagMask_Erased); }

    void SetNotInFont(bool b) { mFlag.SetMaskedBits(FlagMask_NotInFont, b ? FlagMask_NotInFont : 0); }
    bool IsNotInFont() const { return mFlag.IsAnyBitOn(FlagMask_NotInFont); }

    bool IsPlottingOrNotInFont() const { return mFlag.IsAnyBitOn(FlagMask_Plotting | FlagMask_NotInFont); }
    bool IsRequestedOrKeeped() const { return mFlag.IsAnyBitOn(FlagMask_Requested | FlagMask_Kept); }

    uint16_t GetCachePosX() const { return mCachePosX; }
    uint16_t GetCachePosY() const { return mCachePosY; }
    uint16_t GetGlyphWidth() const { return mGlyphWidth; }
    uint16_t GetGlyphHeight() const { return mGlyphHeight; }
    uint16_t GetAdvanceX() const { return mAdvanceX; }
    int16_t GetLeftOffset() const { return mLeftOffset; }

    void SetCachePosX(uint16_t posX) { mCachePosX = posX; }
    void SetCachePosY(uint16_t posY) { mCachePosY = posY; }
    void SetGlyphWidth(uint16_t width) { mGlyphWidth = width; }
    void SetGlyphHeight(uint16_t height) { mGlyphHeight = height; }
    void SetAdvanceX(uint16_t advance) { mAdvanceX = advance; }
    void SetLeftOffset(int16_t offset) { mLeftOffset = offset; }

    uint8_t GetLineNo() const { return mLineNo; }
    void SetLineNo(uint8_t no) { mLineNo = no; }
    uint8_t GetLineKind() const { return mLineKind; }
    void SetLineKind(uint8_t kind) { mLineKind = kind; }

    bool IsErasable() const { return (mFlag.storage == 0); }

    static uint8_t CalculateLineKind(uint16_t size);

    nn::util::IntrusiveListNode m_Link;
    nn::util::IntrusiveListNode m_LineLink;

private:

    enum FlagBit {
        FlagBit_Requested,
        FlagBit_Plotting,
        FlagBit_Used,
        FlagBit_Erased,
        FlagBit_Kept,
        FlagBit_SystemReserved,
        FlagBit_NotInFont
    };

    enum FlagMask {
        FlagMask_Requested = 1 << FlagBit_Requested,
        FlagMask_Plotting = 1 << FlagBit_Plotting,
        FlagMask_Used = 1 << FlagBit_Used,
        FlagMask_Erased = 1 << FlagBit_Erased,
        FlagMask_Kept = 1 << FlagBit_Kept,
        FlagMask_SystemReserved = 1 << FlagBit_SystemReserved,
        FlagMask_NotInFont = 1 << FlagBit_NotInFont
    };

    static const uint8_t Red = 1;
    static const uint8_t Black = 0;

    union KeyType {
        uint32_t raw;
        struct {
            uint16_t code;
            uint16_t size;
        } detail;
    };

    NN_STATIC_ASSERT(sizeof(KeyType) == 4);

    //----------------------------------------------------------

    GlyphNode* mLeft;
    GlyphNode* mRight;
    KeyType mKey;
    nn::util::BitPack8 mFlag;
    uint8_t mColor;
    uint8_t mLineKind;
    uint8_t mLineNo;
    uint16_t mCachePosX;
    uint16_t mCachePosY;
    uint16_t mGlyphWidth;
    uint16_t mGlyphHeight;
    uint16_t mAdvanceX;
    int16_t mLeftOffset;
};

typedef nn::util::IntrusiveList<GlyphNode, nn::util::IntrusiveListMemberNodeTraits<GlyphNode, &GlyphNode::m_Link> > GlyphList;
typedef nn::util::IntrusiveList<GlyphNode, nn::util::IntrusiveListMemberNodeTraits<GlyphNode, &GlyphNode::m_LineLink> > GlyphLineList;

//---------------------------------------------------------------------------
//! @brief テクスチャキャッシュ内のグリフを管理するためのTreeMapです。
//!
//! @details
//! 左傾赤黒木という平衡二分木のアルゴリズムを利用してます。
//!
//! ライブラリユーザが直接使用するクラスではないため、詳しいDoxyコメントは
//! 付けていません。
//---------------------------------------------------------------------------
class GlyphTreeMap
{
public:
    GlyphTreeMap();

    void Initialize(nn::AlignedAllocateFunctionWithUserData allocateFunction, void* pUserDataForAllocateFunction, uint32_t nodeCount);

    void Finalize(nn::FreeFunctionWithUserData freeFunction, void* pUserDataForFreeFunction);

    //! @brief 引数で指定した文字コードとサイズに等しいグリフを検索して返します。
    //!
    //! @param[in] code 検索するグリフの文字コードです。
    //! @param[in] size 検索するグリフのサイズです。
    //! @param[in] face 追加するグリフのフォント字形です。
    //! @return 見つかったグリフ。見つからなかった場合はNULL
    //!
    GlyphNode* Find(uint16_t code, uint16_t size) const;

    //! @brief 引数で与えた文字コードとサイズのグリフをツリーマップに追加します。
    //!
    //! @param[in] code 追加するグリフの文字コードです。
    //! @param[in] size 追加するグリフのサイズです。
    //! @param[in] face 追加するグリフのフォント字形です。
    //! @return 追加したグリフ。グリフのノード数が最大値を超えている場合は内部でアサートします。製品版ではNULLが返されます。
    //!
    GlyphNode* Insert(uint16_t code, uint16_t size);

    //! @brief 引数で指定した文字コードとサイズに等しいグリフを検索し、削除します。
    //!
    //! @details
    //! 等しい文字コードとサイズを持っているグリフが存在しない場合、実行時例外で
    //! 停止してしまいますので、
    //! 等しい文字コードとサイズを持っているグリフが含まれていることを確認した上でご使用ください。
    //!
    //! @param[in] code 削除するグリフの文字コードです。
    //! @param[in] size 削除するグリフのサイズです。
    //! @param[in] face 追加するグリフのフォント字形です。
    //!
    void Erase(uint16_t code, uint16_t size);

    //! @brief ScalableFontのCompleteTextureCacheでグリフのフラグを更新する処理を行います。
    //!
    void UpdateFlagsForCompleteTextureCache()
    {
        if (mRoot)
        {
            UpdateFlagsForCompleteTextureCacheRecursive(mRoot);
        }
    }

    typedef void(*GlyphFunc)(GlyphNode* pGlyphNode, void *pUserData);

    void GlyphFunction(GlyphFunc func, void *pUserData)	{ GlyphFunctionRecursive(mRoot, func, pUserData); }
    void GlyphFunctionRecursive(GlyphNode* node, GlyphFunc func, void *pUserData);

private:
    GlyphNode* mRoot;
    detail::FreeList mFreeList;
    uint32_t mNodeCountMax;

    GlyphNode* Find(GlyphNode* node, uint32_t key_raw) const;
    GlyphNode* Insert(GlyphNode* h, GlyphNode* node);
    GlyphNode* Erase(GlyphNode* h, uint32_t key_raw);
    static inline GlyphNode* RotateLeft(GlyphNode* h);
    static inline GlyphNode* RotateRight(GlyphNode* h);
    static inline void FlipColors(GlyphNode* h);
    static inline bool IsRed(GlyphNode* h);
    static inline GlyphNode* FixUp(GlyphNode* h);
    static inline GlyphNode* MoveRedLeft(GlyphNode* h);
    static inline GlyphNode* MoveRedRight(GlyphNode* h);
    static inline GlyphNode* EraseMin(GlyphNode* h);
    static inline GlyphNode* GetMin(GlyphNode* h);

    static void UpdateFlagsForCompleteTextureCacheRecursive(GlyphNode* node);
};

inline GlyphNode* GlyphTreeMap::RotateLeft(GlyphNode* h)
{
    GlyphNode* x = h->mRight;
    h->mRight = x->mLeft;
    x->mLeft = h;
    x->mColor = h->mColor;
    h->mColor = GlyphNode::Red;
    return x;
}

inline GlyphNode* GlyphTreeMap::RotateRight(GlyphNode* h)
{
    GlyphNode* x = h->mLeft;
    h->mLeft = x->mRight;
    x->mRight = h;
    x->mColor = h->mColor;
    h->mColor = GlyphNode::Red;
    return x;
}

inline void GlyphTreeMap::FlipColors(GlyphNode* h)
{
    h->mColor = ! h->mColor;
    h->mLeft->mColor = ! h->mLeft->mColor;
    h->mRight->mColor = ! h->mRight->mColor;
}

inline bool GlyphTreeMap::IsRed(GlyphNode* h)
{
    if (h) {
        return h->mColor == GlyphNode::Red;
    } else {
        return false;
    }
}

inline GlyphNode* GlyphTreeMap::FixUp(GlyphNode* h)
{
    if (IsRed(h->mRight)) {
        h = RotateLeft(h);
    }
    if(IsRed(h->mLeft) && IsRed(h->mLeft->mLeft)) {
        h = RotateRight(h);
    }
    if(IsRed(h->mLeft) && IsRed(h->mRight)) {
        FlipColors(h);
    }
    return h;
}

inline GlyphNode* GlyphTreeMap::MoveRedLeft(GlyphNode* h)
{
    FlipColors(h);
    if (IsRed(h->mRight->mLeft)) {
        h->mRight = RotateRight(h->mRight);
        h = RotateLeft(h);
        FlipColors(h);
    }
    return h;
}

inline GlyphNode* GlyphTreeMap::MoveRedRight(GlyphNode* h)
{
    FlipColors(h);
    if (IsRed(h->mLeft->mLeft)) {
        h = RotateRight(h);
        FlipColors(h);
    }
    return h;
}

inline GlyphNode* GlyphTreeMap::EraseMin(GlyphNode* h)
{
    if (h->mLeft == NULL) {
        return NULL;
    }
    if ( ! IsRed(h->mLeft) && ! IsRed(h->mLeft->mLeft)) {
        h = MoveRedLeft(h);
    }
    h->mLeft = EraseMin(h->mLeft);

    return FixUp(h);
}

inline GlyphNode* GlyphTreeMap::GetMin(GlyphNode* h)
{
    while (h->mLeft) {
        h = h->mLeft;
    }
    return h;
}
