﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include <regex>
#include <chrono>
#include <autoTestAssistTool/LogReaderTypes.h>
#include <autoTestAssistTool/LogReaderLibrary.h>
#include "Common\autoTestAssistTool_StringConverter.h"
#include "LogAccessor.h"

namespace
{
    detail::LogAccessor g_LogAccessor;
}

LogReaderResult StartLogStoring(const SerialNumberString& serialNumber, int logMaxLineCount)
{
    return g_LogAccessor.Start(logMaxLineCount, serialNumber.value);
}

LogReaderResult GetLogLineSize(size_t* pOutSize)
{
    std::string log;
    LogReaderResult result;
    result = g_LogAccessor.GetCurrent(&log, 0);
    if (result == LogReaderResult_Success)
    {
        *pOutSize = log.length() != 0 ? log.length() + 1 : 0;
        return LogReaderResult_Success;
    }
    return LogReaderResult_NotFound;
}

LogReaderResult GetLogLine(char* buffer, size_t bufferSize)
{
    std::string log;
    LogReaderResult result = g_LogAccessor.GetCurrent(&log, 0);
    if (result != LogReaderResult_Success)
    {
        return LogReaderResult_NotFound;
    }

    if (bufferSize > 0)
    {
        size_t copySize = min(bufferSize - 1, log.length());
        strncpy_s(buffer, bufferSize, log.c_str(), copySize);
        buffer[copySize] = '\0';
    }

    //DEBUG_LOG("[%s] %s\n", g_LogAccessor.GetCurrentTimeString().c_str(), log.c_str());
    return LogReaderResult_Success;
}

LogReaderResult WaitForNextLine(int timeoutMs)
{
    return g_LogAccessor.WaitForNextLine(timeoutMs);
}

LogReaderResult WaitUntilMatched(const char* regexString, int timeoutMs)
{
    std::string str(regexString);
    std::wregex regexStr;
    regexStr.imbue(std::locale(""));
    regexStr.assign(autoTestAssistTool::util::cp_to_wide(str, CP_UTF8));

    std::chrono::nanoseconds restTime = std::chrono::milliseconds(timeoutMs);
    while (restTime.count() >= 0)
    {
        auto before = std::chrono::system_clock::now();

        std::string log;
        LogReaderResult result = g_LogAccessor.GetCurrent(&log, timeoutMs);
        if (result == LogReaderResult_Success && std::regex_match(autoTestAssistTool::util::cp_to_wide(log, CP_UTF8), regexStr))
        {
            return LogReaderResult_Success;
        }
        auto after = std::chrono::system_clock::now();
        restTime -= std::chrono::duration_cast<std::chrono::nanoseconds>(after - before);

        before = std::chrono::system_clock::now();
        result = MoveToNextLine();
        if (result == LogReaderResult_CannotMoveLine)
        {
            auto time = std::chrono::duration_cast<std::chrono::milliseconds>(restTime).count();
            if (WaitForNextLine(static_cast<int>(time)) == LogReaderResult_Timeout)
            {
                break;
            }
        }
        after = std::chrono::system_clock::now();
        restTime -= std::chrono::duration_cast<std::chrono::nanoseconds>(after - before);
    }

    return LogReaderResult_Timeout;
}

LogReaderResult SearchFormerLog(const char* regexString)
{
    std::string str(regexString);
    std::wregex regexStr;
    regexStr.imbue(std::locale(""));
    regexStr.assign(autoTestAssistTool::util::cp_to_wide(str, CP_UTF8));

    while (true)
    {
        std::string log;
        LogReaderResult result = g_LogAccessor.GetCurrent(&log, 0);
        if (result != LogReaderResult_Success)
        {
            return LogReaderResult_NotFound;
        }

        if (std::regex_match(autoTestAssistTool::util::cp_to_wide(log, CP_UTF8), regexStr))
        {
            break;
        }

        if (g_LogAccessor.IsFirst())
        {
            return LogReaderResult_NotFound;
        }
        MoveToPreviousLine();
    }
    return LogReaderResult_Success;
}

void MoveToNewestLine()
{
    g_LogAccessor.MoveToNewestLine();
}

LogReaderResult MoveToNextLine()
{
    return g_LogAccessor.MoveToNextLine();
}

LogReaderResult MoveToPreviousLine()
{
    return g_LogAccessor.MoveToPreviousLine();
}

void MoveToFormerLineByTime(int timeMs)
{
    g_LogAccessor.MoveToFormerLineByTime(timeMs);
}

void StopLogStoring()
{
    g_LogAccessor.Stop();
}

void PrintDebugInfo()
{
    DEBUG_LOG("---- Print Debug info ----\n");
    DEBUG_LOG("Log num : %d\n", g_LogAccessor.GetLogNum());
    g_LogAccessor.PrintFullLog();
    DEBUG_LOG("--------------------------\n");
}
