﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "IOPortSender.h"
#include "PythonCodeExporter.h"
#include <chrono>


namespace
{
    int             g_FilePortTimeout   = 3000;
    int             g_FindPortSpan      = 50;
}

IoPortSender& IoPortSender::GetInstance()
{
    static IoPortSender instance;
    return instance;
}

IoPortSender::IoPortSender() : m_TouchStateValue({ 0 }), m_NextFingerId( 0 )
{
    HidShellLibraryWrapper::GetInstance();
    m_XpadTaskList.clear();
    m_DebugPadTaskList.clear();
    m_VirtualPadTaskList.clear();

    // 入力記録状態設定
    m_IsRecordindHid = false;
}

IoPortSender::~IoPortSender()
{
}

HidInputerResult IoPortSender::BeginHidInputRecording()
{
    m_IsRecordindHid = true;
    return HidInputerResult::HidInputerResult_Success;
}
HidInputerResult IoPortSender::EndHidInputRecording()
{
    if (m_IsRecordindHid)
    {
        int32_t result;
        HidShellHandle hidShellHandle;
        HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();
        // HidShell 接続
        result = hidShellLibrary.m_AquireHidShellExclusiveRight(reinterpret_cast<void**>(&hidShellHandle));
        assert(static_cast<HidShellResult>(result) == HidShellResult::Success);

        m_IsRecordindHid = false;

        // HidShell 接続解除
        result = hidShellLibrary.m_ReleaseHidShellExclusiveRight(reinterpret_cast<void*>(hidShellHandle));
        assert(static_cast<HidShellResult>(result) == HidShellResult::Success);
    }
    return HidInputerResult::HidInputerResult_Success;
}
HidInputerResult IoPortSender::ClearHidRecord()
{
    // 記録中の記録クリアを禁止する
    if (m_IsRecordindHid)
    {
        ERRMSG("IoPortSender::ClearHidRecords: Clear hid record was executed during hid recording.")
        return HidInputerResult::HidInputerResult_Unexpected;
    }
    try
    {
        m_IoPortRecorder.ClearHidRecord();
    }
    catch (...)
    {
        ERRMSG("IoPortSender::ClearHidRecords: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return HidInputerResult::HidInputerResult_Unexpected;
    }
    return HidInputerResult::HidInputerResult_Success;
}

HidInputerResult IoPortSender::ExportHidInputRecords(const char* exportFileName)
{
    // 記録中のエクスポートを禁止する
    if (m_IsRecordindHid)
    {
        ERRMSG("IoPortSender::ExportHidInputRecords: Export hid record was executed during hid recording.")
        return HidInputerResult::HidInputerResult_Unexpected;
    }

    try
    {
        // Python コードエクスポート
        PythonCodeExporter exporter;
        return exporter.ExportHidInputRecords(exportFileName, m_IoPortRecorder.GetRecords());
    }
    catch (...)
    {
        ERRMSG("IoPortSender::ExportHidInputRecords: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return HidInputerResult::HidInputerResult_Unexpected;
    }
}

void IoPortSender::UpdateXpadStatus( HidShellHandle hidShellHandle )
{
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();
    // Xpad
    HidShellBasicXpadState  xPadStateBuffer[HidShellBasicXpadCountMax];
    bool                    isXpadStateAcquired[HidShellBasicXpadCountMax] = { false };
    // タスク実行後のステータス作成
    for (auto it = m_XpadTaskList.begin(); it != m_XpadTaskList.end(); it++)
    {
        // 不正なコントローラー ID
        if ((*it).id < 0 || (*it).id >= HidShellBasicXpadCountMax)
        {
            continue;
        }
        // コントローラー情報取得
        if (isXpadStateAcquired[(*it).id] == false)
        {
            int32_t result;
            result = hidShellLibrary.m_GetHidShellBasicXpadState(reinterpret_cast<void*>(hidShellHandle), &(xPadStateBuffer[(*it).id]), (*it).id, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
            isXpadStateAcquired[(*it).id] = true;

            switch (static_cast<HidShellResult>(result))
            {
                case HidShellResult::Success:
                    break;
                case HidShellResult::StateNotSet:
                    memset(&(xPadStateBuffer[(*it).id]), 0, sizeof(HidShellBasicXpadState));
                    // TIPS: 開発機の再起動後は、入力状態がクリアされ StateNotSet が呼び出されます。
                    // AutoTestAssistTool 上で接続済みの xpad 操作を継続させるために、power, battery にデフォルト値を代入しておきます。
                    xPadStateBuffer[(*it).id].powerState    = HidShellXpadPowerState::OnBattery;
                    xPadStateBuffer[(*it).id].batteryLevel  = HidShellXpadBatteryLevel::Full;
                    break;
                case HidShellResult::PortNotFound:
                    // TIPS: 開発機のスリープ時に実行しようとしたときにこの値が戻ります。
                    // Initialize 時に、接続できるポートを調べて接続しているため、その後の PortNotFound はスリープか開発機の切断だと判断し、
                    // 例外を投げずに、タスクのクリアだけを行います。
                    m_XpadTaskList.clear();
                    return;
                default:
                    throw std::exception((std::string("Get basic xpad state failed in HidShellLibrary. error code: ") + std::to_string(result)).c_str());
            }
        }

        HidShellBasicXpadState &xPadInfo = xPadStateBuffer[(*it).id];
        switch ((*it).taskType)
        {
            case HidTaskType::HidTaskType_SetXpadConnection:
                xPadInfo.powerState = static_cast<HidShellXpadPowerState>((*it).value);
                xPadInfo.batteryLevel = HidShellXpadBatteryLevel::Full;
                break;
            case HidTaskType::HidTaskType_SetXpadButtonState:
                xPadInfo.buttons = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetXpadPressButton:
                xPadInfo.buttons |= (*it).value;
                break;
            case HidTaskType::HidTaskType_SetXpadReleaseButton:
                xPadInfo.buttons &= ~((*it).value);
                break;
            case HidTaskType::HidTaskType_SetXpadStickLX:
                xPadInfo.stickL.x = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetXpadStickLY:
                xPadInfo.stickL.y = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetXpadStickRX:
                xPadInfo.stickR.x = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetXpadStickRY:
                xPadInfo.stickR.y = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetCaptureButton:
                HidShellCaptureButtonState captureButtonState;
                captureButtonState.buttons = (*it).value;
                hidShellLibrary.m_SetHidShellCaptureButtonState(reinterpret_cast<void*>(hidShellHandle), &captureButtonState, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
                break;
            case HidTaskType::HidTaskType_SetHomeButton:
                HidShellHomeButtonState homeButtonState;
                homeButtonState.buttons = (*it).value;
                hidShellLibrary.m_SetHidShellHomeButtonState(reinterpret_cast<void*>(hidShellHandle), &homeButtonState, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
                break;
            default:
                ERRMSG("Invalid task was attempted to be executed at IoPortSender::UpdateXpadStatus.");
                break;
        }
    }
    // ステータス同期
    for (int i = 0; i < HidShellBasicXpadCountMax; i++)
    {
        if (isXpadStateAcquired[i] == false)
        {
            continue;
        }
        hidShellLibrary.m_SetHidShellBasicXpadState( reinterpret_cast<void*>(hidShellHandle), &(xPadStateBuffer[i]), i, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
    }
    // タスクリストリセット
    m_XpadTaskList.clear();
}


void IoPortSender::UpdateDebugPadState( HidShellHandle hidShellHandle )
{
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();

    HidShellDebugPadState   debugPadStateBuffer;
    bool                    isDebugPadStateAcquired = false;

    // タスク実行後のステータス作成
    for (auto it = m_DebugPadTaskList.begin(); it != m_DebugPadTaskList.end(); it++)
    {
        if (isDebugPadStateAcquired == false)
        {
            int32_t result;
            result = hidShellLibrary.m_GetHidShellDebugPadState(reinterpret_cast<void*>(hidShellHandle), &debugPadStateBuffer, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));

            switch (static_cast<HidShellResult>(result))
            {
                case HidShellResult::Success:
                    break;
                case HidShellResult::StateNotSet:
                    memset(&debugPadStateBuffer, 0, sizeof(HidShellDebugPadState));
                    // TIPS: 開発機の再起動後は、入力状態がクリアされ StateNotSet が呼び出されます。
                    // AutoTestAssistTool 上で接続済みの debugpad 操作を継続させるために、attributes にデフォルト値を代入しておきます。
                    debugPadStateBuffer.attributes = 1;
                    break;
                case HidShellResult::PortNotFound:
                    // TIPS: 開発機のスリープ時に実行しようとしたときにこの値が戻ります。
                    // Initialize 時に、接続できるポートを調べて接続しているため、その後の PortNotFound はスリープか開発機の切断だと判断し、
                    // 例外を投げずに、タスクのクリアだけを行います。
                    m_DebugPadTaskList.clear();
                    return;
                default:
                    std::exception((std::string("Get debug pad state failed. error code: ") + std::to_string(result)).c_str());
            }

            isDebugPadStateAcquired = true;
        }
        switch ((*it).taskType)
        {
            case HidTaskType::HidTaskType_SetDebugPadConnection:
                debugPadStateBuffer.attributes = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetDebugPadButtonState:
                debugPadStateBuffer.buttons = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetDebugPadPressButton:
                debugPadStateBuffer.buttons |= (*it).value;
                break;
            case HidTaskType::HidTaskType_SetDebugPadReleaseButton:
                debugPadStateBuffer.buttons &= ~((*it).value);
                break;
            case HidTaskType::HidTaskType_SetDebugPadStickLX:
                debugPadStateBuffer.stickL.x = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetDebugPadStickLY:
                debugPadStateBuffer.stickL.y = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetDebugPadStickRX:
                debugPadStateBuffer.stickR.x = (*it).value;
                break;
            case HidTaskType::HidTaskType_SetDebugPadStickRY:
                debugPadStateBuffer.stickR.y = (*it).value;
                break;
            default:
                ERRMSG("Invalid task was attempted to be executed at IoPortSender::UpdateDebugPadState.");
                break;
        }
    }
    m_DebugPadTaskList.clear();

    if (isDebugPadStateAcquired)
    {
        hidShellLibrary.m_SetHidShellDebugPadState(reinterpret_cast<void*>(hidShellHandle), &debugPadStateBuffer, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
    }
}

void IoPortSender::UpdateVirtualPadStatus(HidShellHandle hidShellHandle)
{
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();
    HidShellAbstractedPadState  virtualPadStateBuffer[HidShellAbstractedPadCountMax];
    bool                        isVirtualpadStateAcquired[HidShellAbstractedPadCountMax] = { false };

    // タスク実行後のステータス作成
    for (auto it = m_VirtualPadTaskList.begin(); it != m_VirtualPadTaskList.end(); it++)
    {
        // 不正なコントローラー ID
        if ((*it).id < 0 || (*it).id >= HidShellBasicXpadCountMax)
        {
            continue;
        }
        // コントローラー情報取得
        if (isVirtualpadStateAcquired[(*it).id] == false)
        {
            int32_t result;
            result = hidShellLibrary.m_GetHidShellAbstractedPadState(
                reinterpret_cast<void*>(hidShellHandle), &(virtualPadStateBuffer[(*it).id]),
                (*it).id, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));

            isVirtualpadStateAcquired[(*it).id] = true;

            switch (static_cast<HidShellResult>(result))
            {
                case HidShellResult::Success:
                    break;
                case HidShellResult::StateNotSet:
                    memset(&(virtualPadStateBuffer[(*it).id]), 0, sizeof(HidShellAbstractedPadState));
                    // TIPS: 開発機の再起動後は、入力状態がクリアされ StateNotSet が呼び出されます。
                    // AutoTestAssistTool 上で接続済みの xpad 操作を継続させるために、power, battery にデフォルト値を代入しておきます。
                    virtualPadStateBuffer[(*it).id].powerInfo.isCharging = true;
                    virtualPadStateBuffer[(*it).id].powerInfo.isPowered = true;
                    virtualPadStateBuffer[(*it).id].powerInfo.batteryLevel  = HidShellAbstractedPadBatteryLevel::High;
                    break;
                case HidShellResult::PortNotFound:
                    // TIPS: 開発機のスリープ時に実行しようとしたときにこの値が戻ります。
                    // Initialize 時に、接続できるポートを調べて接続しているため、その後の PortNotFound はスリープか開発機の切断だと判断し、
                    // 例外を投げずに、タスクのクリアだけを行います。
                    m_XpadTaskList.clear();
                    return;
                default:
                    throw std::exception((std::string("Get virtual pad state failed in HidShellLibrary. error code: ") + std::to_string(result)).c_str());
            }
        }

        HidShellAbstractedPadState &virtualPadInfo = virtualPadStateBuffer[(*it).id];
        switch ((*it).taskType)
        {
            case HidTaskType_SetVirtualPadDeviceType:
                virtualPadInfo.deviceType = (*it).value;
                break;
            case HidTaskType_SetVirtualPadInterfaceType:
                virtualPadInfo.interfaceType = (*it).value;
                break;
            case HidTaskType_SetVirtualPadMainColor:
                virtualPadInfo.color.main[0] = ((*it).value & 0x00FF0000) >> 16;  // Red
                virtualPadInfo.color.main[1] = ((*it).value & 0x0000FF00) >> 8;   // Green
                virtualPadInfo.color.main[2] = ((*it).value & 0x000000FF);        // Blue
                virtualPadInfo.color.main[3] = 255;     // Alpha(255 固定）
                break;
            case HidTaskType_SetVirtualPadSubColor:
                virtualPadInfo.color.sub[0] = ((*it).value & 0x00FF0000) >> 16;  // Red
                virtualPadInfo.color.sub[1] = ((*it).value & 0x0000FF00) >> 8;   // Green
                virtualPadInfo.color.sub[2] = ((*it).value & 0x000000FF);        // Blue
                virtualPadInfo.color.sub[3] = 255;     // Alpha(255 固定）
                break;
            case HidTaskType_SetVirtualPadConnection:
                if ((*it).value)
                {
                    virtualPadInfo.attribute |= AbstractedPadAttributeMask::AbstractedPadAttribute_IsConnected;
                }
                else
                {
                    virtualPadInfo.attribute &= (~AbstractedPadAttributeMask::AbstractedPadAttribute_IsConnected);
                }
                break;
            case HidTaskType_SetVirtualPadButtonState:
                virtualPadInfo.buttons  = (*it).value;
                break;
            case HidTaskType_SetVirtualPadPressButton:
                virtualPadInfo.buttons  |= (*it).value;
                break;
            case HidTaskType_SetVirtualPadReleaseButton:
                virtualPadInfo.buttons  &= ~((*it).value);
                break;
            case HidTaskType_SetVirtualPadStickLX:
                virtualPadInfo.stickL.x = (*it).value;
                break;
            case HidTaskType_SetVirtualPadStickLY:
                virtualPadInfo.stickL.y = (*it).value;
                break;
            case HidTaskType_SetVirtualPadStickRX:
                virtualPadInfo.stickR.x = (*it).value;
                break;
            case HidTaskType_SetVirtualPadStickRY:
                virtualPadInfo.stickR.y = (*it).value;
                break;
            default:
                ERRMSG("Invalid task was attempted to be executed at IoPortSender::UpdateVirtualPadStatus.");
                break;
        }
    }
    // ステータス同期
    for (int i = 0; i < HidShellAbstractedPadCountMax; i++)
    {
        if (isVirtualpadStateAcquired[i] == false)
        {
            continue;
        }
        hidShellLibrary.m_SetHidShellAbstractedPadState( reinterpret_cast<void*>(hidShellHandle), &(virtualPadStateBuffer[i]), i, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
    }
    // タスクリストリセット
    m_VirtualPadTaskList.clear();

}

// タッチパネル状態更新
void IoPortSender::UpdateTouchScreenStatus(HidShellHandle hidShellHandle)
{
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();
    HidShellTouchScreenState touchList = { 0 };

    int currentTouchId = 0;
    for (int i = 0; i < HidShellTouchStateCountMax; i++)
    {
        if (m_TouchStateValue.touches[i].attributes == 0)
        {
            continue;
        }

        touchList.touches[currentTouchId] = m_TouchStateValue.touches[i];
        currentTouchId++;
    }

    touchList.count = currentTouchId;

    // タッチされている箇所が無いときに fingerId をリセットする
    if (currentTouchId == 0)
    {
        m_NextFingerId = 0;
    }

    hidShellLibrary.m_SetHidShellTouchScreenState(reinterpret_cast<void*>(hidShellHandle), &touchList, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
}

// Xpad タスク登録
void IoPortSender::SetXpadTask(int32_t controllerId, int32_t value, HidTaskType taskType)
{
    HidTask task = { controllerId, value, taskType };
    m_XpadTaskList.push_back(task);
}

// Debug Pad タスク登録
void IoPortSender::SetDebugPadTask( int32_t value, HidTaskType taskType)
{
    HidTask task = { 0, value, taskType };
    m_DebugPadTaskList.push_back(task);
}

// Abstracted Pad タスク登録
void IoPortSender::SetVirtualPadTask(int32_t controllerId, int32_t value, HidTaskType taskType)
{
    HidTask task = {controllerId, value, taskType};
    m_VirtualPadTaskList.push_back(task);
}

// Xpad > 接続状態
void IoPortSender::SetXpadConnection(int32_t controllerId, int32_t value)
{
    // 接続解除前に状態リセット
    if (value == 0)
    {
        SetXpadDefault(controllerId);
    }

    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, value, HidTaskType::HidTaskType_SetXpadConnection);
}

// Xpad > リセット
void IoPortSender::SetXpadDefault(int32_t controllerId)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, ~(static_cast<int32_t>(0)), HidTaskType::HidTaskType_SetXpadReleaseButton);
    SetXpadTask(controllerId, 0, HidTaskType::HidTaskType_SetXpadStickLX);
    SetXpadTask(controllerId, 0, HidTaskType::HidTaskType_SetXpadStickLY);
    SetXpadTask(controllerId, 0, HidTaskType::HidTaskType_SetXpadStickRX);
    SetXpadTask(controllerId, 0, HidTaskType::HidTaskType_SetXpadStickRY);
}

void IoPortSender::SetXPadState(int32_t controllerId, ControllerButtonValueType buttons,
    const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int32_t stickCount)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, buttons, HidTaskType::HidTaskType_SetXpadButtonState);
    for (int i = 0; i < stickCount; i++)
    {
        switch (stickList[i])
        {
            case Stick::Stick_L:
                SetXpadTask(controllerId, stickXList[i], HidTaskType::HidTaskType_SetXpadStickLX);
                SetXpadTask(controllerId, stickYList[i], HidTaskType::HidTaskType_SetXpadStickLY);
                break;
            case Stick::Stick_R:
                SetXpadTask(controllerId, stickXList[i], HidTaskType::HidTaskType_SetXpadStickRX);
                SetXpadTask(controllerId, stickYList[i], HidTaskType::HidTaskType_SetXpadStickRY);
                break;
            default:
                break;
        }
    }
}

// Xpad > ボタン操作
void IoPortSender::SetXpadPressButton(int32_t controllerId, int32_t value)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, value, HidTaskType::HidTaskType_SetXpadPressButton);
}
void IoPortSender::SetXpadReleaseButton(int32_t controllerId, int32_t value)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, value, HidTaskType::HidTaskType_SetXpadReleaseButton);
}
// Xpad > スティック操作
void IoPortSender::SetXpadStickL(int32_t controllerId, int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, x, HidTaskType::HidTaskType_SetXpadStickLX);
    SetXpadTask(controllerId, y, HidTaskType::HidTaskType_SetXpadStickLY);
}
void IoPortSender::SetXpadStickR(int32_t controllerId, int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, x, HidTaskType::HidTaskType_SetXpadStickRX);
    SetXpadTask(controllerId, y, HidTaskType::HidTaskType_SetXpadStickRY);
}

// キャプチャボタン
void IoPortSender::SetCaptureButton(int32_t controllerId, int32_t value)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, value, HidTaskType::HidTaskType_SetCaptureButton);
}
// ホームボタン
void IoPortSender::SetHomeButton(int32_t controllerId, int32_t value)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetXpadTask(controllerId, value, HidTaskType::HidTaskType_SetHomeButton);
}


// Debug Pad > 接続
void IoPortSender::SetDebugPadConnection(int32_t value)
{
    // 接続解除前に状態リセット
    if (value == 0)
    {
        SetDebugPadDefault();
    }

    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetDebugPadTask(value, HidTaskType::HidTaskType_SetDebugPadConnection);
}

// Debug Pad > リセット
void IoPortSender::SetDebugPadDefault()
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetDebugPadTask(~(static_cast<int32_t>(0)), HidTaskType::HidTaskType_SetDebugPadReleaseButton);
    SetDebugPadTask(0, HidTaskType::HidTaskType_SetDebugPadStickLX);
    SetDebugPadTask(0, HidTaskType::HidTaskType_SetDebugPadStickLY);
    SetDebugPadTask(0, HidTaskType::HidTaskType_SetDebugPadStickRX);
    SetDebugPadTask(0, HidTaskType::HidTaskType_SetDebugPadStickRY);
}

void IoPortSender::SetDebugPadState(ControllerButtonValueType buttons,
    const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int32_t stickCount)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetDebugPadTask(buttons, HidTaskType::HidTaskType_SetDebugPadButtonState);
    for (int i = 0; i < stickCount; i++)
    {
        switch (stickList[i])
        {
            case Stick::Stick_L:
                SetDebugPadTask(stickXList[i], HidTaskType::HidTaskType_SetDebugPadStickLX);
                SetDebugPadTask(stickYList[i], HidTaskType::HidTaskType_SetDebugPadStickLY);
                break;
            case Stick::Stick_R:
                SetDebugPadTask(stickXList[i], HidTaskType::HidTaskType_SetDebugPadStickRX);
                SetDebugPadTask(stickYList[i], HidTaskType::HidTaskType_SetDebugPadStickRY);
                break;
            default:
                break;
        }
    }
}

// Debug Pad > ボタン操作
void IoPortSender::SetDebugPadPressButton(int32_t value)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetDebugPadTask(value, HidTaskType::HidTaskType_SetDebugPadPressButton);
}

void IoPortSender::SetDebugPadReleaseButton(int32_t value)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetDebugPadTask(value, HidTaskType::HidTaskType_SetDebugPadReleaseButton);
}

// Debug Pad > スティック操作
void IoPortSender::SetDebugPadStickL(int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetDebugPadTask(x, HidTaskType::HidTaskType_SetDebugPadStickLX);
    SetDebugPadTask(y, HidTaskType::HidTaskType_SetDebugPadStickLY);
}
void IoPortSender::SetDebugPadStickR(int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    SetDebugPadTask(x, HidTaskType::HidTaskType_SetDebugPadStickRX);
    SetDebugPadTask(y, HidTaskType::HidTaskType_SetDebugPadStickRY);
}

// タッチスクリーン > 初期化
void IoPortSender::ResetTouchState()
{
    std::lock_guard<std::mutex> lock(m_TouchJobMutex);
    memset( &m_TouchStateValue, 0, sizeof(HidShellTouchScreenState));
}

void IoPortSender::SetTouchState(int32_t touchIdList[], int32_t fingerIdList[], int32_t posXList[], int32_t posYList[], int32_t touchCount)
{
    std::lock_guard<std::mutex> lock(m_TouchJobMutex);

    memset( &m_TouchStateValue, 0, sizeof(HidShellTouchScreenState));
    m_TouchStateValue.count = touchCount;
    for (int i = 0; i < touchCount; i++)
    {
        m_TouchStateValue.touches[touchIdList[i]].attributes    = 1;
        m_TouchStateValue.touches[touchIdList[i]].fingerId      = fingerIdList[i];
        m_TouchStateValue.touches[touchIdList[i]].x             = posXList[i];
        m_TouchStateValue.touches[touchIdList[i]].y             = posYList[i];

        if (fingerIdList[i] >= m_NextFingerId)
        {
            m_NextFingerId = fingerIdList[i] + 1;
        }
    }
}

// タッチスクリーン > タッチ
void IoPortSender::SetTouchPress(int32_t touchId, int32_t x, int32_t y, int32_t fingerId)
{
    // fingerId チェック
    for (int i = 0; i < MaxTouchCount; i++)
    {
        if (m_TouchStateValue.touches[i].attributes != 1)
        {
            continue;
        }

        if (m_TouchStateValue.touches[i].fingerId == fingerId)
        {
            throw InvalidFingerIdException(fingerId);
        }
    }

    std::lock_guard<std::mutex> lock(m_TouchJobMutex);
    m_TouchStateValue.touches[touchId].attributes = 1;
    m_TouchStateValue.touches[touchId].fingerId = fingerId;
    m_TouchStateValue.touches[touchId].x = x;
    m_TouchStateValue.touches[touchId].y = y;

    if (fingerId >= m_NextFingerId)
    {
        m_NextFingerId = (fingerId + 1);
    }
}

void IoPortSender::SetTouchPress(int32_t touchId, int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_TouchJobMutex);
    m_TouchStateValue.touches[touchId].attributes = 1;
    m_TouchStateValue.touches[touchId].fingerId = m_NextFingerId;
    m_TouchStateValue.touches[touchId].x = x;
    m_TouchStateValue.touches[touchId].y = y;

    m_NextFingerId++;
}

// タッチスクリーン > タッチ解除
void IoPortSender::IoPortSender::SetTouchRelease(int32_t touchId)
{
    std::lock_guard<std::mutex> lock(m_TouchJobMutex);
    m_TouchStateValue.touches[touchId].attributes = 0;
}
// タッチスクリーン > 座標更新
void IoPortSender::SetTouchMove(int32_t touchId, int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_TouchJobMutex);
    m_TouchStateValue.touches[touchId].x = x;
    m_TouchStateValue.touches[touchId].y = y;
}

// VirtualPad
void IoPortSender::SetVirtualPadState(int32_t controllerId, ControllerButtonValueType buttons,
    const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int32_t stickCount)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, buttons, HidTaskType::HidTaskType_SetVirtualPadButtonState);
    for (int i = 0; i < stickCount; i++)
    {
        switch (stickList[i])
        {
            case Stick::Stick_L:
                SetVirtualPadTask(controllerId, stickXList[i], HidTaskType::HidTaskType_SetVirtualPadStickLX);
                SetVirtualPadTask(controllerId, stickYList[i], HidTaskType::HidTaskType_SetVirtualPadStickLY);
                break;
            case Stick::Stick_R:
                SetVirtualPadTask(controllerId, stickXList[i], HidTaskType::HidTaskType_SetVirtualPadStickRX);
                SetVirtualPadTask(controllerId, stickYList[i], HidTaskType::HidTaskType_SetVirtualPadStickRY);
                break;
            default:
                break;
        }
    }
}

// VirtualPad > メイン色
void IoPortSender::SetVirtualPadMainColor(int32_t controllerId, int32_t mainColor)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, mainColor, HidTaskType::HidTaskType_SetVirtualPadMainColor);
}

// VirtualPad > サブ色
void IoPortSender::SetVirtualPadSubColor(int32_t controllerId, int32_t subColor)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, subColor, HidTaskType::HidTaskType_SetVirtualPadSubColor);
}

// VirtualPad > デバイスタイプ
void IoPortSender::SetVirtualPadDeviceType(int32_t controllerId, int32_t deviceType)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, deviceType, HidTaskType::HidTaskType_SetVirtualPadDeviceType);
}

// VirtualPad > インターフェースタイプ
void IoPortSender::SetVirtualPadInterfaceType(int32_t controllerId, int32_t interfaceType)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, interfaceType, HidTaskType::HidTaskType_SetVirtualPadInterfaceType);
}

// VirtualPad > 接続状態
void IoPortSender::SetVirtualPadConnection(int32_t controllerId, int32_t value)
{
    if (value==0)
    {
        SetVirtualPadDefault(controllerId);
    }
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, value, HidTaskType::HidTaskType_SetVirtualPadConnection);
}

// VirtualPad > リセット
void IoPortSender::SetVirtualPadDefault(int32_t controllerId)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, ~(static_cast<int32_t>(0)), HidTaskType::HidTaskType_SetVirtualPadReleaseButton);
    SetVirtualPadTask(controllerId, 0, HidTaskType::HidTaskType_SetVirtualPadStickLX);
    SetVirtualPadTask(controllerId, 0, HidTaskType::HidTaskType_SetVirtualPadStickLY);
    SetVirtualPadTask(controllerId, 0, HidTaskType::HidTaskType_SetVirtualPadStickRX);
    SetVirtualPadTask(controllerId, 0, HidTaskType::HidTaskType_SetVirtualPadStickRY);
}

// VirtualPad > ボタン操作
void IoPortSender::SetVirtualPadPressButton(int32_t controllerId, int32_t value)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, value, HidTaskType::HidTaskType_SetVirtualPadPressButton);
}

void IoPortSender::SetVirtualPadReleaseButton(int32_t controllerId, int32_t value)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, value, HidTaskType::HidTaskType_SetVirtualPadReleaseButton);
}

// VirtualPad > スティック操作
void IoPortSender::SetVirtualPadStickL(int32_t controllerId, int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, x, HidTaskType::HidTaskType_SetVirtualPadStickLX);
    SetVirtualPadTask(controllerId, y, HidTaskType::HidTaskType_SetVirtualPadStickLY);
}

void IoPortSender::SetVirtualPadStickR(int32_t controllerId, int32_t x, int32_t y)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    SetVirtualPadTask(controllerId, x, HidTaskType::HidTaskType_SetVirtualPadStickRX);
    SetVirtualPadTask(controllerId, y, HidTaskType::HidTaskType_SetVirtualPadStickRY);
}

// Xpad 状態取得
void IoPortSender::GetXpadStates(std::vector<HidShellBasicXpadState>& xpadStates)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();

    int32_t result;
    HidShellHandle hidShellHandle;
    result = hidShellLibrary.m_AquireHidShellExclusiveRight(reinterpret_cast<void**>(&hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);
    xpadStates.clear();

    // タスク実行後のステータス作成
    HidShellBasicXpadState  xpadStateBuffer = HidShellBasicXpadState();
    for (int i = 0; i < HidShellAbstractedPadCountMax; i++)
    {
        result = hidShellLibrary.m_GetHidShellBasicXpadState(
            reinterpret_cast<void*>(hidShellHandle), &(xpadStateBuffer),
            i, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));


        switch (static_cast<HidShellResult>(result))
        {
            case HidShellResult::Success:
                xpadStates.push_back(xpadStateBuffer);
                break;
            case HidShellResult::StateNotSet:
            case HidShellResult::PortNotFound:
                xpadStates.push_back(HidShellBasicXpadState());
                break;
            default:
                throw std::exception((std::string("Get xpad state failed in HidShellLibrary. error code: ") + std::to_string(result)).c_str());
        }
    }

    // HidShell 接続解除
    result = hidShellLibrary.m_ReleaseHidShellExclusiveRight(reinterpret_cast<void*>(hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);
}

// DebugPad 状態取得
void IoPortSender::GetDebugPadStates(HidShellDebugPadState& debugPadState)
{
    std::lock_guard<std::mutex> lock(m_ControllerJobMutex);
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();

    // HidShell 接続
    int32_t result;
    HidShellHandle hidShellHandle;
    result = hidShellLibrary.m_AquireHidShellExclusiveRight(reinterpret_cast<void**>(&hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);

    debugPadState = HidShellDebugPadState();
    HidShellDebugPadState debugPadBuffer;

    result = hidShellLibrary.m_GetHidShellDebugPadState(
            reinterpret_cast<void*>(hidShellHandle), &(debugPadBuffer),
            m_Port, static_cast<uint32_t>(HidShellPortDirection::In));
    switch (static_cast<HidShellResult>(result))
    {
        case HidShellResult::Success:
            debugPadState = debugPadBuffer;
            break;
        case HidShellResult::StateNotSet:
        case HidShellResult::PortNotFound:
            debugPadState = HidShellDebugPadState();
            break;
        default:
            throw std::exception((std::string("Get debug pad state failed in HidShellLibrary. error code: ") + std::to_string(result)).c_str());
    }

    // HidShell 接続解除
    result = hidShellLibrary.m_ReleaseHidShellExclusiveRight(reinterpret_cast<void*>(hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);

}

// VirtualPad 状態取得
void IoPortSender::GetVirtualPadStates(std::vector<HidShellAbstractedPadState>& virtualControllerStates)
{
    std::lock_guard<std::mutex> lock(m_VirtualControllerJobMutex);
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();

    // HidShell 接続
    int32_t result;
    HidShellHandle hidShellHandle;
    result = hidShellLibrary.m_AquireHidShellExclusiveRight(reinterpret_cast<void**>(&hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);
    virtualControllerStates.clear();

    // タスク実行後のステータス作成
    HidShellAbstractedPadState  virtualPadStateBuffer = HidShellAbstractedPadState();
    for (int i = 0; i < HidShellAbstractedPadCountMax; i++)
    {
        result = hidShellLibrary.m_GetHidShellAbstractedPadState(
            reinterpret_cast<void*>(hidShellHandle), &(virtualPadStateBuffer),
            i, m_Port, static_cast<uint32_t>(HidShellPortDirection::In));


        switch (static_cast<HidShellResult>(result))
        {
            case HidShellResult::Success:
                virtualControllerStates.push_back(virtualPadStateBuffer);
                break;
            case HidShellResult::StateNotSet:
            case HidShellResult::PortNotFound:
                virtualControllerStates.push_back(HidShellAbstractedPadState());
                break;
            default:
                throw std::exception((std::string("Get virtual pad state failed in HidShellLibrary. error code: ") + std::to_string(result)).c_str());
        }
    }

    // HidShell 接続解除
    result = hidShellLibrary.m_ReleaseHidShellExclusiveRight(reinterpret_cast<void*>(hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);
}


void IoPortSender::Flush()
{
    std::lock_guard<std::mutex> lockController(m_ControllerJobMutex);
    std::lock_guard<std::mutex> lockVirtualController(m_VirtualControllerJobMutex);
    std::lock_guard<std::mutex> lockTouch(m_TouchJobMutex);
    int32_t result;

    // HidShell 接続
    HidShellHandle hidShellHandle;
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();

    result = hidShellLibrary.m_AquireHidShellExclusiveRight(reinterpret_cast<void**>(&hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);


    UpdateXpadStatus(hidShellHandle);
    UpdateDebugPadState(hidShellHandle);
    UpdateTouchScreenStatus(hidShellHandle);
    UpdateVirtualPadStatus(hidShellHandle);

    // 入力記録
    if (m_IsRecordindHid)
    {
        m_IoPortRecorder.StoreHidInput(hidShellHandle, m_Port, static_cast<int32_t>(HidShellPortDirection::Out));
    }

    // HidShell 接続解除
    result = hidShellLibrary.m_ReleaseHidShellExclusiveRight(reinterpret_cast<void*>(hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);
}

HidInputerResult IoPortSender::SetPortNumberForTarget( std::string serialNumber )
{
    // ターゲット文字数チェック
    HidShellPortName portname;
    if ((sizeof(portname.string) / sizeof(portname.string[0])) < (serialNumber.size() + 1))
    {
        return HidInputerResult::HidInputerResult_Unexpected;
    }
    strcpy_s(portname.string, serialNumber.c_str());

    int32_t result;
    HidShellHandle hidShellHandle;
    HidShellLibraryWrapper& hidShellLibrary = HidShellLibraryWrapper::GetInstance();


    result = hidShellLibrary.m_AquireHidShellExclusiveRight(reinterpret_cast<void**>(&hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);

    int remainingTime = g_FilePortTimeout;
    std::chrono::system_clock::time_point startTime = std::chrono::system_clock::now();

    uint32_t port;

    bool targetFound = false;
    do
    {
        result = hidShellLibrary.m_GetHidShellPort(reinterpret_cast<void*>(hidShellHandle), &port, reinterpret_cast<void*>(&portname));
        switch (static_cast<HidShellResult>(result))
        {
            case HidShellResult::Success:
                m_Port = port;
                targetFound = true;
                break;
            case HidShellResult::PortNotFound:
                break;
            default:
                throw std::exception((std::string("Get port number failed. error code: ") + std::to_string(result)).c_str());
        }

        if (targetFound == false)
        {
            Sleep(min(g_FindPortSpan, remainingTime));
            remainingTime -= static_cast<int>(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now() - startTime).count());
        }
    } while (0 < remainingTime && targetFound == false );


    result = hidShellLibrary.m_ReleaseHidShellExclusiveRight(reinterpret_cast<void*>(hidShellHandle));
    assert(static_cast<HidShellResult>(result) == HidShellResult::Success);

    if (targetFound == false)
    {
        return HidInputerResult::HidInputerResult_TargetNotFound;
    }
    return HidInputerResult::HidInputerResult_Success;
}

HidInputerResult IoPortSender::SetPortNumberForLocal()
{
    m_Port = 0;
    return HidInputerResult::HidInputerResult_Success;
}
