﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <map>
#include <dshow.h>

#include "common.h"
#include "CaptureProcessorException.h"

class CaptureDeviceUtil
{
public:
    // デバイス数取得
    CaptureResult GetCaptureDeviceCount(int* pOutDeviceCount)
    {
        try
        {
            if (m_CaptureDeviceMap.empty())
            {
                CaptureResult result = UpdateDeviceMap();
                if (result == CaptureResult::CaptureResult_NotFound)
                {
                    (*pOutDeviceCount) = 0;
                    return CaptureResult::CaptureResult_Success;
                }
                else if (result != CaptureResult::CaptureResult_Success)
                {
                    return result;
                }
            }

            (*pOutDeviceCount) = static_cast<int>(m_CaptureDeviceMap.size());
        }
        catch (...)
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        return CaptureResult::CaptureResult_Success;
    }

    // デバイス名取得
    CaptureResult GetCaptureDeviceName(char* pOutNameBuffer, int nameBufferSize,int deviceId)
    {
        try
        {
            if (m_CaptureDeviceMap.empty())
            {
                CaptureResult result = UpdateDeviceMap();
                if (result != CaptureResult::CaptureResult_Success)
                {
                    return CaptureResult::CaptureResult_Unexpected;
                }
            }

            // 未取得のデバイス ID が指定された場合
            if (!m_CaptureDeviceMap.count(deviceId))
            {
                return CaptureResult::CaptureResult_Unexpected;
            }

            // 不正なバッファサイズ
            if (nameBufferSize <= 0)
            {
                return CaptureResult::CaptureResult_Unexpected;
            }

            std::string deviceName = m_CaptureDeviceMap[deviceId];
            if (deviceName.size() <= 0)
            {
                return CaptureResult::CaptureResult_Unexpected;
            }
            strncpy_s(pOutNameBuffer, nameBufferSize, deviceName.c_str(), min(nameBufferSize - 1, static_cast<int>(deviceName.size())));
            pOutNameBuffer[min(nameBufferSize - 1, static_cast<int>(deviceName.size()))] = '\0';
        }
        catch (...)
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        return CaptureResult::CaptureResult_Success;
    }

    // デバイス ID 更新
    CaptureResult UpdateDeviceMap()
    {
        // 記録済みのキャプチャデバイスクリア
        m_CaptureDeviceMap.clear();

        ICreateDevEnum* createDeviceEnum  = NULL;
        IEnumMoniker* enumMoniker   = NULL;
        ULONG nFetched              = 0;


        // COM 初期化
        CoInitialize(NULL);

        // デバイスを列挙するための ICreateDevEnum を作成
        HRESULT result = CoCreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER,
                    IID_ICreateDevEnum, (PVOID *)&createDeviceEnum);
        if (FAILED(result))
        {
            // ICreateDevEnum 作成失敗
            return CaptureResult::CaptureResult_Unexpected;
        }

        // ICreateDevEnum から、ビデオデバイスを列挙
        result = createDeviceEnum->CreateClassEnumerator(CLSID_VideoInputDeviceCategory, &enumMoniker, 0);

        if (result == S_FALSE && enumMoniker == NULL)
        {
            // ビデオデバイスが検知できない場合
            return CaptureResult::CaptureResult_NotFound;
        }
        else if (enumMoniker == NULL || FAILED(result))
        {
            // その他のエラー
            return CaptureResult::CaptureResult_Unexpected;
        }


        IMoniker *moniker;
        int deviceCounter = 0;
        while (enumMoniker->Next(1, &moniker, &nFetched) == S_OK)
        {
            // variant 初期化
            VARIANT         varName;
            VariantInit(&varName);

            IPropertyBag*   propertyBag;
            result = moniker->BindToStorage(0, 0, IID_IPropertyBag, (void**)(&propertyBag));
            if (FAILED(result))
            {
                // バインド失敗時はスキップ
                moniker->Release();
                continue;
            }

            result = propertyBag->Read( L"FriendlyName", &varName, 0 );

            if (SUCCEEDED(result))
            {
                const int deviceNameBufferSize = 256;
                char deviceNameTmp[deviceNameBufferSize];



                // BSTR 形式を C 文字列へ変換
                WideCharToMultiByte(CP_UTF8, 0, varName.bstrVal, -1, deviceNameTmp, deviceNameBufferSize, 0, 0);

                m_CaptureDeviceMap[deviceCounter] = deviceNameTmp;
                deviceCounter = deviceCounter + 1;
            }

            // variant クリア
            VariantClear(&varName);

            moniker->Release();
            propertyBag->Release();
        }

        // COM 解放
        enumMoniker->Release();
        createDeviceEnum->Release();
        CoUninitialize();

        return CaptureResult::CaptureResult_Success;
    }

    static CaptureDeviceUtil& CaptureDeviceUtil::GetInstance()
    {
        static CaptureDeviceUtil instance;
        return instance;
    }

private:
    CaptureDeviceUtil()
    {
        m_CaptureDeviceMap.clear();
    }
    ~CaptureDeviceUtil()
    {
        m_CaptureDeviceMap.clear();
    }
    CaptureDeviceUtil& operator=(const CaptureDeviceUtil &copy);
    CaptureDeviceUtil(const CaptureDeviceUtil &copy);

    // キャプチャデバイス名・ID
    std::map<int, std::string>          m_CaptureDeviceMap;

};
