﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <map>
#include <memory>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_Utility.h>
#include <nn/oe.h>
#include <nn/os/os_Argument.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

using namespace nn;

namespace {

class Param
{
public:
    Param() NN_NOEXCEPT
      : baseAppId({0x01004b9000490000}),
        count(16),
        sizeMin(4 * 1024 * 1024),
        sizeMax(64 * 1024 * 1024),
        userId(fs::InvalidUserId),
        uidIdx(0),
        seed(0)
    {
        os::GenerateRandomBytes(&seed, sizeof(seed));
    }

    ncm::ApplicationId GetApplicationId(int offset)
    {
        ncm::ApplicationId appId = { baseAppId.value + offset * 0x10000 };
        return appId;
    }

    void SetUserId()
    {
        std::unique_ptr<account::Uid[]> pUser(new account::Uid[uidIdx + 1]);
        int userCount = 0;
        auto result = account::ListAllUsers(&userCount, pUser.get(), uidIdx + 1);
        if (result.IsFailure() || (userCount != uidIdx + 1))
        {
            NN_LOG("[TestSaveDataCreatorLog] There is no account indexed by %d\n", uidIdx);
        }
        userId = fs::ConvertAccountUidToFsUserId(pUser[uidIdx]);
    }

    void GetSaveDataSize(int64_t* outAvailableSize, int64_t* outJournalSize)
    {
        static std::mt19937 s_mt(seed);
        auto targetSize = std::uniform_int_distribution<int64_t>(sizeMin, sizeMax)(s_mt);
        int64_t journalSize = ((targetSize > 32 * 1024 * 1024) ? (8 * 1024 * 1024) : (512 * 1024)) + 48 * 1024;
        int64_t availableSize = targetSize - journalSize;
        while (NN_STATIC_CONDITION(true))
        {
            int64_t saveDataSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::QuerySaveDataTotalSize(&saveDataSize, availableSize, journalSize));
            if (saveDataSize > targetSize)
            {
                if (availableSize < 1 * 1024 * 1024)
                {
                    break;
                }
                availableSize -= 1 * 1024 * 1024;
                continue;
            }
            else
            {
                break;
            }
        }
        *outAvailableSize = availableSize;
        *outJournalSize = journalSize;
    }

    void Show()
    {
        NN_LOG("base: 0x%016llx\n", baseAppId.value);
        NN_LOG("count: %d\n", count);
        NN_LOG("sizemin: %lld\n", sizeMin);
        NN_LOG("sizemax: %lld\n", sizeMax);
        NN_LOG("uid: %016llx_%016llx\n", userId._data[0], userId._data[1]);
        NN_LOG("seed: %u\n", seed);
    }

    bool Check()
    {
        bool ret = true;

        if (sizeMin < 1 * 1024 * 1024)
        {
            NN_LOG("sizeMin must be larger than 1 MiB.\n");
            ret = false;
        }

        if (sizeMax > (int64_t)4 * 1024 * 1024 * 1024 - 1)
        {
            NN_LOG("sizeMax must be smaller than 4 GiB - 1.\n");
            ret = false;
        }

        if (sizeMin > sizeMax)
        {
            NN_LOG("sizeMax must be equal to or larger than sizeMin.\n");
            ret = false;
        }

        {
            bool isUserExist = false;
            account::Uid user = {{ userId._data[0], userId._data[1] }};
            if (!user)
            {
                NN_LOG("There is no valid user.\n");
                ret = false;
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserExistence(&isUserExist, user));
                if (!isUserExist)
                {
                    NN_LOG("There is no user has specified uid.\n");
                    ret = false;
                }
            }
        }

        return ret;
    }

public:
    ncm::ApplicationId baseAppId;
    int count;
    int64_t sizeMin;
    int64_t sizeMax;
    fs::UserId userId;
    int uidIdx;
    unsigned int seed;
};

class Option
{
public:
    Option(int argc, char** argv)
    {
        for (int i = 0; i < argc - 1; i++)
        {
            auto pArg = argv[i];
            auto pNextArg = argv[i + 1];

            if (pArg[0] != '-' || pNextArg[0] == '-')
            {
                continue;
            }

            m_Map[pArg] = pNextArg;
        }
    }

    bool HasKey(const char* key)
    {
        return m_Map.find(key) != m_Map.cend();
    }

    const char* GetValue(const char* key)
    {
        auto it = m_Map.find(key);
        return it != m_Map.cend() ? (*it).second.c_str() : nullptr;
    }

private:
    std::map<std::string, std::string> m_Map;
};

void CreateTestSaveData(ncm::ApplicationId applicationId, fs::UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize)
{
    NN_LOG("Creating savedata (appId: 0x%016llx, uid: %016llx_%016llx, dataSize: %lld, journalSize: %lld).\n", applicationId.value, userId._data[0], userId._data[1], size, journalSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateSaveData(applicationId, userId, applicationId.value, size, journalSize, 0));
    const char* MountName = "save";
    const char* fileName = "save:/file.dat";
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::MountSaveData(MountName, applicationId, userId));
    const int64_t BufferSize = journalSize - 48 * 1024;
    std::unique_ptr<char> buffer(new char[BufferSize]);
    {
        uint32_t count = 0;
        uint32_t* pBuffer32 = reinterpret_cast<uint32_t*>(buffer.get());
        for (size_t i = 0; i < BufferSize / 4; ++i)
        {
            pBuffer32[i] = count;
            ++count;
        }
    }
    {
        int64_t wrote = 0;
        int64_t toWrote = size - 48 * 1024;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateFile(fileName, toWrote));
        fs::CommitSaveData(MountName);
        fs::FileHandle file;
        while (toWrote > 0)
        {
            int64_t writeSize = std::min(BufferSize, toWrote);
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&file, fileName, fs::OpenMode_Write));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(file, wrote, buffer.get(), writeSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
            toWrote -= writeSize;
            wrote += writeSize;
            fs::CloseFile(file);
            fs::CommitSaveData(MountName);
        }
    }
    fs::Unmount(MountName);
}

void DeleteTestSaveData()
{
    std::unique_ptr<fs::SaveDataIterator> iter;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenSaveDataIterator(&iter, fs::SaveDataSpaceId::User));

    int64_t count;
    nn::fs::SaveDataInfo info;
    while (NN_STATIC_CONDITION(true))
    {
        count = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
        if( count == 0 )
        {
            return;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::DeleteSaveData(fs::SaveDataSpaceId::User, info.saveDataId));
        iter.reset();
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenSaveDataIterator(&iter, fs::SaveDataSpaceId::User));
    }
}

}

extern "C" void nnMain()
{
    NN_LOG("=== TestSaveData Creator ===\n");
    NN_LOG("Usage1: \"RunOnTarget TestSaveDataCreator.nsp -- [option]\" to create test savedata.\n");
    NN_LOG("  --base <id>: base application ID. (0x01004b9000490000 as default)\n");
    NN_LOG("  --count <num>: application num from base. (16 as default)\n");
    NN_LOG("  --sizemin <bytes>: minimum size of save data. (4194304 as default)\n");
    NN_LOG("  --sizemax <bytes>: maximum size of save data. (67108864 as default)\n");
    NN_LOG("  --uid <uid>: uid of user account. (exclusive with --uididx)\n");
    NN_LOG("  --uididx <index>: index of user account. (0 as default. exclusive with --uid)\n");
    NN_LOG("  --seed <value>: seed to determine random save data size.\n");
    NN_LOG("Usage2: \"RunOnTarget TestSaveDataCreator.nsp\" to delete all test savedata.\n");

    auto argc = os::GetHostArgc();
    if (argc == 1)
    {
        NN_LOG("Delete All Test save data.\n");
        DeleteTestSaveData();
        NN_LOG("Success.\n");
        return;
    }

    Option option(os::GetHostArgc(), os::GetHostArgv());

    oe::Initialize();
    account::Initialize();

    Param param;

    const char* baseOption = "--base";
    const char* countOption = "--count";
    const char* sizeMinOption = "--sizemin";
    const char* sizeMaxOption = "--sizemax";
    const char* uidOption = "--uid";
    const char* uidIdxOption = "--uididx";
    const char* seedOption = "--seed";

    if (option.HasKey(baseOption))
    {
        param.baseAppId = {std::strtoull(option.GetValue(baseOption), nullptr, 16)};
    }
    if (option.HasKey(countOption))
    {
        param.count = std::atoi(option.GetValue(countOption));
    }
    if (option.HasKey(sizeMinOption))
    {
        param.sizeMin = std::strtoll(option.GetValue(sizeMinOption), nullptr, 10);
    }
    if (option.HasKey(sizeMaxOption))
    {
        param.sizeMax = std::strtoll(option.GetValue(sizeMaxOption), nullptr, 10);
    }
    if (option.HasKey(uidOption))
    {
        auto str = std::string(option.GetValue(uidOption));
        auto uidHigh = std::strtoull(str.substr(0, 16).c_str(), nullptr, 16);
        auto uidLow = std::strtoull(str.substr(16, 16).c_str(), nullptr, 16);
        param.userId = {{uidHigh, uidLow}};
    }
    else if (option.HasKey(uidIdxOption))
    {
        param.uidIdx = std::atoi(option.GetValue(uidIdxOption));
        param.SetUserId();
    }
    else
    {
        param.SetUserId();
    }
    if (option.HasKey(seedOption))
    {
        param.seed = static_cast<unsigned int>(std::atoi(option.GetValue(seedOption)));
    }

    param.Show();
    if (!param.Check())
    {
        return;
    }

    for (int i = 0; i < param.count; i++)
    {
        auto appId = param.GetApplicationId(i);
        int64_t availableSize = 0;
        int64_t journalSize = 0;
        param.GetSaveDataSize(&availableSize, &journalSize);
        CreateTestSaveData(appId, param.userId, appId.value, availableSize, journalSize);
    }

    NN_LOG("Succeeded to create test savedata.\n");

    return;
}
