﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/nn_Log.h>

#include "ShopMonitoringTool_SysLog.h"

Log::Log()
{
    m_BufferTailPos = 0;
    m_LineCount = 0;
    nn::os::InitializeMutex(&m_Mutex, false, 0);
}

Log::~Log()
{
    m_BufferTailPos = 0;
    m_LineCount = 0;
    nn::os::FinalizeMutex(&m_Mutex);
}

void Log::Initialize() NN_NOEXCEPT
{
    // 時刻ライブラリの初期化
    nn::time::Initialize();
}

Log& Log::GetInstance() NN_NOEXCEPT
{
    static Log instance;
    return instance;
}

Log::LogLine Log::GetLine(int lineNum) NN_NOEXCEPT
{
    return m_LogLine[lineNum];
}

Log::LogLevel Log::GetLevel(int lineNum) NN_NOEXCEPT
{
    return m_LogLine[lineNum].m_LogLevel;
}

const char* Log::GetBuffer(int lineNum) NN_NOEXCEPT
{
    return &m_LogBuffer[m_LogLine[lineNum].m_BufferPos];
}

int Log::GetCount() NN_NOEXCEPT
{
    return m_LineCount;
}

void Log::PrintToLog(LogLevel level, const char *format, ...) NN_NOEXCEPT
{
    std::va_list list;
    va_start(list, format);
    std::vsnprintf(m_WorkBuffer, sizeof(m_WorkBuffer) / sizeof(*m_WorkBuffer) - 1, format, list);
    va_end(list);
    NN_LOG("%s",m_WorkBuffer);
    char* pBufferToken = strtok(m_WorkBuffer, "\n");
    this->AppendLog(pBufferToken, level);
    while (pBufferToken != nullptr)
    {
        pBufferToken = strtok(NULL, "\n");
        if (pBufferToken != nullptr) {
            this->AppendLog(pBufferToken, level);
        }
    }
}

void Log::AppendLog(char *buf, LogLevel level) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    int bufLength = (int)strlen(buf) + 1;

    // バッファよりも大きい
    if (TimeLogLength + bufLength > MaxLogBufferSize)
    {
        return;
    }

    // 格納しきれないなら先頭に戻る
    if (m_BufferTailPos + TimeLogLength + bufLength >= MaxLogBufferSize)
    {
        m_BufferTailPos = 0;
    }

    int currentBufferPos = m_BufferTailPos;

    // 時刻
    nn::time::PosixTime posixTime;
    nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
    nn::util::SNPrintf(&m_LogBuffer[currentBufferPos], TimeLogLength + 1, TimeForm, calendarTime.hour, calendarTime.minute, calendarTime.second);
    memcpy(&m_LogBuffer[currentBufferPos + TimeLogLength], buf, bufLength * sizeof(char));
    bufLength += TimeLogLength;
    m_BufferTailPos += bufLength;

    m_LogLine[m_LineCount % MaxLogLineSize].m_LineNumber   = m_LineCount;
    m_LogLine[m_LineCount % MaxLogLineSize].m_LogLevel     = level;
    m_LogLine[m_LineCount % MaxLogLineSize].m_BufferPos    = currentBufferPos;
    m_LogLine[m_LineCount % MaxLogLineSize].m_BufferSize   = bufLength;
    m_LineCount++;

    nn::os::UnlockMutex(&m_Mutex);
}
