﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_SysGraphics.h"

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

Graphics::Graphics()
{
    m_Initialized = false;
}

Graphics::~Graphics()
{
    m_Initialized = false;
}

Graphics& Graphics::GetInstance() NN_NOEXCEPT
{
    static Graphics instance;
    return instance;
}

void Graphics::Initialize() NN_NOEXCEPT
{
    InitializeGraphicsFramework();
    InitializePrimitiveRenderer();
    InitializeDebugFont();
    InitializeUserShader();
    m_Initialized = true;
}

bool Graphics::IsInitialized() const NN_NOEXCEPT
{
    return m_Initialized;
}

void Graphics::Finalize() NN_NOEXCEPT
{
    m_Initialized = false;
    FinalizeUserShader();
    FinalizeDebugFont();
    FinalizePrimitiveRenderer();
    FinalizeGraphicsFramework();
}

void* Graphics::GetFramework() NN_NOEXCEPT
{
    static nns::gfx::GraphicsFramework instance;
    return (void*)&instance;
}

void* Graphics::GetWriter() NN_NOEXCEPT
{
    static nn::gfx::util::DebugFontTextWriter instance;
    return (void*)&instance;
}

void* Graphics::GetRenderer(void* setRenderer) NN_NOEXCEPT
{
    static nns::gfx::PrimitiveRenderer::Renderer* instance;
    if (setRenderer != nullptr)
    {
        instance = (nns::gfx::PrimitiveRenderer::Renderer*)setRenderer;
    }
    return (void*)instance;
}

void Graphics::ClearColor() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();
    nn::gfx::CommandBuffer* pCommand = pFramework->GetRootCommandBuffer(m_BufferIndex);
    nn::gfx::ColorTargetView* target = pFramework->GetColorTargetView();
    pCommand->ClearColor(target, 0.1f, 0.5f, 0.1f, 1.0f, nullptr);
}

void Graphics::BeginDraw() NN_NOEXCEPT
{
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = (nns::gfx::PrimitiveRenderer::Renderer*)GetRenderer();
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();
    nn::gfx::CommandBuffer* pCommand = pFramework->GetRootCommandBuffer(m_BufferIndex);

    pFramework->ProcessFrame();

    pFramework->BeginFrame(m_BufferIndex);

    nn::gfx::ColorTargetView* target = pFramework->GetColorTargetView();

    pCommand->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, nullptr);
    pCommand->SetRenderTargets(1, &target, nullptr);
    pCommand->SetViewportScissorState(pFramework->GetViewportScissorState());

    pRenderer->Update(m_BufferIndex);

    pRenderer->SetDepthStencilState(pCommand, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);
}

void Graphics::EndDraw() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();
    nn::gfx::util::DebugFontTextWriter* pWriter = (nn::gfx::util::DebugFontTextWriter*)GetWriter();
    nn::gfx::CommandBuffer* pCommand = pFramework->GetRootCommandBuffer(m_BufferIndex);

    pWriter->Draw(pCommand);

    nn::gfx::ColorTargetView* target = pFramework->GetColorTargetView();
    pCommand->SetRenderTargets(1, &target, nullptr);
    pCommand->FlushMemory(nn::gfx::GpuAccess_ColorBuffer);

    pFramework->EndFrame(m_BufferIndex, true);

    m_BufferIndex = 1 - m_BufferIndex;
}

int Graphics::GetBufferIndex() NN_NOEXCEPT
{
    return m_BufferIndex;
}
