﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>
#include <string>

#include <nn/nn_Log.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/ec/ec_Result.h>
#include <nn/ec/system/ec_DeviceAccountApi.h>
#include <nn/ec/system/ec_DeviceAuthenticationApi.h>
#include <nn/ec/system/ec_DeviceLinkApi.h>
#include <nn/ec/system/ec_TicketApi.h>
#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/nifm.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include "ShopMonitoringTool_ShopCommandManager.h"
#include "ShopMonitoringTool_ShopCommandUtil.h"
#include "ShopMonitoringTool_ShopCommandJson.h"

void ShopCommandManager::SetUserIndex(int userIndex)
{
    m_UserIndex = userIndex;
}

void ShopCommandManager::SetTimeOut(nn::util::optional<nn::TimeSpan> timeout)
{
    m_TimeOut = timeout;
}

void ShopCommandManager::SetDownloadTicketApplicationIdString(std::string downloadTicketApplicationIdString)
{
    m_DownloadTicketApplicationIdString = downloadTicketApplicationIdString;
}

void ShopCommandManager::SetDownloadTicketKeyIdString(std::string downloadTicketKeyIdString)
{
    m_DownloadTicketKeyIdString = downloadTicketKeyIdString;
}

void ShopCommandManager::SetDownloadDemoApplicationIdString(std::string downloadDemoApplicationIdString)
{
    m_DownloadDemoApplicationIdString = downloadDemoApplicationIdString;
}

void ShopCommandManager::SetPurchaseNsUidString(std::string purchaseNsUidString)
{
    m_PurchaseNsUidString = purchaseNsUidString;
}

void AppendLogParseResponse(char* buffer)
{
    if (strlen(buffer) < 1)
    {
        APPLOG_INFO("Empty response\n");
        return;
    }

    nne::rapidjson::Document document;
    if (document.Parse(buffer).HasParseError())
    {
        APPLOG_WARNING("Json Parse Error\n");
        return;
    }
    nne::rapidjson::StringBuffer stringBuffer;
    nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(stringBuffer);
    document.Accept(writer);
    APPLOG_INFO("%s\n", stringBuffer.GetString());
}

nn::Result ShopCommandManager::WaitAsync(nn::ec::system::AsyncResult* async)
{
    auto timeout = m_TimeOut;
    if (timeout && !async->GetEvent().TimedWait(*timeout))
    {
        APPLOG_ERROR("Async timeout\n");
        async->Cancel();
    }

    NN_RESULT_DO(async->Get());
    NN_RESULT_SUCCESS;
}

template<typename T>
nn::Result ShopCommandManager::WaitAsync(T* outValue, nn::ec::system::AsyncValue<T>* async)
{
    auto timeout = m_TimeOut;
    if (timeout && !async->GetEvent().TimedWait(*timeout))
    {
        APPLOG_ERROR("Async timeout\n");
        async->Cancel();
    }

    NN_RESULT_DO(async->Get(outValue));
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::CheckUidFromIndex(bool* outValue, nn::util::optional<nn::account::Uid>* pUid, int userIndex)
{
    NN_RESULT_DO(GetUidFromIndex(pUid, userIndex));

    if (!pUid)
    {
        APPLOG_INFO("Not found user index %d\n", userIndex);

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::RegisterDeviceAccount(bool* outValue)
{
    nn::ec::system::AsyncResult async;
    NN_RESULT_DO(nn::ec::system::RequestRegisterDeviceAccount(&async));
    NN_RESULT_DO(WaitAsync(&async));

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::UnregisterDeviceAccount(bool* outValue)
{
    nn::ec::system::AsyncResult async;
    NN_RESULT_DO(nn::ec::system::RequestUnregisterDeviceAccount(&async));
    NN_RESULT_DO(WaitAsync(&async));

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DeviceRegistrationInfo(bool* outValue)
{
    nn::ec::system::AsyncDeviceRegistrationInfo async;
    NN_RESULT_DO(nn::ec::system::RequestDeviceRegistrationInfo(&async));

    nn::ec::system::DeviceRegistrationInfo regInfo;
    NN_RESULT_TRY(WaitAsync(&regInfo, &async))
        NN_RESULT_CATCH(nn::ec::ResultDeviceNotRegistered)
    {
        APPLOG_ERROR("Device not registered\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_END_TRY

    APPLOG_INFO("   DeviceAccountID : %s\n", regInfo.accountInfo.id);
    APPLOG_INFO("   DeviceAccountStatus : %s\n", regInfo.accountStatus.data);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DeviceAccountStatus(bool* outValue)
{
    nn::ec::system::AsyncDeviceAccountStatus asyncStatus;
    NN_RESULT_DO(nn::ec::system::RequestDeviceAccountStatus(&asyncStatus));

    nn::ec::system::DeviceAccountStatus status;
    NN_RESULT_DO(WaitAsync(&status, &asyncStatus));

    APPLOG_INFO("   DeviceAccountStatus : %s\n", status.data);
    APPLOG_INFO("   NeedsTicketSync : %s\n", status.needsTicketSync ? "true" : "false");

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DeviceAccountInfo(bool* outValue)
{
    nn::ec::system::DeviceAccountInfo info;
    NN_RESULT_TRY(nn::ec::system::GetDeviceAccountInfo(&info))
        NN_RESULT_CATCH(nn::ec::ResultDeviceAccountNotRegistered)
    {
        APPLOG_ERROR("Device account is not registered\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_END_TRY

        APPLOG_INFO("   DeviceAccountId : %s\n", info.id);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DeviceAuthenticationToken(bool* outValue)
{
    nn::ec::system::AsyncDeviceAuthenticationToken async;
    NN_RESULT_DO(nn::ec::system::RequestDeviceAuthenticationToken(&async));

    nn::ec::system::DeviceAuthenticationToken token;
    NN_RESULT_DO(WaitAsync(&token, &async));

    APPLOG_INFO("   DeviceAuthToken : %s\n", token.data);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::ShopAccountStatus(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    APPLOG_INFO("Request shop account status for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);

    char readBuffer[2048];
    auto result = GetShopAccountStatusResponse(readBuffer, sizeof(readBuffer), *uid);
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(result);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DeviceLinkStatus(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    APPLOG_INFO("Request device link status for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);

    char readBuffer[2048];
    auto result = GetDeviceLinkStatusResponse(readBuffer, sizeof(readBuffer), *uid);
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(result);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::HasDeviceLink(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    nne::rapidjson::Document document(nn::ec::system::HasDeviceLink(*uid) ? nne::rapidjson::kTrueType : nne::rapidjson::kFalseType);

    nne::rapidjson::StringBuffer buffer;
    nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
    document.Accept(writer);

    APPLOG_INFO("   HasDeviceLink : %s\n", buffer.GetString());
    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DeleteAllRights(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    APPLOG_INFO("Request delete all rights for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);

    char readBuffer[2048];
    auto result = GetDeleteAllRightsResponse(readBuffer, sizeof(readBuffer), *uid);
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(result);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DownloadTicket(bool* outValue)
{
    std::string rightsIdString = m_DownloadTicketApplicationIdString + m_DownloadTicketKeyIdString;

    nn::es::RightsIdIncludingKeyId rightsId = {};
    for (int i = 0; i < sizeof(nn::es::RightsIdIncludingKeyId); i++)
    {
        rightsId._data[i] = static_cast<uint8_t>(std::stoi(rightsIdString.substr(i * 2, 2), 0, 16));
    }

    nn::ec::system::AsyncResult async;
    NN_RESULT_DO(nn::ec::system::RequestDownloadTicket(&async, rightsId));
    NN_RESULT_DO(WaitAsync(&async));

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::Prepurchase(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    if (!uid)
    {
        APPLOG_ERROR("Not found user index %d\n", m_UserIndex);

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    auto nsuid = std::strtoll(m_PurchaseNsUidString.c_str(), nullptr, 10);
    APPLOG_INFO("Request prepurchase ns-uid %lld for uid %016llx-%016llx\n", nsuid, uid->_data[0], uid->_data[1]);

    const char* typeOption = nullptr;
    char readBuffer[2048];
    auto result = GetPrepurchaseResponse(readBuffer, sizeof(readBuffer), *uid, nsuid, typeOption ? typeOption : "titles");
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(result);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

bool ParsePrepurchaseResponse(int64_t* outPriceId, int64_t* outAmount, int64_t* outPostBalance, char* buffer)
{
    nne::rapidjson::Document document;
    if (document.ParseInsitu(buffer).HasParseError())
    {
        return false;
    }

    auto contentsObj = document.FindMember("contents");
    if (contentsObj == document.MemberEnd())
    {
        return false;
    }
    auto& item = contentsObj->value[0];
    auto paymentObj = item.FindMember("payment_amount");
    if (paymentObj == item.MemberEnd())
    {
        return false;
    }
    auto priceObj = paymentObj->value.FindMember("price");
    if (priceObj == paymentObj->value.MemberEnd())
    {
        return false;
    }
    auto regularPriceObj = priceObj->value.FindMember("regular_price");
    if (regularPriceObj == priceObj->value.MemberEnd())
    {
        return false;
    }
    auto priceIdObj = regularPriceObj->value.FindMember("id");
    if (priceIdObj == regularPriceObj->value.MemberEnd())
    {
        return false;
    }
    auto amountObj = regularPriceObj->value.FindMember("raw_value");
    if (amountObj == regularPriceObj->value.MemberEnd())
    {
        return false;
    }
    if (!priceIdObj->value.IsInt() || !amountObj->value.IsString())
    {
        return false;
    }

    *outPriceId = priceIdObj->value.GetInt64();
    *outAmount = std::strtoll(amountObj->value.GetString(), nullptr, 10);

    //

    auto postBalanceObj = document.FindMember("post_balance");
    if (postBalanceObj == document.MemberEnd())
    {
        return false;
    }
    auto postBalanceValueObj = postBalanceObj->value.FindMember("raw_value");
    if (postBalanceValueObj == postBalanceObj->value.MemberEnd())
    {
        return false;
    }
    if (!postBalanceValueObj->value.IsString())
    {
        return false;
    }

    *outPostBalance = std::strtoll(postBalanceValueObj->value.GetString(), nullptr, 10);

    return true;
}

nn::Result ShopCommandManager::Purchase(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    if (!uid)
    {
        APPLOG_ERROR("Not found user index %d\n", m_UserIndex);

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    const char* typeOption = nullptr;
    auto nsUid = std::strtoll(m_PurchaseNsUidString.c_str(), nullptr, 10);

    int64_t priceId;
    int64_t amount;
    int64_t postBalance;
    char readBuffer[2048];

    auto prepurchaseResult = GetPrepurchaseResponse(readBuffer, sizeof(readBuffer), *uid, nsUid, typeOption ? typeOption : "titles");
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(prepurchaseResult);


    if (!ParsePrepurchaseResponse(&priceId, &amount, &postBalance, readBuffer))
    {
        APPLOG_ERROR("Unexpected prepurchase response\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    // 残高不足で成功扱い
    if (postBalance < 0)
    {
        APPLOG_INFO("Not enough money\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    APPLOG_INFO("Request purchase ns-uid %lld price-id %lld amount %lld for uid %016llx-%016llx\n", nsUid, priceId, amount, uid->_data[0], uid->_data[1]);
    auto result = GetPurchaseResponse(readBuffer, sizeof(readBuffer), *uid, nsUid, priceId, amount, typeOption ? typeOption : "titles");
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(result);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::DownloadDemo(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    if (!uid)
    {
        APPLOG_ERROR("Not found user index %d\n", m_UserIndex);

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    auto metaId = std::strtoull(m_DownloadDemoApplicationIdString.c_str(), nullptr, 16);
    int64_t nsuid;
    char readBuffer[2048];

    NN_RESULT_DO(GetNsuidByContentMetaId(
        &nsuid, readBuffer, sizeof(readBuffer), metaId, ShopIdType::Demo));

    auto result = GetDownloadDemoResponse(readBuffer, sizeof(readBuffer), *uid, nsuid);
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(result);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::LinkDevice(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    APPLOG_INFO("Request link device for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);
    nn::ec::system::AsyncResult async;
    NN_RESULT_DO(nn::ec::system::RequestLinkDevice(&async, *uid));
    NN_RESULT_DO(WaitAsync(&async));

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::UnlinkDevice(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    APPLOG_INFO("Request unlink device for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);
    nn::ec::system::AsyncResult async;
    NN_RESULT_DO(nn::ec::system::RequestUnlinkDevice(&async, *uid));
    NN_RESULT_DO(WaitAsync(&async));

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::UnlinkDeviceAll(bool* outValue)
{
    char readBuffer[2048];
    auto result = GetUnlinkDeviceAllResponse(readBuffer, sizeof(readBuffer));
    AppendLogParseResponse(readBuffer);
    NN_RESULT_DO(result);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::SyncTicket(bool* outValue)
{
    auto timeout = m_TimeOut;
    nn::os::TimerEvent timer(nn::os::EventClearMode_ManualClear);
    if (timeout)
    {
        timer.StartOneShot(*timeout);
    }

    nn::ec::system::AsyncProgressResult asyncProgressResult;
    NN_RESULT_DO(nn::ec::system::RequestSyncTicket(&asyncProgressResult));

    while (!asyncProgressResult.TryWait())
    {
        if (timer.TryWait())
        {
            asyncProgressResult.Cancel();
            break;
        }

        auto progress = asyncProgressResult.GetProgress();
        APPLOG_INFO("%lld / %lld\n", progress.done, progress.total);
        asyncProgressResult.GetEvent().TimedWait(nn::TimeSpan::FromSeconds(1));
    }
    NN_RESULT_DO(asyncProgressResult.Get());
    auto progress = asyncProgressResult.GetProgress();
    APPLOG_INFO("%lld / %lld\n", progress.done, progress.total);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::Search(bool* outValue)
{
    static const int BufferSize = 4096 * 1024;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);

    NN_RESULT_DO(GetSearchResponse(buffer.get(), BufferSize));

    nne::rapidjson::Document document;
    if (document.ParseInsitu(buffer.get()).HasParseError())
    {
        APPLOG_ERROR("Invalid download task list json\n");
    }

    nne::rapidjson::StringBuffer stringBuffer;
    nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(stringBuffer);
    document.Accept(writer);

    APPLOG_INFO("%s\n", stringBuffer.GetString());
    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ShopCommandManager::Start(bool* outValue)
{
    nn::util::optional<nn::account::Uid> uid;
    NN_RESULT_DO(CheckUidFromIndex(outValue, &uid, m_UserIndex));

    {
        APPLOG_INFO("Request device authentication token...\n");
        nn::ec::system::AsyncDeviceAuthenticationToken async;
        NN_RESULT_DO(nn::ec::system::RequestCachedDeviceAuthenticationToken(&async, nn::TimeSpan::FromHours(3)));
        nn::ec::system::DeviceAuthenticationToken token;
        NN_RESULT_DO(async.Get(&token));
        APPLOG_INFO("   DeviceAuthToken : %s\n", token.data);
    }

    nn::util::optional<nn::ec::system::DeviceAccountInfo> deviceAccountInfo;
    {
        APPLOG_INFO("Check device account is saved\n");
        deviceAccountInfo.emplace();;
        NN_RESULT_TRY(nn::ec::system::GetDeviceAccountInfo(&(*deviceAccountInfo)))
            NN_RESULT_CATCH(nn::ec::ResultDeviceAccountNotRegistered) {}
        {
            deviceAccountInfo = nn::util::nullopt;
        }
        NN_RESULT_END_TRY
    }

    if (!deviceAccountInfo)
    {
        APPLOG_INFO("Device account is not saved\n");
        APPLOG_INFO("Request device registration info...\n");

        nn::util::optional<nn::ec::system::DeviceRegistrationInfo> deviceRegistrationInfo;
        {
            nn::ec::system::AsyncDeviceRegistrationInfo async;
            NN_RESULT_DO(nn::ec::system::RequestDeviceRegistrationInfo(&async));
            deviceRegistrationInfo.emplace();
            NN_RESULT_TRY(async.Get(&(*deviceRegistrationInfo)))
                NN_RESULT_CATCH(nn::ec::ResultDeviceNotRegistered)
            {
                deviceRegistrationInfo = nn::util::nullopt;
            }
            NN_RESULT_END_TRY
        }

        if (deviceRegistrationInfo)
        {
            APPLOG_INFO("   DeviceAccountId : %s\n", deviceRegistrationInfo->accountInfo.id);
            APPLOG_INFO("   DeviceAccountStatus : %s\n", deviceRegistrationInfo->accountStatus.data);

            APPLOG_INFO("Registered device is found even device account is not saved\n");
            APPLOG_INFO("Seems this device is initialized\n");
            APPLOG_INFO("Request unlink device all...\n");

            char readBuffer[2048];
            NN_RESULT_DO(GetUnlinkDeviceAllResponse(readBuffer, sizeof(readBuffer)));
            AppendLogParseResponse(readBuffer);

            APPLOG_INFO("Request unregister device account...\n");

            nn::ec::system::AsyncResult async;
            NN_RESULT_DO(nn::ec::system::RequestUnregisterDeviceAccount(&async));
            NN_RESULT_DO(async.Get());
        }

        {
            APPLOG_INFO("Request register device account...\n");
            nn::ec::system::AsyncResult async;
            NN_RESULT_DO(nn::ec::system::RequestRegisterDeviceAccount(&async));
            NN_RESULT_DO(async.Get());
        }

        {
            APPLOG_INFO("Check device account is saved");
            deviceAccountInfo.emplace();
            NN_RESULT_DO(nn::ec::system::GetDeviceAccountInfo(&(*deviceAccountInfo)));
        }
    }

    APPLOG_INFO("   DeviceId : %s\n", deviceAccountInfo->id);

    {
        APPLOG_INFO("Check shop account status...\n");

        char readBuffer[2048];
        NN_RESULT_DO(GetShopAccountStatusResponse(readBuffer, sizeof(readBuffer), *uid));
        AppendLogParseResponse(readBuffer);
    }

    {
        APPLOG_INFO("Check device link status...\n");

        char readBuffer[2048];
        NN_RESULT_DO(GetDeviceLinkStatusResponse(readBuffer, sizeof(readBuffer), *uid));
        AppendLogParseResponse(readBuffer);

        if (std::strcmp(readBuffer, "[]") == 0)
        {
            APPLOG_INFO("No device is linked to this user\n");
            APPLOG_INFO("Request link device...\n");

            nn::ec::system::AsyncResult async;
            NN_RESULT_DO(nn::ec::system::RequestLinkDevice(&async, *uid));
            NN_RESULT_DO(async.Get());
        }
        else
        {
            APPLOG_INFO("Device already linked to this user\n");
            APPLOG_INFO("Skip link device\n");
        }
    }

    {
        APPLOG_INFO("Check device account status...\n");

        nn::ec::system::DeviceAccountStatus status;
        {
            nn::Result result;
            {
                nn::ec::system::AsyncDeviceAccountStatus async;
                NN_RESULT_DO(nn::ec::system::RequestDeviceAccountStatus(&async));
                result = async.Get(&status);
            }

            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(nn::ec::ResultInvalidDeviceAccountToken)
                {
                    APPLOG_INFO("Device token is expired\n");
                    APPLOG_INFO("Request synchronize registration...\n");
                    {
                        nn::ec::system::AsyncResult async;
                        NN_RESULT_DO(nn::ec::system::RequestSyncRegistration(&async));
                        NN_RESULT_DO(async.Get());
                    }

                    APPLOG_INFO("Check device account status again...\n");
                    {
                        nn::ec::system::AsyncDeviceAccountStatus async;
                        NN_RESULT_DO(nn::ec::system::RequestDeviceAccountStatus(&async));
                        NN_RESULT_DO(async.Get(&status));
                    }
                }
            NN_RESULT_END_TRY
        }

        APPLOG_INFO("   DeviceAccountStatus : %s\n", status.data);
        APPLOG_INFO("   NeedsTicketSync : %s\n", status.needsTicketSync ? "true" : "false");

        if (status.needsTicketSync)
        {
            APPLOG_INFO("This device account needs to sync ticket\n");
            APPLOG_INFO("Request sync ticket...\n");
            nn::ec::system::AsyncProgressResult async;
            NN_RESULT_DO(nn::ec::system::RequestSyncTicket(&async));
            NN_RESULT_DO(async.Get());
        }
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
} // NOLINT (readability/fn_size)
