﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <stdint.h>

#define ERR_MESS_ILLEGALPARAM NN_TEXT("モード指示子が不正です。")
#define ERR_MESS_VERSIONINFO NN_TEXT("バージョンが不正です。")
#define ERR_MESS_TOOSMALL \
    NN_TEXT("指定されたバージョンではデータを格納できません。")
#define ERR_MESS_DATAFORMAT    \
    NN_TEXT("指定されたモード指示子でエンコードできない文字が含まれています。")
#define ERR_MESS_SYSTEM            NN_TEXT("システムエラー。")
#define ERR_MESS_AUTO_DIV_VER \
    NN_TEXT("指定されたバージョンではデータを格納できません。")
#define ERR_MESS_AUTO_DIV_SIZE \
    NN_TEXT("コンテンツサイズが大きすぎます。分割数が16個を超えてしまいます。")
#define ERR_MESS_ARGUMENT_ILLEGAL NN_TEXT("引数が不正です")
#define ERR_MESS_IMAGE2BIG NN_TEXT("イメージサイズが大きすぎます。")

/** 分割情報 */
typedef struct {
    CHAININFO chain;
    VERSIONINFO version[2]; ///< バージョン
    USERDATA userdata;
    int32_t cellpitch;      ///< セル幅
    int32_t qzone;          ///< クワイエットゾーン幅
    FORMATINFO format[2];   ///< 形式情報
    bool fakemodeon;
    MODE fakechainmode;
    MODE fakemode;
    MODE mode;
    uint8_t rightparity;
} DIVDATA;

/** 分割状態 */
typedef enum {
    DIV_NONE = 0,
    DIV_EQUAL = 1,
    DIV_FREE = 2
} HOWTODIVIDE;

/** QRコード情報 */
typedef struct {
    int32_t divcount;       ///< 分割数
    DIVDATA *divdata;
    USERDATA userdata_all;
    HOWTODIVIDE howtodiv;   ///< 分割状態
} QRCPARAM;

/// QRコード生成クラス
class CQREncode {
    mw::qre::ImageInfo m_imageInfo[16];
    char m_ErrorMessage[256];           ///< エラーメッセージ
    int32_t m_divCount;                 ///< 分割数
    int32_t m_version;                  ///< フォーマットバージョン
    int32_t m_size;                     ///< イメージの一辺のサイズ
    bool create;

public:

    /**
     * コンストラクタ
     */
    CQREncode();

    /**
     * デストラクタ
     */
    ~CQREncode();

    /**
     * QRコード生成
     *
     * @param   qrInfo(in)  QRコード情報構造体
     * @return  生成結果（trueは正常：falseは失敗）
     */
    bool QRE_EncodeData(mw::qre::EncodeData *qrInfo);

    bool QRE_EncodeWithImg(mw::qre::EncodeData *qrInfo, const mw::qre::ImageInfo *imgInfo);

    /**
     * BMPファイルサイズ取得
     *
     * @param   index(in)    分割番号 (0～15)
     * @return  BMPファイルサイズ
     */
    int32_t QRE_GetBMPSize(int32_t index, bool is_bmp);

    /**
     * BMPデータ取得
     *
     * @param   pData(out)  画像データ
     * @param   size(in)    画像サイズ
     * @param   mode(in)    取得モード (mode 0: 保存 1: 表示)
     * @param   index(in)   分割番号 (0～15)
     * @return  取得結果（trueは正常：falseは失敗）
     */
    bool QRE_GetBMPData(uint8_t *pData, int32_t size, int32_t index, mw::qre::Rect *pos);
    bool QRE_GetBMPData(mw::qre::ImageInfo *, int32_t);

    /**
     * 分割数取得
     *
     * @return  分割数
     */
    int32_t QRE_GetDivCount();

    /**
     * フォーマットバージョン取得
     *
     * @return  フォーマットバージョン
     */
    int32_t QRE_GetVersion();

    /**
     * エラー情報取得
     *
     * @param   pMess(out) エラー情報文字列格納用
     * @return  取得結果（trueは正常：falseは失敗）
     */
    bool QRE_GetErrorMessage(char *pMess, int32_t size);

private:

    /**
     * データ作成
     *
     * @param   qrcparam(out)   QRコード情報
     * @param   qrinfo(in)      QRコード情報
     * @return  成功/失敗
     */
    bool LoadData(QRCPARAM &qrcparam, const mw::qre::EncodeData *qrinfo, MODE mode);

    /**
     * 開放
     *
     * @param   qrcparam(in)    QRコード情報
     * @return  成功/失敗
     */
    bool FreeData(QRCPARAM &qrcparam);

    /**
     * 分割情報設定(分割なし)
     *
     * @param   qrcparam(out)       QRコード情報
     * @param   userdata_all(out)   データ列
     * @return  成功/失敗
     */
    bool DivideNone(QRCPARAM &qrcparam, USERDATA &userdata_all);

    /**
     * 分割情報設定(均等分割)
     *
     * @param   div(in)             分割数
     * @param   qrcparam(out)       QRコード情報
     * @param   userdata_all(out)   データ列
     * @return  成功/失敗
     */
    bool DivideEqual(int32_t div, QRCPARAM &qrcparam, USERDATA &userdata_all);

    /**
     * QRシンボル生成
     *
     * @param   divdata(in)         分割情報
     * @param   divID(in)           分割番号
     * @return  成功/失敗
     */
    bool MakeOneSymbol(DIVDATA &divdata, int32_t divID);

    /**
     * マトリクス開放
     *
     * @param   matrix(out)         マトリクス
     * @param   cellsize(in)        セルサイズ
     * @return  成功/失敗
     */
    void FreeMatrix(uint8_t **matrix, int32_t cellsize);

    bool CreateRawData(int32_t size, int32_t pitch, int32_t qzone, uint8_t **matrix,
                       int32_t divID);

    bool CreateBMPFile(uint8_t *, int32_t);

    /**
     * 自動分割チェック
     *
     * @param   qrinfo(in)      QRコード情報
     * @param   div(out)        分割数
     * @param   ver(out)        バージョン
     * @return  成功/失敗
     */
    bool CheckAutoDiv(const mw::qre::EncodeData *qrinfo, int32_t *div, int32_t *ver, MODE mode);

    bool CompositeImage(const mw::qre::ImageInfo *img, mw::qre::Rect *pos, int32_t divID);

    bool WhiteOutImage(mw::qre::Rect *pos, int32_t divID);


    /**
     * エラーメッセージ設定
     * @param   mess(in)        追加メッセージ
     */
    void SetErrorMessage(const char *mess);
public:
    void ReleaseMemory();
};
