﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/repair.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/manu/manu_Api.h>
#include <nn/repair/repair_Authentication.h>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_Result.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_CommandLineOption.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_ProductModel.h>
#include <nn/util/util_ScopeExit.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/nn_SdkLog.h>

#define RETURN_IF_FAILURE(result, ...) \
    do { \
    if (result.IsFailure()){SendString(__VA_ARGS__); return result;} \
    } while(NN_STATIC_CONDITION(false))

namespace {

static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
const size_t messageBufferSize = 1024;
char messageBuffer[messageBufferSize] = {};

static nn::os::MutexType s_Mutex;

static const int TOOL_MAJOR_VERSION = 8;
static const int TOOL_MINOR_VERSION = 0;
static const int RESTORE_THREAD_STACK_SIZE = 0x10000;

static glv::GLV           s_GlvRootView;
static nn::repair::StreamTextView*    s_pTextView;
static nn::os::ThreadType s_SdWaiterThreadType;
static nn::os::ThreadType s_GameCardWaiterThreadType;
static nn::os::ThreadType s_UnlockWaiterThreadType;
static nn::os::ThreadType s_RestoreWaiterThreadType;
static nn::os::ThreadType s_ProcessingThreadType;
static nn::os::EventType  s_SdInsertionEvent;
static nn::os::EventType  s_GameCardInsertionEvent;
static nn::os::EventType  s_UnlockPushedEvent;
static nn::os::EventType  s_RestoreStartEvent;

NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_SdThreadStack[0x1000];
NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_GameCardThreadStack[0x1000];
NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_UnlockWaiterThreadStack[0x1000];
NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_RestoreWaiterThreadStack[0x1000];

static nn::repair::LabelButton* s_UnlockButton;
static nn::repair::LabelButton* s_StartButton;
static nn::repair::LabelButton* s_ExitButton;

static const char* const s_AutoExitOption = "--autoexit";
static const char* const s_AutoStartOption = "--autostart";

static void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

static void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

static void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

static const std::string MountName = "Target";
static const std::string DirectoryPath = MountName + ":/" + nn::repair::ArchiveDirectoryPath;

void SendString(const char* pFormat, ...) NN_NOEXCEPT
{
    nn::os::LockMutex(&s_Mutex);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlockMutex(&s_Mutex);
    };

    va_list args;
    va_start(args, pFormat);
    auto len = vsnprintf(messageBuffer, messageBufferSize, pFormat, args);
    va_end(args);

    #if !defined(NN_REPAIR_DEVELOP_MODE)
        size_t writeSize;
        auto result = nn::manu::UsbWrite(&writeSize, messageBuffer, messageBufferSize, len);
        if (result.IsFailure())
        {
            NN_SDK_LOG("%s:%d %016x\n", __FILE__, __LINE__, result.GetInnerValueForDebug());
        }
    #else
        NN_UNUSED(len);
    #endif

    s_pTextView->AppendValue(messageBuffer);
}

void SendStringToScreen(const char* pFormat, ...) NN_NOEXCEPT
{
    nn::os::LockMutex(&s_Mutex);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlockMutex(&s_Mutex);
    };

    va_list args;
    va_start(args, pFormat);
    auto len = vsnprintf(messageBuffer, messageBufferSize, pFormat, args);
    va_end(args);

    NN_UNUSED(len);

    s_pTextView->AppendValue(messageBuffer);
}

void ShowSerialNumber() NN_NOEXCEPT
{
    char number[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(number);
    SendString("Serial Number : %s\n", number);
}

void ShowInternalInformation() NN_NOEXCEPT
{
    // シリアルナンバー表示
    ShowSerialNumber();
}

nn::Result ReadSourceSerialNumber(char* pOutNumber) NN_NOEXCEPT
{
    // シリアルナンバー保存先のファイルパス
    std::string path = DirectoryPath + std::string(nn::repair::SerialNumberFileName);

    NN_RESULT_DO(nn::fs::MountSdCard(MountName.c_str()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    nn::fs::FileHandle handle;
    NN_RESULT_DO(
            nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t fileSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    auto buffer = new char[fileSize + 1];
    NN_UTIL_SCOPE_EXIT
    {
        delete [] buffer;
    };

    memset(buffer, '\0', fileSize + 1);
    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, buffer, fileSize));

    std::strncpy(pOutNumber, buffer, nn::repair::SerialNumberLength);
    pOutNumber[nn::repair::SerialNumberLength - 1] = '\0';  // ファイルより読み出した文字列長がシリアルナンバー文字列長より大きかった場合の対策用

    NN_RESULT_SUCCESS;
}

nn::Result ShowRestoreDetails() NN_NOEXCEPT
{
    char srcNumber[nn::repair::SerialNumberLength];
    std::memset(srcNumber, '-', nn::repair::SerialNumberLength);
    srcNumber[nn::repair::SerialNumberLength - 1] = '\0';   // 終端をNULL文字埋め
    ReadSourceSerialNumber(srcNumber);

    char dstNumber[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(dstNumber);

    SendString("(src) %s ==> (dst) %s\n", srcNumber, dstNumber);

    NN_RESULT_SUCCESS;
}

nn::Result RecordTargetInformation(uint32_t rebootlessNupVersion) NN_NOEXCEPT
{
    // 保存先のファイルパス
    const std::string path = DirectoryPath + std::string(nn::repair::HistoryFileName);

    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsFailure())
    {
        NN_RESULT_DO(nn::fs::CreateFile(path.c_str(), 0));
    }

    nn::fs::FileHandle handle;
    NN_RESULT_DO(
            nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t offset;
    NN_RESULT_DO(nn::fs::GetFileSize(&offset, handle));

    // デバイス ID の取得
    char id[nn::repair::DeviceIdLength];
    nn::repair::GetDeviceId(id);

    // シリアルナンバーの取得
    char number[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(number);

    // ProductModel の取得
    auto model = nn::settings::system::GetProductModel();

    std::ostringstream history;
    history << "[restore] " << "device_id=" << id << ", " << "serial_number=" << number << ", ";
    history << "version=" << TOOL_MAJOR_VERSION << "." << TOOL_MINOR_VERSION << ", ";
    history << "rebootless-NUP=" << rebootlessNupVersion << ", ";
    history << "model=" << model << "\n";
    NN_RESULT_DO(nn::fs::WriteFile(handle, offset, history.str().c_str(), history.str().size(), nn::fs::WriteOption()));
    NN_RESULT_DO(nn::fs::FlushFile(handle));

    NN_RESULT_SUCCESS;
}

nn::Result CanBeTransferredSaveData(uint32_t mySystemUpdateVersion) NN_NOEXCEPT
{
    const std::string path = DirectoryPath + std::string(nn::repair::HistoryFileName);
    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsFailure())
    {
        // 履歴ファイルがないのは v2.X でバックアップした場合
        // よってすべてのリストアツールでリストア可能
        NN_RESULT_SUCCESS;
    }

    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));

    std::unique_ptr<char []> buffer(new char[static_cast<int>(size + 1)]);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, buffer.get(), static_cast<size_t>(size)));
    *(buffer.get() + size) = '\0';

    std::stringstream stream(buffer.get());
    std::string line;

    bool isToolVersionCheck = true;
    bool isSystemUpdateVersionCheck = true;

    while (std::getline(stream, line, '\n'))
    {
        if (line.find("[backup]") != std::string::npos)
        {
            std::stringstream ss(line);
            std::string field;

            while (!ss.eof() && std::getline(ss, field, ','))
            {
                if (field.find("version") != std::string::npos)
                {
                    auto offset = field.find("=") + 1;
                    std::istringstream issSaveData(field.substr(offset, field.size() - offset));
                    float saveDataVersion;
                    issSaveData >> saveDataVersion;

                    std::ostringstream oss;
                    oss << TOOL_MAJOR_VERSION << "." << TOOL_MINOR_VERSION;
                    std::istringstream issTool(oss.str());
                    float toolVersion;
                    issTool >> toolVersion;

                    SendString("tool version(origin) = %g\n", saveDataVersion);
                    SendString("tool version(me)     = %g\n", toolVersion);

                    // セーブデータのバージョンが同じ、もしくは古い場合のみリストア可能
                    // チェックするのはメジャーバージョンのみ
                    isToolVersionCheck = (static_cast<int>(saveDataVersion) <= static_cast<int>(toolVersion)) ? true : false;
                }
                else if (field.find("rebootless-NUP") != std::string::npos)
                {
                    auto offset = field.find("=") + 1;
                    std::istringstream issSaveData(field.substr(offset, field.size() - offset));
                    uint32_t yourSystemUpdateVersion;
                    issSaveData >> yourSystemUpdateVersion;

                    SendString("SystemUpdateVersion(origin) = %d\n", yourSystemUpdateVersion);
                    SendString("SystemUpdateVersion(me)     = %d\n", mySystemUpdateVersion);

                    // SystemUpdateVersion = 再起動不要NUPバージョン
                    // 巻き戻しは不可
                    isSystemUpdateVersionCheck = (yourSystemUpdateVersion <= mySystemUpdateVersion) ? true : false;
                }
                else if (field.find("model") != std::string::npos)
                {
                    auto offset = field.find("=") + 1;
                    std::istringstream issSaveData(field.substr(offset, field.size() - offset));
                    uint32_t yourModel;
                    issSaveData >> yourModel;
                    auto myModel = static_cast<uint32_t>(nn::settings::system::GetProductModel());

                    SendString("ProductModel(origin) = %d\n", yourModel);
                    SendString("ProductModel(me)     = %d\n", myModel);

                    NN_RESULT_THROW_UNLESS(myModel == yourModel,
                            nn::repair::ResultInvalidModelTransferring());
                }
            }
        }
    }

    if( !isToolVersionCheck )
    {
        SendString("ToolVersion Check Fail \n");
    }
    if( !isSystemUpdateVersionCheck )
    {
        SendString("SystemUpdateVersion Check Fail \n");
    }

    // 履歴ファイルがあるが中にバックアップのログがない
    // v2.X でバックアップして、複数回リストアしたケースなのでリストア可能
    NN_RESULT_THROW_UNLESS(isToolVersionCheck && isSystemUpdateVersionCheck,
            nn::repair::ResultVersionDownTransferring());

    NN_RESULT_SUCCESS;
}

nn::Result CheckSaveData(uint32_t myVersion) NN_NOEXCEPT
{
    auto result = CanBeTransferredSaveData(myVersion);

    if (nn::repair::ResultVersionDownTransferring::Includes(result))
    {
        SendString("ERROR : This data is too new to restore this target.\n");
    }
    else if (nn::repair::ResultInvalidModelTransferring::Includes(result))
    {
        SendString("ERROR : This data is from an invalid product model.\n");
    }

    return result;
}

void ShowStorageSppedMode() NN_NOEXCEPT
{
    SendString("Storage speed mode : ");

    const char* eMmcSpeedMode = nullptr;
    nn::fs::MmcSpeedMode eMmcMode;
    auto result = nn::fs::GetMmcSpeedMode(&eMmcMode);
    if (result.IsFailure())
    {
        eMmcSpeedMode = "Failed to get the speed mode";
    }
    else
    {
        switch (eMmcMode)
        {
        case nn::fs::MmcSpeedMode_Identification:
            eMmcSpeedMode = "Identification";
            break;
        case nn::fs::MmcSpeedMode_LegacySpeed:
            eMmcSpeedMode = "Legacy";
            break;
        case nn::fs::MmcSpeedMode_HighSpeed:
            eMmcSpeedMode = "HighSpeed";
            break;
        case nn::fs::MmcSpeedMode_Hs200:
            eMmcSpeedMode = "HS200";
            break;
        case nn::fs::MmcSpeedMode_Hs400:
            eMmcSpeedMode = "HS400";
            break;
        case nn::fs::MmcSpeedMode_Unknown:
            eMmcSpeedMode = "Unknown";
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* sdCardSpeedMode = nullptr;
    nn::fs::SdCardSpeedMode sdcardMode;
    result = nn::fs::GetSdCardSpeedMode(&sdcardMode);

    if (result.IsFailure())
    {
        sdCardSpeedMode = "Failed to get the speed mode";
    }
    else
    {
        switch (sdcardMode)
        {
        case nn::fs::SdCardSpeedMode_Identification:
            sdCardSpeedMode = "Identification";
            break;
        case nn::fs::SdCardSpeedMode_DefaultSpeed:
            sdCardSpeedMode = "Default";
            break;
        case nn::fs::SdCardSpeedMode_HighSpeed:
            sdCardSpeedMode = "HighSpeed";
            break;
        case nn::fs::SdCardSpeedMode_Sdr12:
            sdCardSpeedMode = "SDR12";
            break;
        case nn::fs::SdCardSpeedMode_Sdr25:
            sdCardSpeedMode = "SDR25";
            break;
        case nn::fs::SdCardSpeedMode_Sdr50:
            sdCardSpeedMode = "SDR50";
            break;
        case nn::fs::SdCardSpeedMode_Sdr104:
            sdCardSpeedMode = "SDR104";
            break;
        case nn::fs::SdCardSpeedMode_Ddr50:
            sdCardSpeedMode = "DDR50";
            break;
        case nn::fs::SdCardSpeedMode_Unknown:
            sdCardSpeedMode = "Unknown";
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    SendString("eMMC = %s, SdCard = %s", eMmcSpeedMode, sdCardSpeedMode);
}

nn::Result Restore() NN_NOEXCEPT
{
    auto result = nn::fs::MountSdCard(MountName.c_str());
    RETURN_IF_FAILURE(result, "Failed to mount device\n");
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    nn::repair::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::repair::Finalize();
    };

    SendString("- Get SystemUpdateVersion\n");
    uint32_t rebootlessNupVersion;
    result = nn::repair::GetSystemUpdateVersion(&rebootlessNupVersion);
    RETURN_IF_FAILURE(result, "Failed to GetSystemVersion\n");

    SendString("- Check save data\n");
    result = CheckSaveData(rebootlessNupVersion);
    RETURN_IF_FAILURE(result, "Failed to check save data\n");
    SendString("- Check done\n");

    SendString("- Restore save data\n");
    nn::repair::ReportCallbackStruct callback = { SendString, nullptr, SendStringToScreen, nullptr };

    #if !defined(NN_REPAIR_DEVELOP_MODE)
        result = nn::repair::ImportSaveDataSecurely(
                DirectoryPath.c_str(), &callback);
    #else
        result = nn::repair::ImportSaveData(
                DirectoryPath.c_str(), &callback);
    #endif
    RETURN_IF_FAILURE(result, "Failed to import save data\n");
    {
        uint64_t size;
        int64_t time;
        int64_t count;
        nn::repair::GetTransportRateInfo(&size, &time, &count);
        if( time > 0 )
        {
            SendString("- Total %lld KiB %lld file %lld sec (%d KiB/sec) \n", size / 1024, count, time / 1000, (int)(size * 1000 / time) / 1024 );
        }
    }
    SendString("- Restore done\n");

    // リストア先の本体情報を書き出す
    SendString("- Record information\n");
    result = RecordTargetInformation(rebootlessNupVersion);
    RETURN_IF_FAILURE(result, "Failed to record target information\n");
    SendString("- Record done\n");

    NN_RESULT_SUCCESS;
}

void ApplicationExit()
{
    // SD 挿入待ちイベントの終了
    nn::os::FinalizeEvent(&s_SdInsertionEvent);

    // ゲームカード挿入待ちイベントの終了
    nn::os::FinalizeEvent(&s_GameCardInsertionEvent);

    nn::os::FinalizeEvent(&s_RestoreStartEvent);

#if !defined(NN_REPAIR_DEVELOP_MODE)
    // 中断時は Repair サービスへの入力待ち状態になっているので適当な値を送って終了させる
    auto message = nn::repair::RestoreRequestMessage::MakeZero();
    nn::manu::WriteToHost(&message, sizeof(message), nn::repair::GetOptionArgument(nn::repair::RequestOption), 0, sizeof(message));
    nn::manu::FinalizeUfio();
#endif

    nn::os::FinalizeMutex(&s_Mutex);

    std::quick_exit(0);
}

void ProcessingThread(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    SendString("=== Restore Start ===\n");

    if (nn::repair::HasOption(s_AutoExitOption))
    {
        SendString("=== Auto Exit mode ===\n");
    }

    auto result = Restore();

    SendString("M: %d, D: %d, I: %d\n",
               result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());

    if (result.IsFailure())
    {
        SendString("[[ FAILED ]]\n");
    }
    else
    {
        SendString("[[ SUCCESS ]]\n");
    }

    SendStringToScreen("\n\n\n\n\n --- \n");

    // eMMC 及び microSD の速度表示
    ShowStorageSppedMode();

    // お尻が表示されない問題への暫定対応
    // （glv::TextView にバッファされた未表示の残りカスを明示的に flush させる手段が不明)
    SendStringToScreen("\n\n\n\n\n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");

    if (nn::repair::HasOption(s_AutoExitOption))
    {
        ApplicationExit();
    }
}

void RestoreMouseDownHandler() NN_NOEXCEPT
{
    // リストア開始イベントのチェック
    if (nn::os::TryWaitEvent(&s_RestoreStartEvent))
    {
        nn::os::StartThread(&s_ProcessingThreadType);
    }
}

void RestoreWaiter(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    while (!nn::os::TryWaitEvent(&s_SdInsertionEvent)
            || !nn::os::TryWaitEvent(&s_GameCardInsertionEvent)
            || !nn::os::TryWaitEvent(&s_UnlockPushedEvent))
    {
        nn::os::YieldThread();
    }

    s_StartButton->enable(glv::Property::Visible);
    nn::os::SignalEvent(&s_RestoreStartEvent);

    SendString("Ready to restore !\n");
    SendString("Push START button\n");

    if (nn::repair::HasOption(s_AutoStartOption))
    {
        RestoreMouseDownHandler();
    }
}

void SdcardWaiter(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    SendString("Please insert your micro sd card\n");

    std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
    auto result = sdcard->WaitAttach();
    NN_ASSERT(result.IsSuccess(), "Failed to wait for sd attach\n");

    // 移行元シリアルと移行先シリアルの表示
    result = ShowRestoreDetails();
    if (result.IsFailure())
    {
        SendString("Failed to display the detailes\n");
    }

    nn::os::SignalEvent(&s_SdInsertionEvent);
}

void GameCardWaiter(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    SendString("Please insert RepairTimeReviser Game Card\n");

    for (;;)
    {
        if (nn::fs::IsGameCardInserted())
        {
            SendString("RepairTimeReviser Game Card is inserted\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        nn::os::YieldThread();
    }

    nn::os::SignalEvent(&s_GameCardInsertionEvent);
}

bool IsAlarmingRestore() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSdCard(MountName.c_str()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    const std::string path = DirectoryPath + std::string(nn::repair::HistoryFileName);
    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsFailure())
    {
        // v3.0 より古いバージョンでバックアップ/リストアした場合ファイルが存在しない
        // 過去の履歴を確認できないので警告は出さない
        return false;
    }

    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));

    std::unique_ptr<char []> buffer(new char[static_cast<int>(size + 1)]);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, buffer.get(), static_cast<size_t>(size)));
    *(buffer.get() + size) = '\0';

    std::stringstream stream(buffer.get());
    std::string line;

    char id[nn::repair::DeviceIdLength];
    nn::repair::GetDeviceId(id);

    bool isDuplicating = false;
    while (std::getline(stream, line, '\n'))
    {
        // 異なるデバイス ID のリストアログがある場合は警告を出す
        if ((line.find("[restore]") != std::string::npos) && (line.find(id) == std::string::npos))
        {
            if (!isDuplicating)
            {
                isDuplicating = true;
                SendString("\n");
                SendString("WARNING : This data has been restored to another target.\n");
                SendString("If this is intended please push UNLOCK button.\n");
            }

            std::stringstream ss(line);
            std::string field;

            while (!ss.eof())
            {
                std::getline(ss, field, ',');

                if (field.find("device_id") != std::string::npos)
                {
                    auto offset = field.find("=") + 1;
                    const std::string deviceId = field.substr(offset, field.size() - offset);
                    SendString("[history] ");
                    SendString("Device Id : %s, ", deviceId.c_str());
                }
                else if (field.find("serial_number") != std::string::npos)
                {
                    auto offset = field.find("=") + 1;
                    const std::string serialNumber = field.substr(offset, field.size() - offset);
                    SendString("Serial Number : %s\n", serialNumber.c_str());
                }
            }
        }
    }

    if (isDuplicating)
    {
        // 異なるデバイス ID のログがあった
        return true;
    }

    // 異なるデバイス ID のリストアログがない場合は警告は出さない
    return false;
}

void UnlockWaiter(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // SD カードとゲームカードの挿入待ち
    while (!nn::os::TryWaitEvent(&s_SdInsertionEvent) || !nn::os::TryWaitEvent(&s_GameCardInsertionEvent))
    {
        nn::os::YieldThread();
    }

    // 過去に異なる本体にリストアしている場合は警告する
    if (IsAlarmingRestore())
    {
        s_UnlockButton->enable(glv::Property::Visible);
    }
    else
    {
        nn::os::SignalEvent(&s_UnlockPushedEvent);
    }
}

void RestoreWindowCreateHandler() NN_NOEXCEPT
{
    // 本体情報の表示
    ShowInternalInformation();

    // SD カードの挿入待ち
    nn::os::InitializeEvent(&s_SdInsertionEvent, false, nn::os::EventClearMode_ManualClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &s_SdWaiterThreadType, SdcardWaiter, nullptr, s_SdThreadStack,
        sizeof(s_SdThreadStack), nn::os::DefaultThreadPriority));

    nn::os::StartThread(&s_SdWaiterThreadType);

    // ゲームカードの挿入待ち
    nn::os::InitializeEvent(&s_GameCardInsertionEvent, false, nn::os::EventClearMode_ManualClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &s_GameCardWaiterThreadType, GameCardWaiter, nullptr, s_GameCardThreadStack,
        sizeof(s_GameCardThreadStack), nn::os::DefaultThreadPriority));

    nn::os::StartThread(&s_GameCardWaiterThreadType);

    // ロック解除待ち
    nn::os::InitializeEvent(&s_UnlockPushedEvent, false, nn::os::EventClearMode_ManualClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                &s_UnlockWaiterThreadType, UnlockWaiter, nullptr, s_UnlockWaiterThreadStack,
                sizeof(s_UnlockWaiterThreadStack), nn::os::DefaultThreadPriority));

    nn::os::StartThread(&s_UnlockWaiterThreadType);

    // リストア開始イベント待ち
    nn::os::InitializeEvent(&s_RestoreStartEvent, false, nn::os::EventClearMode_ManualClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &s_RestoreWaiterThreadType, RestoreWaiter, nullptr, s_RestoreWaiterThreadStack,
        sizeof(s_RestoreWaiterThreadStack), nn::os::DefaultThreadPriority));

    nn::os::StartThread(&s_RestoreWaiterThreadType);

    if (nn::repair::HasOption(s_AutoStartOption))
    {
        // ゲームカードの挿入のみスキップする
        nn::os::SignalEvent(&s_GameCardInsertionEvent);
    }
}

void RestoreWindowDestroyHandler() NN_NOEXCEPT
{
    nn::os::DestroyThread(&s_SdWaiterThreadType);
    nn::os::DestroyThread(&s_GameCardWaiterThreadType);
    nn::os::DestroyThread(&s_RestoreWaiterThreadType);
}

void UnlockMouseDownHandler() NN_NOEXCEPT
{
    nn::os::SignalEvent(&s_UnlockPushedEvent);
}

void ExitMouseDownHandler() NN_NOEXCEPT
{
    // 処理スレッドの破棄
    nn::os::DestroyThread(&s_ProcessingThreadType);

    ApplicationExit();
}

void Main( glv::Window& window ) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 512), 24.f);
    s_pTextView->pos(120, 100);
    *pGlvRootView << s_pTextView;

    // ツール情報ラベル表示
    auto titleLabel = new glv::Label("Repair Restore Tool", false);
    nn::repair::GetToolInformationLabel(
            titleLabel, "Repair Restore Tool", TOOL_MAJOR_VERSION, TOOL_MINOR_VERSION);
    *pGlvRootView << titleLabel;

    // 開始ボタン表示
    s_StartButton = new nn::repair::LabelButton(
            "START",
            [&]{RestoreMouseDownHandler();},
            [&]{RestoreWindowCreateHandler();},
            [&]{RestoreWindowDestroyHandler();});
    s_StartButton->pos(120, 640);
    s_StartButton->disable(glv::Property::Visible);
    *pGlvRootView << s_StartButton;

    // 解除ボタン表示
    s_UnlockButton = new nn::repair::LabelButton(
            "UNLOCK",
            [&]{UnlockMouseDownHandler();});
    s_UnlockButton->pos(540, 640);
    s_UnlockButton->disable(glv::Property::Visible);
    *pGlvRootView << s_UnlockButton;

    // 終了ボタン表示
    s_ExitButton = new nn::repair::LabelButton(
            "EXIT",
            [&]{ExitMouseDownHandler();});
    s_ExitButton->pos(1030, 640);
    *pGlvRootView << s_ExitButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

#if !defined(NN_REPAIR_DEVELOP_MODE)
    extern "C" void nninitInitializeSdkModule()
    {
        nn::fs::InitializeWithMultiSessionForTargetTool();
    }

    extern "C" void nninitFinalizeSdkModule()
    {
    }

    extern "C" void nndiagStartup()
    {
    }
#endif

extern "C" void nnMain()
{
    SetupPeripherals();

    #if !defined(NN_REPAIR_DEVELOP_MODE)
        nn::manu::InitializeUfio();
    #endif

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    nn::os::InitializeMutex(&s_Mutex, false, 0);

    // リストア処理スレッド
    NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ProcessingThreadStack[RESTORE_THREAD_STACK_SIZE];
    auto result = nn::os::CreateThread(
            &s_ProcessingThreadType, ProcessingThread, nullptr, s_ProcessingThreadStack,
            sizeof(s_ProcessingThreadStack), nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}

