﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <glv.h>
#include <nn/nifm.h>
#include <nn/socket.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TFormatString.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_NetworkSettingsParser.h>
#include <sstream>

#define RETURN_IF_FAILURE(result, ...) \
    do { \
    if (result.IsFailure()){s_pTextView->AppendValue(__VA_ARGS__); return result;} \
    } while(NN_STATIC_CONDITION(false))

namespace {

const char* ToolName = "NX AutoNUPTool";
const int MajorVersion = 1;
const int MinorVersion = 8;

nn::nifm::NetworkProfileData g_NetworkSettings;
bool g_AirplaneModeEnabled = false;

/*
テスト用
static const char* s_json = R"(
{
    "network_settings": {
        "name": "default-ethernet",
        "id": "8cd3766b-89f9-40c4-aedc-451f18089206",
        "auto_connection": true,
        "large_capacity": true,
        "nic_type": "Ethernet",
        "ssid_config": {
            "name": "aaaa",
            "non_broadcast": false
        },
        "security": {
            "authentication": "Open",
            "encryption": "None",
            "key_material": ""
        },
        "ip_settings": {
            "auto_ip": true,
            "address": "1.0.0.0",
            "subnet_mask": "0.2.0.0",
            "default_gateway": "0.0.4.0",
            "auto_dns": true,
            "preferred_dns": "0.0.0.6",
            "alternate_dns": "0.0.0.1"
        },
        "proxy": {
            "enables": false,
            "hostname": "hoge",
            "port": 80,
            "authenticates": false,
            "username": "1234",
            "password": "hoge"
        },
        "mtu": 1400
    }
}
)";
*/

static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;

static glv::GLV           s_GlvRootView;
static nn::repair::StreamTextView*    s_pTextView;
static nn::os::ThreadType s_ThreadType;
static nn::os::ThreadType s_NupThead;
static nn::os::EventType  s_SdInsertionEvent;

static void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

static void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

static void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

void EnableWifiConnection() NN_NOEXCEPT
{
    g_AirplaneModeEnabled = !nn::nifm::IsWirelessCommunicationEnabled();
    nn::nifm::SetWirelessCommunicationEnabled(true);
}

void RestoreAirplaneMode() NN_NOEXCEPT
{
    if (g_AirplaneModeEnabled) {
        nn::nifm::SetWirelessCommunicationEnabled(false);
    }
}

void WaitNetworkConnectionAvailable(nn::nifm::TemporaryNetworkProfile* temporaryNetworkProfile,
                                    nn::nifm::NetworkConnection* networkConnection) NN_NOEXCEPT
{
    nn::nifm::RequestHandle requestHandle = networkConnection->GetRequestHandle();
    nn::util::Uuid profileId = temporaryNetworkProfile->GetId();
    nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);

    while (NN_STATIC_CONDITION(1))
    {
        s_pTextView->AppendValue("Connecting...\n");
        networkConnection->SubmitRequestAndWait();

        if (networkConnection->IsAvailable())
        {
            s_pTextView->AppendValue("Network is available.\n");
            break;
        }
        else
        {
            s_pTextView->AppendValue("Network is not available. retry.\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
        }
    }
}

nn::Result CheckAlreadyDownloaded(bool *pOutValue) NN_NOEXCEPT
{
    nn::ns::SystemUpdateControl control;
    NN_RESULT_DO(control.Occupy());

    if (control.HasDownloaded())
    {
        *pOutValue = true;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ApplyDownloadedSystem(bool *pOutValue) NN_NOEXCEPT
{
    nn::ns::SystemUpdateControl control;
    NN_RESULT_DO(control.Occupy());

    control.ApplyDownloadedUpdate();

    *pOutValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result DownloadLatest(bool *pOutValue) NN_NOEXCEPT
{
    nn::ns::SystemUpdateControl control;
    NN_RESULT_DO(control.Occupy());

    nn::ns::AsyncResult asyncResult;
    NN_RESULT_DO(control.RequestDownloadLatestUpdate(&asyncResult));

    while (!asyncResult.TryWait())
    {
        char buffer[1024];
        auto progress = control.GetDownloadProgress();
        nn::util::SNPrintf(buffer, 1024, "%12lld / %lld\n", progress.loaded, progress.total);
        s_pTextView->AppendValue(buffer);
        NN_LOG("%12lld / %lld\n", progress.loaded, progress.total);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_RESULT_DO(asyncResult.Get());
    char buffer[1024];
    auto progress = control.GetDownloadProgress();
    nn::util::SNPrintf(buffer, 1024, "%12lld / %lld\n", progress.loaded, progress.total);
    s_pTextView->AppendValue(buffer);

    *pOutValue = true;

    NN_RESULT_SUCCESS;
}

nn::Result CheckNeedsSystemUpdate(bool *pOutValue) NN_NOEXCEPT
{
    nn::ns::SystemUpdateControl control;
    NN_RESULT_DO(control.Occupy());

    nn::ns::LatestSystemUpdate latest;
    nn::ns::AsyncLatestSystemUpdate asyncLatest;
    NN_RESULT_DO(control.RequestCheckLatestUpdate(&asyncLatest));
    NN_RESULT_DO(asyncLatest.Get(&latest));

    if (latest == nn::ns::LatestSystemUpdate::UpToDate)
    {
        *pOutValue = false;
    }
    else
    {
        *pOutValue = true;
    }

    NN_RESULT_SUCCESS;
}

static const std::string MountName = "Target";
static const std::string DirectoryPath = MountName + ":/" + "RepairSaveData/";

void DisplayFirmwareVersion() NN_NOEXCEPT
{
    glv::Label firmwareVesionLabel;
    nn::repair::GetFirmwareVersionLabel(&firmwareVesionLabel);

    s_pTextView->AppendValue(firmwareVesionLabel.getValue() + "\n");
}

nn::Result ImportNetworkSettingsFromSdCard(bool* pOutValue) NN_NOEXCEPT
{
    *pOutValue = false;

    auto result = nn::fs::MountSdCardForDebug(MountName.c_str());
    RETURN_IF_FAILURE(result, "Failed to mount device\n");
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    nn::fs::FileHandle handle;
    std::string SerialFileName = (MountName + ":/settings.json");

    NN_RESULT_DO(nn::fs::OpenFile(&handle, SerialFileName.c_str(), nn::fs::OpenMode::OpenMode_Read));

    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    int totalSize = static_cast<int>(fileSize);
    std::unique_ptr<char[]> storage(new char[totalSize]);

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, storage.get(), totalSize));

    auto text = std::string(storage.get(), totalSize);

    if (!nn::repair::ImportNetworkSettings(&g_NetworkSettings, text.c_str()))
    {
        NN_LOG("Invaild NetworkSettings Format.\n");
        NN_RESULT_SUCCESS;
    }

    *pOutValue = true;

    NN_RESULT_SUCCESS;
}

nn::Result NetWorkUpdate() NN_NOEXCEPT
{
    //Wifi 設定を有効にする
    EnableWifiConnection();
    NN_UTIL_SCOPE_EXIT
    {
        //処理が終わったら機内モードを戻す
        RestoreAirplaneMode();
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(g_NetworkSettings);
    nn::nifm::NetworkConnection networkConnection;

    WaitNetworkConnectionAvailable(&temporaryNetworkProfile, &networkConnection);

    bool needsUpdate = false;
    NN_RESULT_DO(CheckNeedsSystemUpdate(&needsUpdate));
    s_pTextView->AppendValue(needsUpdate ? "Needs update\n" : "Already up to date\n");

#ifdef UPDATE_GENTLY
    if (needsUpdate == false)
    {
        NN_RESULT_SUCCESS;
    }
#endif

    bool downloaded = false;
    NN_RESULT_DO(CheckAlreadyDownloaded(&downloaded));
    if (!downloaded)
    {
        bool downloadResult = false;
        NN_RESULT_DO(DownloadLatest(&downloadResult));
    }

    bool applyResult = false;
    NN_RESULT_DO(ApplyDownloadedSystem(&applyResult));

    NN_RESULT_SUCCESS;
}

void NetworkUpdateThread(void *arg) NN_NOEXCEPT
{
    // SD が挿入されたら処理を始める
    nn::os::WaitEvent(&s_SdInsertionEvent);

    s_pTextView->AppendValue("=== Auto NUP Start ===\n");

    nn::Result result = NetWorkUpdate();

    std::ostringstream ErrorMessage;
    ErrorMessage << "M: " << result.GetModule() << ", ";
    ErrorMessage << "D: " << result.GetDescription() << ", ";
    ErrorMessage << "I: " << result.GetInnerValueForDebug() << "\n";
    s_pTextView->AppendValue(ErrorMessage.str().c_str());

    if( result.IsSuccess() )
    {
        s_pTextView->AppendValue("[[ SUCCESS ]]\n");
    }
    else if( nn::ns::ResultAlreadyUpToDate().Includes(result) )
    {
        s_pTextView->AppendValue("[[ SUCCESS (UP TO DATE) ]]\n");
    }
    else
    {
        s_pTextView->AppendValue("[[ FAILED ]]\n");
    }

    // 連続実行は認めない
    nn::os::ClearEvent(&s_SdInsertionEvent);
}

void AutoNupMouseDownHandler()
{
    static bool s_Start = false;

    if (!s_Start)
    {
        s_Start = true;
        NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ThreadStack[0x10000];
        auto result = nn::os::CreateThread(
            &s_NupThead, NetworkUpdateThread, nullptr, s_ThreadStack,
            sizeof(s_ThreadStack), nn::os::DefaultThreadPriority);
        NN_ASSERT(result.IsSuccess(), "Failed to create thread\n");
        nn::os::StartThread(&s_NupThead);
    }
}

void SdcardWaiter(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::os::InitializeEvent(&s_SdInsertionEvent, false, nn::os::EventClearMode_ManualClear);

    s_pTextView->AppendValue("Please insert your micro sd card\n");

    std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());

    for (;;)
    {
        auto result = sdcard->WaitAttach();
        NN_ASSERT(result.IsSuccess(), "Failed to wait for sd attach\n");

        bool isValid = false;
        result = ImportNetworkSettingsFromSdCard(&isValid);
        if (result.IsFailure())
        {
            s_pTextView->AppendValue("Read settings.json in SD card failed. Please reinsert.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(sdcard->WaitDetach());
            continue;
        }

        if (isValid)
        {
            s_pTextView->AppendValue("Ready to Auto NUP !\n");
            s_pTextView->AppendValue("Push START button\n");
            break;
        }
        else
        {
            s_pTextView->AppendValue("Invalid Network Settings Format\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(sdcard->WaitDetach());
        }
    }

    nn::os::SignalEvent(&s_SdInsertionEvent);
}

void AutoNupWindowCreateHandler() NN_NOEXCEPT
{
    NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ThreadStack[0x1000];
    auto result = nn::os::CreateThread(
            &s_ThreadType, SdcardWaiter, nullptr, s_ThreadStack,
            sizeof(s_ThreadStack), nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Failed to create thread\n");

    nn::os::StartThread(&s_ThreadType);
}

void AutoNupWindowDestroyHandler() NN_NOEXCEPT
{
    nn::os::DestroyThread(&s_ThreadType);
}

void Main( glv::Window& window ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    nn::ns::Initialize();

    glv::GLV* const pGlvRootView = &s_GlvRootView;

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 512), 32.f);
    s_pTextView->pos(120, 100);
    *pGlvRootView << s_pTextView;

    // タイトルラベル表示
    auto titleLabel = new glv::Label("", false);
    nn::repair::GetToolInformationLabel(titleLabel, ToolName, MajorVersion, MinorVersion);
    *pGlvRootView << titleLabel;

    //現在のファームウェアバージョンの表示
    DisplayFirmwareVersion();

    // 開始ボタン表示
    auto startButton = new nn::repair::LabelButton(
            "START",
            [&]{AutoNupMouseDownHandler();},
            [&]{AutoNupWindowCreateHandler();},
            [&]{AutoNupWindowDestroyHandler();});
    startButton->pos(540, 640);
    *pGlvRootView << startButton;

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *pGlvRootView << shutdownButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

extern "C" void nnMain()
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_ABORT_UNLESS_NOT_NULL(window);

    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}
